#include "mbed.h"
#include "MS5607SPI.h"
#include "arm_math.h"

#define N_SAMPLES 1024
#define PEAK_SEARCH_START 1 //first bin to include in peak search
#define PEAK_SEARCH_END 100 //last bin to include in peak search
#define SUM_AROUND_PEAK 2 //+/- this many bins

#define REGRESSION_SLOPE 53.77986f //from Excel fit to test data
#define REGRESSION_INTERCEPT -80.07f

Serial pc(USBTX, USBRX);

DigitalOut myled(LED1);
PwmOut drive(p26);
AnalogIn motorVolt(p19);
AnalogIn motorAmp(p20);
DigitalOut ps(p9);

DigitalOut trig(p21);

Timer sample_timer;
Timer iter_timer;

MS5607SPI pressure_sensor(p5, p6, p7, p8); // mosi, miso, sclk, cs

arm_rfft_fast_instance_f32 rfft;

float p_data[N_SAMPLES];
float p_data_raw[N_SAMPLES];
int p_data_rawi[N_SAMPLES];
int t_data[N_SAMPLES];
float f_data[N_SAMPLES];
float mag_data[N_SAMPLES / 2 + 1];

unsigned int tData[N_SAMPLES];

unsigned int coefficients[7];
char str[32];
float motorDuty;
float motorCurr;
float sample_rate;

float blockMag;

//run flow block functionality and print some of sample out to console for analysis
float flowBlock(float duty, int dataPrint) {
    if (dataPrint){
        pc.printf("STARTING FLOW BLOCK TEST- %3.0f%% Duty Cycle\n", duty*100);
    }
    
    //turn motor on hard and settle to required duty cycle (voltage)
    drive.write(1.0f);
    wait_ms(100);
    drive.write(duty);
    
    int iter_counter = 0; //Iteration counter
    iter_timer.reset();
    iter_timer.start();
    float temperature_raw = pressure_sensor.getRawTemperature();

    int a = 0;
    while (a<2000) {
        pressure_sensor.getRawPressure();
        a++;
    }
    
    sample_timer.reset();
    sample_timer.start();

    //Capture raw pressure samples
    for(int i = 0; i < N_SAMPLES; i++) {
        trig = 1;
        p_data_raw[i] = pressure_sensor.calculatePressure(pressure_sensor.getRawPressure(), temperature_raw);
        trig = 0;
        p_data[i] = p_data_raw[i];
    }
    
    sample_timer.stop();
    motorCurr = 0;
    for(int x=0; x<10; x++){
        motorCurr = motorCurr + motorAmp.read();
    }    
    //turn off motor        
    drive.write(0.0);
    
    
    
    //http://www.keil.com/pack/doc/CMSIS/DSP/html/group__RealFFT.html
    //Compute the RFFT
    //
    //p_data is trashed in the process
    //Result is packaged as [real_0, real_N/2, real_1, imag_1, real_2, imag_2 ... real_N/2-1, imag_N/2-1]
    arm_rfft_fast_init_f32(&rfft, N_SAMPLES);
    arm_rfft_fast_f32(&rfft, p_data, f_data, 0);

    //http://www.keil.com/pack/doc/CMSIS/DSP/html/group__cmplx__mag.html
    //Convert to magntiude, skip over the DC and fundamental terms

    arm_cmplx_mag_f32(&f_data[2], &mag_data[1], (N_SAMPLES / 2) - 1);
    //Fill in the first and last terms from the first entry in f_data
    mag_data[0] = f_data[0];
    mag_data[N_SAMPLES / 2] = f_data[1];


    //Find peak in spectrum
    float max_mag = -1;
    int max_mag_i = -1;
    if (dataPrint) {
        
        for (int x=0; x < N_SAMPLES; x++){
            pc.printf("%f, %f\n", p_data_raw[x], mag_data[x]);
            }
        
    }
    
    for(int i = PEAK_SEARCH_START + SUM_AROUND_PEAK; i < PEAK_SEARCH_END; i++) {
        if (dataPrint) {
            //pc.printf("%10f \n", mag_data[i]);
        }
        if(mag_data[i] > max_mag) {
            max_mag_i = i;
            max_mag = mag_data[i];
        }
    }
    //pc.printf("max mag=%f\n", max_mag);
    
    //Sum surrounding
    float sum = 0.f;
    for(int i = max_mag_i - SUM_AROUND_PEAK; i < (max_mag_i + SUM_AROUND_PEAK + 1); i++) {
        sum += mag_data[i];
        //printf("%10f ", sum);
    }

   

    //Scale by N_SAMPLES to recover pressure in Pa
    sum /= N_SAMPLES;

    //Apply inverse of logarithmic regression from test data
    float t90 = exp((sum + 80.07) / 53.77986);


    //Output iteration data
    iter_counter++;
    sample_rate = 1000000 / (sample_timer.read_us() / N_SAMPLES);
    if (dataPrint) {
        pc.printf("   titer    fsamp  avpress     fmax sum_peak         t90\n");
        pc.printf("%8.2f, %8.2f, %8.1f, %8.1f, %3.5e, %8.1f\n", iter_timer.read(), sample_rate, f_data[0] / N_SAMPLES, (max_mag_i * sample_rate / N_SAMPLES), sum, t90);
        //pc.printf("$SAM\n");
        //pc.printf("$%d\n", sample_timer.read_us());
    }
    
    return sum;
    
}
//run flow block functionality and print some of sample out to console for analysis
void flowBlockRaw(float duty) {
    
    pc.printf("START\n");

    //turn motor on hard and settle to required duty cycle (voltage)
    drive.write(1.0f);
    wait_ms(100);
    drive.write(duty);
    
    int iter_counter = 0; //Iteration counter
    iter_timer.reset();
    iter_timer.start();
    float temperature_raw = pressure_sensor.getRawTemperature();

    int a = 0;
    while (a<2000) {
        pressure_sensor.getRawPressure();
        a++;
    }
    
    sample_timer.reset();
    sample_timer.start();

    //Capture raw pressure samples
    for(int i = 0; i < N_SAMPLES; i++) {
        t_data[i] = sample_timer.read_us();
        p_data_rawi[i] = pressure_sensor.getRawPressure();  
    }
    
    sample_timer.stop();
    
    //print to console
    for(int i = 0; i < N_SAMPLES; i++) {
        pc.printf("%d, %u\n", t_data[i], p_data_rawi[i]);
        p_data[i] = (float)p_data_rawi[i];
    }
     
    pc.printf("STOP\n");      
    drive.write(0.0);
    
}

int main() {
    pc.baud(115200);
    pc.printf("MS5607 TEST\n");
    drive.period_ms(10);
    motorDuty = 1.0f;
    drive.write(motorDuty);
    ps = 0;
    
    pressure_sensor.reset();
    //pc.printf("CRC check %d\n", pressure_sensor.checkCRC());
    //pressure_sensor.printCoefficients(str);
    //pc.printf(str);
    
    
    motorDuty = 1.0f;
    
    for (int i = 0; i<10; i++){
         
        //pc.printf("%3.0f, %f \n", motorDuty*100, flowBlock(motorDuty, 1));
        //motorDuty = motorDuty + 0.01;
        flowBlock(motorDuty, 1);
        
        
            
        }
    pc.printf("END\n");
    
}
