#include "mbed.h"
#include "DevSPI.h"

#include "PowerStep01.h"

powerstep01_init_u_t init = {
    /* common parameters */
    .cm.cp.cmVmSelection = POWERSTEP01_CM_VM_CURRENT, // enum powerstep01_CmVm_t
    582, // Acceleration rate in step/s2, range 14.55 to 59590 steps/s^2
    582, // Deceleration rate in step/s2, range 14.55 to 59590 steps/s^2
    48, // Maximum speed in step/s, range 15.25 to 15610 steps/s
    0, // Minimum speed in step/s, range 0 to 976.3 steps/s
    POWERSTEP01_LSPD_OPT_OFF, // Low speed optimization bit, enum powerstep01_LspdOpt_t
    244.16, // Full step speed in step/s, range 7.63 to 15625 steps/s
    POWERSTEP01_BOOST_MODE_OFF, // Boost of the amplitude square wave, enum powerstep01_BoostMode_t
    281.25, // Overcurrent threshold settings via enum powerstep01_OcdTh_t
    STEP_MODE_1_16, // Step mode settings via enum motorStepMode_t
    POWERSTEP01_SYNC_SEL_DISABLED, // Synch. Mode settings via enum powerstep01_SyncSel_t
    (POWERSTEP01_ALARM_EN_OVERCURRENT|
    POWERSTEP01_ALARM_EN_THERMAL_SHUTDOWN|
    POWERSTEP01_ALARM_EN_THERMAL_WARNING|
    POWERSTEP01_ALARM_EN_UVLO|
    POWERSTEP01_ALARM_EN_STALL_DETECTION|
    POWERSTEP01_ALARM_EN_SW_TURN_ON|
    POWERSTEP01_ALARM_EN_WRONG_NPERF_CMD), // Alarm settings via bitmap enum powerstep01_AlarmEn_t
    POWERSTEP01_IGATE_64mA, // Gate sink/source current via enum powerstep01_Igate_t
    POWERSTEP01_TBOOST_0ns, // Duration of the overboost phase during gate turn-off via enum powerstep01_Tboost_t
    POWERSTEP01_TCC_500ns, // Controlled current time via enum powerstep01_Tcc_t
    POWERSTEP01_WD_EN_DISABLE, // External clock watchdog, enum powerstep01_WdEn_t
    POWERSTEP01_TBLANK_375ns, // Duration of the blanking time via enum powerstep01_TBlank_t
    POWERSTEP01_TDT_125ns, // Duration of the dead time via enum powerstep01_Tdt_t
    /* current mode parameters */
    328.12, // Hold torque in mV, range from 7.8mV to 1000 mV
    328.12, // Running torque in mV, range from 7.8mV to 1000 mV
    328.12, // Acceleration torque in mV, range from 7.8mV to 1000 mV
    328.12, // Deceleration torque in mV, range from 7.8mV to 1000 mV
    POWERSTEP01_TOFF_FAST_8us, //Maximum fast decay time , enum powerstep01_ToffFast_t
    POWERSTEP01_FAST_STEP_12us, //Maximum fall step time , enum powerstep01_FastStep_t
    3.0, // Minimum on-time in us, range 0.5us to 64us
    21.0, // Minimum off-time in us, range 0.5us to 64us
    POWERSTEP01_CONFIG_INT_16MHZ_OSCOUT_2MHZ, // Clock setting , enum powerstep01_ConfigOscMgmt_t
    POWERSTEP01_CONFIG_SW_HARD_STOP, // External switch hard stop interrupt mode, enum powerstep01_ConfigSwMode_t
    POWERSTEP01_CONFIG_TQ_REG_TVAL_USED, // External torque regulation enabling , enum powerstep01_ConfigEnTqReg_t
    POWERSTEP01_CONFIG_VS_COMP_DISABLE, // Motor Supply Voltage Compensation enabling , enum powerstep01_ConfigEnVscomp_t
    POWERSTEP01_CONFIG_OC_SD_DISABLE, // Over current shutwdown enabling, enum powerstep01_ConfigOcSd_t
    POWERSTEP01_CONFIG_UVLOVAL_LOW, // UVLO Threshold via powerstep01_ConfigUvLoVal_t
    POWERSTEP01_CONFIG_VCCVAL_15V, // VCC Val, enum powerstep01_ConfigVccVal_t
    POWERSTEP01_CONFIG_TSW_048us, // Switching period, enum powerstep01_ConfigTsw_t
    POWERSTEP01_CONFIG_PRED_DISABLE // Predictive current enabling , enum powerstep01_ConfigPredEn_t
};

DigitalIn button(USER_BUTTON);
PowerStep01 *motor;

void flag_irq_handler(void)
{
    /* Set ISR flag. */
    motor->isrFlag = TRUE;
    /* Get the value of the status register. */
    unsigned int statusRegister = motor->get_status();
    printf("    WARNING: \"FLAG\" interrupt triggered.\r\n");
    /* Check SW_F flag: if not set, the SW input is opened */
    if ((statusRegister & POWERSTEP01_STATUS_SW_F ) != 0) {
        printf("    SW closed (connected to ground).\r\n");
    }
    /* Check SW_EN bit */
    if ((statusRegister & POWERSTEP01_STATUS_SW_EVN) == POWERSTEP01_STATUS_SW_EVN) {
        printf("    SW turn_on event.\r\n");
    }
    /* Check Command Error flag: if set, the command received by SPI can't be */
    /* performed. This occurs for instance when a move command is sent to the */
    /* Powerstep01 while it is already running */
    if ((statusRegister & POWERSTEP01_STATUS_CMD_ERROR) == POWERSTEP01_STATUS_CMD_ERROR) {
        printf("    Non-performable command detected.\r\n");
    }
    /* Check UVLO flag: if not set, there is an undervoltage lock-out */
    if ((statusRegister & POWERSTEP01_STATUS_UVLO)==0) {
        printf("    undervoltage lock-out.\r\n");
    }
    /* Check thermal STATUS flags: if  set, the thermal status is not normal */
    if ((statusRegister & POWERSTEP01_STATUS_TH_STATUS)!=0) {
        //thermal status: 1: Warning, 2: Bridge shutdown, 3: Device shutdown
        printf("    Thermal status: %d.\r\n", (statusRegister & POWERSTEP01_STATUS_TH_STATUS)>>11);
    }
    /* Check OCD  flag: if not set, there is an overcurrent detection */
    if ((statusRegister & POWERSTEP01_STATUS_OCD)==0) {
        printf("    Overcurrent detection.\r\n");
    }
    /* Reset ISR flag. */
    motor->isrFlag = FALSE;
}

/**
 * @brief  This is an example of error handler.
 * @param[in] error Number of the error
 * @retval None
 * @note   If needed, implement it, and then attach it:
 *           + motor->attach_error_handler(&my_error_handler);
 */
void error_handler(uint16_t error)
{
    printf("Error %d detected\r\n\n", error);
    while (true);
}

/* Main ----------------------------------------------------------------------*/

int main()
{
    printf("STARTING MAIN PROGRAM\r\n");

    /* Initializing SPI bus. */
    DevSPI dev_spi(D11, D12, D13);

    /* Initializing Motor Control Component. */
    motor = new PowerStep01(D2, D4, D8, D9, D10, dev_spi);
    if (motor->init(&init) != COMPONENT_OK) {
        exit(EXIT_FAILURE);
    }

    /* Attaching and enabling an interrupt handler. */
    motor->attach_flag_irq(&flag_irq_handler);
    motor->enable_flag_irq();

    /* Attaching an error handler */
    motor->attach_error_handler(&error_handler);

    printf("Press button to start motor control.\r\n");
    while(button);
    
    printf("--> Going to start position.\r\n");
    motor->move(StepperMotor::FWD, 2000);
    motor->wait_while_active();

    while(button) {
        //----- Go to position 7680
        printf("--> Go to position -2000 steps.\r\n");
        /* Request device to go to position 7680 */
        motor->go_to(-2000);
        /* Wait for the motor ends moving */
        motor->wait_while_active();
        printf("--> Go to position 2000 steps.\r\n");
        motor->go_to(2000);
        /* Wait for the motor ends moving */
        motor->wait_while_active();
    }
    
    //----- Go Home
    /* Printing to the console. */
    printf("--> Go to home position.\r\n");

    /* Request device to go to Home */
    motor->go_home();
    motor->wait_while_active();
    /* Request a soft stop of device and keep the power bridges enabled */
    motor->soft_hiz();
    /* Wait for the motor of device ends moving */
    motor->wait_while_active();
}