/*------------------------------------------------------------------------------
Library header file for heater operations
Date: 16/07/2018


------------------------------------------------------------------------------*/

#ifndef Heater_H
#define Heater_H
#include "mbed.h"
#include "ADS8568_ADC.h"
#include "FastPWM.h"
#include "memspcr.pb.h"

#define R_SMOOTH_FACTOR 0.9  // resisstance smoothing factor (0 = no smoothing, 1 = infinite smoothing)

class Heater
{
    //This class provides the interface through which each heater can be controlled
public:
    /** Constructor
             * @param thermal passes in all the control information needed for the heater
             * @param i_port is the ADC port relating to the current read
             * @param v_port is the ADC port relating to the voltage read
             * @param * drive is a pointer to the main heater
             * @param * guard is a pointer to the guard heater
    **/
    Heater(const int i_port, const int v_port, float cal_a, float cal_b, FastPWM * drive, FastPWM * guard, ADS8568_ADC * adc, DigitalIn adc_busy, const memspcr_ThermalConfiguration & thermal = memspcr_ThermalConfiguration_init_zero);


    //Public member functions

    void read();        //Updates the resistance and error values for the heater
    void update();      //Holds R_ref for hold_time miliseconds
    void turn_on();     //Turns the heater on
    void turn_off();    //Turns the heater off

   //Getters and setters
    void Set_ref(float R);
    void Set_D(float D);
    float Get_D() const;
    int Get_i() const;
    int Get_v() const;
    float Get_R() const;
    float Get_R_avg();
    float Get_R_var();
    float Get_R_ref() const;
    float Get_error() const;
    float Get_error_integrated() const;

protected:

    const memspcr_ThermalConfiguration thermal;
    int curr;               //Latest current reading from ADC
    int v;                  //Latest voltage reading from ADC
    float R;                //Latest resistance calculated from ADC current and voltage
    float R_smooth;         //Smoothed resistance for more noise-tolerant heater control 
    float R_avg;            //Average resistance since last log event
    float R_var;            //Variance resistance since last log event
    float R_acc;            //Accumulated sum of resistance values since last log event
    float R2_acc;           //Accumulated sum of squares of resistance values since last log event 
    int n_acc;              //Number of resistance measurements since last log event
    float R_ref;            //Current referance for resistance
    float error;            //R_ref - R
    float error_integrated; //Integrated error
    ADS8568_ADC * adc;
    DigitalIn adc_busy;
    int i_port;             //ADC port which corresponds to current measurements
    int v_port;             //ADC port which corresponds to voltage measurements
    float cal_a;
    float cal_b;            //Resistance calibration ADC units to Ohms: R_Ohm = cal_a + cal_b * R_ADC
    FastPWM * drive;        //Pointer to the driver
    FastPWM * guard;        //Pointer to the guard

};

#endif