/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __BLE_DEVICE_INFORMATION_SERVICE_H__
#define __BLE_DEVICE_INFORMATION_SERVICE_H__

#include "ble/BLE.h"

/**
* @class DeviceInformationService
* @brief BLE Device Information Service
* Service: https://developer.bluetooth.org/gatt/services/Pages/ServiceViewer.aspx?u=org.bluetooth.service.device_information.xml
* Manufacturer Name String Char: https://developer.bluetooth.org/gatt/characteristics/Pages/CharacteristicViewer.aspx?u=org.bluetooth.characteristic.manufacturer_name_string.xml
*/
class DeviceInformationService {
public:
    /**
     * @brief Device Information Service Constructor: copies device-specific information 
     * into the BLE stack.
     *
     * @param[ref] _ble
     *                BLE object for the underlying controller.
     * @param[in] manufacturersName
     *                The name of the manufacturer of the device.
     * @param[in] modelNumber
     *                The model number that is assigned by the device vendor.
     * @param[in] serialNumber
     *                The serial number for a particular instance of the device. 
     * @param[in] hardwareRevision
     *                The hardware revision for the hardware within the device.
     * @param[in] firmwareRevision
     *                The device's firmware version. 
     * @param[in] softwareRevision
     *                The device's software version.
     */
    DeviceInformationService(BLE        &_ble,
                             const char *manufacturersName = "unknown",
                             const char *modelNumber       = "unknown",
                             const char *serialNumber      = "unknown",
                             const char *hardwareRevision  = "unknown",
                             const char *firmwareRevision  = "unknown",
                             const char *softwareRevision  = "unknown") :
        ble(_ble),
        manufacturersNameStringCharacteristic(/*GattCharacteristic::UUID_MANUFACTURER_NAME_STRING_CHAR*/0x3101,
                                              (uint8_t *)manufacturersName,
                                              strlen(manufacturersName), /* Min length */
                                              strlen(manufacturersName), /* Max length */
                                              GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ),
        modelNumberStringCharacteristic(/*GattCharacteristic::UUID_MODEL_NUMBER_STRING_CHAR*/0x3102,
                                        (uint8_t *)modelNumber,
                                        strlen(modelNumber), /* Min length */
                                        strlen(modelNumber), /* Max length */
                                        GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ),
        serialNumberStringCharacteristic(/*GattCharacteristic::UUID_SERIAL_NUMBER_STRING_CHAR*/0x3103,
                                         (uint8_t *)serialNumber,
                                         strlen(serialNumber), /* Min length */
                                         strlen(serialNumber), /* Max length */
                                         GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ),
        hardwareRevisionStringCharacteristic(/*GattCharacteristic::UUID_HARDWARE_REVISION_STRING_CHAR*/0x3104,
                                             (uint8_t *)hardwareRevision,
                                             strlen(hardwareRevision), /* Min length */
                                             strlen(hardwareRevision), /* Max length */
                                             GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ),
        firmwareRevisionStringCharacteristic(/*GattCharacteristic::UUID_FIRMWARE_REVISION_STRING_CHAR*/0x3105,
                                             (uint8_t *)firmwareRevision,
                                             strlen(firmwareRevision), /* Min length */
                                             strlen(firmwareRevision), /* Max length */
                                             GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ),
        softwareRevisionStringCharacteristic(/*GattCharacteristic::UUID_SOFTWARE_REVISION_STRING_CHAR*/0x3106,
                                             (uint8_t *)softwareRevision,
                                             strlen(softwareRevision), /* Min length */
                                             strlen(softwareRevision), /* Max length */
                                             GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ)
    {
        
        GattCharacteristic *charTable[] = {&manufacturersNameStringCharacteristic,
                                           &modelNumberStringCharacteristic,
                                           &serialNumberStringCharacteristic,
                                           &hardwareRevisionStringCharacteristic,
                                           &firmwareRevisionStringCharacteristic,
                                           &softwareRevisionStringCharacteristic};
        GattService         deviceInformationService(/*GattService::UUID_DEVICE_INFORMATION_SERVICE*/0x3100, charTable,
                                                     sizeof(charTable) / sizeof(GattCharacteristic *));

        ble.addService(deviceInformationService);
    }

protected:
    BLE                &ble;
    
    //ReadOnlyGattCharacteristic manufacturersNameStringCharacteristic;
    
    GattCharacteristic  manufacturersNameStringCharacteristic;
    GattCharacteristic  modelNumberStringCharacteristic;
    GattCharacteristic  serialNumberStringCharacteristic;
    GattCharacteristic  hardwareRevisionStringCharacteristic;
    GattCharacteristic  firmwareRevisionStringCharacteristic;
    GattCharacteristic  softwareRevisionStringCharacteristic;
};

#endif /* #ifndef __BLE_DEVICE_INFORMATION_SERVICE_H__*/