/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __BLE_BATTERY_SERVICE_H__
#define __BLE_BATTERY_SERVICE_H__

#include "ble/BLE.h"

/**
* @class BatteryService
* @brief BLE Battery Service. This service displays the battery level from 0% to 100%, represented as an 8bit number.
* Service:  https://developer.bluetooth.org/gatt/services/Pages/ServiceViewer.aspx?u=org.bluetooth.service.battery_service.xml
* Battery Level Char:  https://developer.bluetooth.org/gatt/characteristics/Pages/CharacteristicViewer.aspx?u=org.bluetooth.characteristic.battery_level.xml
*/
class BatteryService {
public:

    const static uint16_t BATTERY_SERVICE_UUID                = 0x3000;
    
    const static uint16_t BATTERY_LEVEL_1_CHARACTERISTIC_UUID = 0x3001;
    const static uint16_t BATTERY_LEVEL_2_CHARACTERISTIC_UUID = 0x3002;
    const static uint16_t BATTERY_LEVEL_3_CHARACTERISTIC_UUID = 0x3003;

    /**
     * @param[ref] _ble
     *               BLE object for the underlying controller.
     * @param[in] level
     *               8bit batterly level. Usually used to represent percentage of batterly charge remaining.
     */
    BatteryService(BLE &_ble, uint8_t level1 = 100, uint8_t level2 = 100, uint8_t level3 = 100) :
        ble(_ble),
        batteryLevel1(level1),
        batteryLevel2(level2),
        batteryLevel3(level3),
        batteryLevelCharacteristic1(BATTERY_LEVEL_1_CHARACTERISTIC_UUID, &batteryLevel1, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        batteryLevelCharacteristic2(BATTERY_LEVEL_2_CHARACTERISTIC_UUID, &batteryLevel2, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        batteryLevelCharacteristic3(BATTERY_LEVEL_3_CHARACTERISTIC_UUID, &batteryLevel3, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY)
    {

        GattCharacteristic *charTable[] = {&batteryLevelCharacteristic1, &batteryLevelCharacteristic2, &batteryLevelCharacteristic3};
        GattService         batteryService(BATTERY_SERVICE_UUID, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));

        ble.gattServer().addService(batteryService);
        //printf("adress of _ble: %#x\n", &_ble);
        //printf("adress of ble: %#x\n", &ble);
    }

    /**
     * @brief Update the battery level with a new value. [Valid values lie between 0 and 100];
     * anything outside this range will be ignored.
     *
     * @param newLevel
     *              Update to battery level.
     */
    void updateBatteryLevel1(uint8_t newLevel) {
        //printf("try to update batlevel 1\n");
        if (newLevel != batteryLevel1)
        {
            batteryLevel1 = newLevel;
            ble.gattServer().write(batteryLevelCharacteristic1.getValueHandle(), &batteryLevel1, 1);
            //printf("BLE battery Level 1 update : %i\n", batteryLevel1);
            //printf("adress of ble: %#x\n", &ble);
        }
        
    }
    void updateBatteryLevel2(uint8_t newLevel) {
        if (newLevel != batteryLevel2)
        {
            batteryLevel2 = newLevel;
            ble.gattServer().write(batteryLevelCharacteristic2.getValueHandle(), &batteryLevel2, 1);
        }
    }
    void updateBatteryLevel3(uint8_t newLevel) {
        if (newLevel != batteryLevel3)
        {
            batteryLevel3 = newLevel;
            ble.gattServer().write(batteryLevelCharacteristic3.getValueHandle(), &batteryLevel3, 1);
            //printf("battery level updated\n");
        }
    }

protected:
    BLE &ble;
    uint8_t    batteryLevel1;
    uint8_t    batteryLevel2;
    uint8_t    batteryLevel3;
    ReadOnlyGattCharacteristic<uint8_t> batteryLevelCharacteristic1;
    ReadOnlyGattCharacteristic<uint8_t> batteryLevelCharacteristic2;
    ReadOnlyGattCharacteristic<uint8_t> batteryLevelCharacteristic3;
};

#endif /* #ifndef __BLE_BATTERY_SERVICE_H__*/