#ifndef NANOBIT_DISPLAY_H
#define NANOBIT_DISPLAY_H

//  
#define MICROBIT_ID_DISPLAY             6


/**
  * Core Configuration settings.
  */

#define MICROBIT_DISPLAY_ROW_COUNT          3
#define MICROBIT_DISPLAY_COLUMN_COUNT       9


#define MICROBIT_DISPLAY_REFRESH_PERIOD     0.006


/**
  * Default parameters.
  */
#define MICROBIT_DISPLAY_MAX_BRIGHTNESS     255
#define MICROBIT_DISPLAY_MIN_BRIGHTNESS     2
#define MICROBIT_DEFAULT_BRIGHTNESS         (MICROBIT_DISPLAY_MAX_BRIGHTNESS/2)


#define MICROBIT_DISPLAY_WIDTH              5
#define MICROBIT_DISPLAY_HEIGHT             5


#define NO_CONN 0

#include "mbed.h"

enum DisplayMode {
    DISPLAY_MODE_BLACK_AND_WHITE
};

struct MatrixPoint {
    uint8_t x;
    uint8_t y;

    MatrixPoint(uint8_t _x, uint8_t _y) : x(_x), y(_y) {};
};

/**
  * Class definition for a MicroBitDisplay.
  *
  * A MicroBitDisplay represents the LED matrix array on the MicroBit device.
  */
class MicroBitDisplay 
{
    uint8_t width;
    uint8_t height;
    uint8_t brightness;
    uint8_t strobeRow;
    uint8_t strobeBitMsk;
    
    Ticker                  systemTicker;

    Timeout renderTimer;

    uint8_t bitmap[MICROBIT_DISPLAY_WIDTH * MICROBIT_DISPLAY_HEIGHT];
    
    static const MatrixPoint matrixMap[MICROBIT_DISPLAY_COLUMN_COUNT][MICROBIT_DISPLAY_ROW_COUNT];

    
    /**
      *  Called by the display in an interval determined by the brightness of the display, to give an impression
      *  of brightness.
      */
    void renderFinish();
    
    /**
      * Translates a bit mask to a bit mask suitable for the nrf PORT0 and PORT1.
      * Brightness has two levels on, or off.
      */
    void render();
    
    

public:

    /**
      * Constructor.
      * Create a representation of a display of a given size.
      * The display is initially blank.
      *
      * @param x the width of the display in pixels.
      * @param y the height of the display in pixels.
      *
      * Example:
      * @code
      * MicroBitDisplay display(MICROBIT_ID_DISPLAY, 5, 5),
      * @endcode
      */
    MicroBitDisplay(uint16_t id=MICROBIT_ID_DISPLAY, uint8_t x=MICROBIT_DISPLAY_WIDTH, uint8_t y=MICROBIT_DISPLAY_HEIGHT);

    /**
      * Frame update method, invoked periodically to strobe the display.
      */
    virtual void systemTick();
    
    /**
      * Sets the display brightness to the specified level.
      * @param b The brightness to set the brightness to, in the range 0..255.
      *
      * Example:
      * @code
      * uBit.display.setBrightness(255); //max brightness
      * @endcode
      */
    void setBrightness(int b);


    
    /**
      * Fetches the current brightness of this display.
      * @return the brightness of this display, in the range 0..255.
      *
      * Example:
      * @code
      * uBit.display.getBrightness(); //the current brightness
      * @endcode
      */
    int getBrightness();

    /**
      * Enables the display, should only be called if the display is disabled.
      *
      * Example:
      * @code
      * uBit.display.enable(); //reenables the display mechanics
      * @endcode
      */
    void enable();

    /**
      * Disables the display, should only be called if the display is enabled.
      * Display must be disabled to avoid MUXing of edge connector pins.
      *
      * Example:
      * @code
      * uBit.display.disable(); //disables the display
      * @endcode
      */
    void disable();

    /**
      * Clears the current image on the display.
      * Simplifies the process, you can also use uBit.display.image.clear
      *
      * Example:
      * @code
      * uBit.display.clear(); //clears the display
      * @endcode
      */
    void clear();



    void setPixel(int x, int y, bool status=true) {
        bitmap[y*width+x] = status;    
    }
};



#endif
