/*
Title Block
**  Project     : datalogger_ft810_bitmap_practice
**  Engineers   : Curtis Mattull
**  Processor   : MK20
**  Version     : 1.0
**  Compiler    : mbed
**  Date        : 7/26/2016
**  Abstract    : This program serves as a learning aid to reading/writing to sd card.
    Note        : Thare are 2 locations to comment for changing lcd screens:
                    -line __ within main(),       variable:   display_type
                    -line 12 of FT_LCD_TYPE.h,  definition: beyondtek
*/

#include "mbed.h"
#include "FT_Platform.h"
#include "FT_color.h"
#include "stdio.h"
#include "string.h"
#include "float.h"
#include "SDFileSystem.h"
#include "FATFileSystem.h"

/************************
    function prototypes
*************************/

//for display type
void rotate_newhaven(void);
void rotate_beyondtek(void);
//for gui states
void    gui_manager(void);
void    main_menu(uint32_t tracker, uint8_t tag);       //  main menu            0
void    mode_a(uint32_t tracker, uint8_t tag);          //  mode A               1
void    mode_b(uint32_t tracker, uint8_t tag);          //  mode B               2
void    mode_c(uint32_t tracker, uint8_t tag);          //  mode C               3
//for plot states
void    plot_nothing(void);
void    plot_square(void);
void    plot_triangle(void);
void    plot_sine(void);
void    collect_data(void);
//for sd card
uint8_t read_calibration(void);
uint8_t sd_test(void);
uint8_t bitmap_test(uint16_t x,uint16_t y);
int8_t  Load_jpg(char* filename, ft_int16_t* x_size, ft_int16_t* y_size, ft_uint32_t address);
//general use
void    display_message(uint8_t error,  const ft_char8_t *pass, const ft_char8_t *fail);
void    start_screen(ft_char8_t *str);
void    error_screen(ft_char8_t *str1, ft_char8_t *str2);

/************************
    function pointers
*************************/

void (*draw_screen[4])(uint32_t tracker, uint8_t tag) = {   main_menu,      //  0
        mode_a,         //  1
        mode_b,         //  2
        mode_c
                                                        };       //  3

void (*plot[4])(void) = {   plot_nothing,      //  0
                            plot_square,         //  1
                            plot_triangle,         //  2
//                            plot_sine
                            collect_data
                        };       //  3
                    


void (*display_rotate[2])() = { rotate_newhaven,    //  0
                                rotate_beyondtek
                              }; //  1

/************************
    global objects and variables
*************************/

// create an LCD object
FT800 TFT   (   D11,    D12,    D13,    D9,     D8,     D14 );
//          (   mosi,   miso,   sclk,   ss,     intr,   pd );
//Create an SDFileSystem object
SDFileSystem sd (   D11,    D12,    D13,    D10,    "sd");
//SDFileSystem  (   mosi,   miso,   sclk,   ss,     name);

DigitalIn   sdcard_present(D7);   // SD card detect input pin.      card_present() is the name of a function within the SDFileSystem Class!!!
AnalogIn    analog_in9(A9);     // analog 9 pin connected to the PSE input   4
AnalogIn    analog_in8(A8);     // analog 9 pin connected to the PFW input   5
AnalogIn    analog_in3(A3);     // analog 9 pin connected to the ITV1 input 10
AnalogIn    analog_in2(A2);     // analog 9 pin connected to the ITV2 input 11 from usb, starting at 1

const uint8_t   font_button =   28,
                font_title =    31,
                font_author =   21;

const ft_char8_t    text_menu[32]   = {"Data Logger\0"},
                    text_a[32]      = {"Plot adc\0"},
                    text_b[32]      = {"Settings\0"},
                    text_c[32]      = {"ISE40A\0"},
                    text_back[32]   = {"Back\0"},
                    text_author[32] = {"by Curtis Mattull\0"};

int16_t i = 0x00;
static const uint32_t   mask_reg_tracker_tag        = 0x0000FFFF;
static const uint32_t   mask_reg_tracker_tracker    = 0xFFFF0000;
static const uint8_t    mask_tag                    = 0xFF;

char buffer[50];                                                                                                    // temporary buffer for writing characters to the LCD

ft_int16_t x_size, y_size;                                                                                          // stores size data for loaded jpegs, not currently used

/************************
    function:   main()
    description:
        -calibrate touch screen
        -display welcome screen
        -manages touch screen input, system states, and calls draw_screen functions
*************************/
int main()
{
//    static const uint8_t display_type = 0x00;   //not beyondtek display
    static const uint8_t display_type = 0x01;
    uint8_t error[4]    = { 0x00,   // sd_test error
                            0x00,   // load_bitmap error
                            0x00,   // bitmap write error
                            0x00
                          };  // gui_manager error

    (*display_rotate[display_type])();  // for proper orientation
    error[3] = read_calibration();
    display_message( error[3], "touch calibration load success!\0", "touch calibration save success!\0");
    TFT.Sleep(100);                     // Wait ___ to show

    //load jpg into graphics ram
    error[1] = Load_jpg("/sd/Logo.jpg", &x_size, &y_size, RAM_G);
    display_message( error[1], "load jpg success!\0", "load jpg failed!\0");
    TFT.Sleep(100);                     // Wait ___ to show

    start_screen("gui by\0");           //unneccessary..
    TFT.Sleep(100);                     // Wait ___ to show

//    //sd card test
//    error[0] = sd_test();
//    display_message( error[0], "sd test passed!\0", "sd test failed!\0");
//    TFT.Sleep(250);                     // Wait ___ to show

//    //display bitmap
//    error[2] = draw_bitmap(25,25);
//    TFT.Sleep(250);                     // Wait ___ to show
//    display_message( error[2], "draw_bitmap success!\0", "draw_bitmap failed!\0");
//    TFT.Sleep(250);                     // Wait ___ to show

    //draw initial screen
    main_menu(0x0,0x0);
    //program should not return from gui_manager...
    gui_manager();

    // Wait ___ to show
    display_message( 0x1, "\0", "gui_manager failed!\0");
    while(1) {
        TFT.Sleep(1000);
    }

    return 0;
}




/************************
    function: rotate_newhaven
    description:  rotates
*************************/
void rotate_newhaven(void)
{
    TFT.MemWrite(REG_ROTATE, 1);    // rotate screen
    TFT.Rotate(1);
    TFT.Flush_Co_Buffer();          // Download the command list into fifo
    TFT.WaitCmdfifo_empty();        // Wait till coprocessor completes the operation
}



/************************
    function: rotate_beyondtek
    description:  rotates
*************************/
void rotate_beyondtek(void)
{
    //dont rotate
}



/************************
    function: gui_manager()
    description:  tracks input data and updates screen accordingly
*************************/
void gui_manager(void)
{
    static const uint8_t state_lookup[32] = {0x0, 0x1, 0x2, 0x3, 0x0, 0x0, 0x0, 0x0,    0x1, 0x1, 0x1, 0x1, 0x0, 0x1, 0x1, 0x1,
                                            0x2, 0x2, 0x2, 0x2, 0x0, 0x2, 0x2, 0x2,    0x3, 0x3, 0x3, 0x3, 0x0, 0x3, 0x3, 0x3,
                                            };
    /*              array index values:     {  0    1    2    3    4    5    6    7       8    9   10   11   12   13   14   15
                                              16   17   18   19   20   21   22   23      24   25   26   27   28   29   30   31};
    */

    static uint8_t  present_state   = 0x00,
                    tag             = 0x00,
                    combo_input     = 0x00;
                    
    static uint16_t tracker         = 0x00;

    static uint32_t reg32           = 0x00;

    //start of gui_manager()
    while(1) {
        tag = 0x0;
        //while(0x0 == tag) 
//        {
            reg32 = TFT.Rd32(REG_TRACKER);
            tag = reg32 & mask_reg_tracker_tag;   //read tag register, update tag variable
            tracker = ( reg32 & mask_reg_tracker_tracker)>> 16;   //read tracker
//        }
        combo_input = ( present_state << 3) + tag;      // concatenate present_state with tag to form combo input
        present_state = state_lookup[combo_input];      // based on combo input, update present state
        (*draw_screen[present_state])(tracker,tag);                // use present_state as operand to draw screen function
    }
}



/************************
    function: main menu
    description:  draw main menu screen
*************************/
void main_menu(uint32_t tracker, uint8_t tag)
{
//    TFT.Wr16(REG_PWM_DUTY, 100);

    //start new display list
    TFT.DLstart();                                                      // start a new display command list
    TFT.DL(CLEAR_COLOR_RGB(255,255,255));                               // set the clear color to white
    TFT.DL(CLEAR(1,1,1));                                               // clear buffers -> color buffer,stencil buffer, tag buffer

    //bitmaps
    TFT.DL(TAG(0));     // assign TAG value 0 to bitmap
    TFT.DL(BITMAP_SOURCE(RAM_G));
    TFT.DL(BITMAP_LAYOUT(RGB565,x_size*2,y_size));
    TFT.DL(BITMAP_SIZE(NEAREST,BORDER,BORDER,x_size,y_size));

    TFT.DL(BEGIN(BITMAPS));     //start drawing bitmaps
    TFT.DL(VERTEX2II(10,10,0,0));                                      // draw logo image with bit handle 0
    TFT.DL(END());

    //buttons
    TFT.DL(COLOR_RGB(0xFF,  0xFF,   0xFF)); //this sets the button text color
    TFT.FgColor(0x007CC4);  // this sets the button color
    
    //button A
    TFT.Track( 50,  165, 200, 150, 1);  // track aree of button A
    TFT.DL(TAG(1));     // assign TAG value 1 to "mode A" button
    TFT.Button(50, 165, 200, 150, font_title, 0, text_a);

    //button B
    TFT.Track( 300, 165, 200, 150, 2);  // track aree of button B
    TFT.DL(TAG(2));                         // assign TAG value 2 to "mode B" button
    TFT.Button(300,  165, 200, 150, font_title, 0, text_b);

    //button C
    TFT.Track( 550, 165, 200, 150, 3);  // track aree of button C
    TFT.DL(TAG(3));                         // assign TAG value 3 to "mode C" button
    TFT.Button(550, 165, 200, 150, font_title, 0, text_c);

    TFT.DL(TAG(0));                         // assign TAG value 0 to everything else

    //title text
    TFT.DL(COLOR_RGB(0x00, 0x7C, 0xC4));
    TFT.Text((TFT.DispWidth/2), TFT.DispHeight/8, font_title, OPT_CENTERX, text_menu);        // draw Text with font 31

    //author text
    TFT.DL(COLOR_RGB(0x00,  0x00,   0x00));
    TFT.Text((TFT.DispWidth*7/8)+9, (TFT.DispHeight*15/16), font_button, OPT_CENTERX, text_author);            // draw Text with font 31

    //display the screen
    TFT.DL(DISPLAY());                     // display the image
    TFT.Swap();                            // swap the current display list
    TFT.Flush_Co_Buffer();                 // download the command list into fifo
    TFT.WaitCmdfifo_empty();               // wait till coprocessor completes the operation

}



/************************
    function: mode_a
    description: draws mode a screen
*************************/
void mode_a(uint32_t tracker, uint8_t tag)
{
    static const uint8_t plot_lookup[32] =  {0x0, 0x0, 0x2, 0x3, 0x0, 0x0, 0x1, 0x0,    0x1, 0x1, 0x2, 0x3, 0x1, 0x0, 0x1, 0x1,
                                            0x2, 0x2, 0x2, 0x3, 0x2, 0x0, 0x1, 0x2,    0x3, 0x3, 0x2, 0x3, 0x3, 0x0, 0x1, 0x3 };
    static const uint8_t    col_div   = 10,
                            row_div   = 8;
    static const uint8_t    col_step  = (TFT.DispWidth/col_div),
                            row_step  = (TFT.DispHeight/row_div);
    int8_t  row = 0,
            col = 0;

    //              array index values:     {  0    1    2    3    4    5    6    7       8    9   10   11   12   13   14   15
    //                                        16   17   18   19   20   21   22   23      24   25   26   27   28   29   30   31};
    static uint8_t  plot_state = 0x00,
                    plot_combo = 0x00;

    plot_combo = ( plot_state << 3) + tag;      // concatenate present_state with tag to form combo input
    plot_state = plot_lookup[plot_combo];      // based on present state and touch input, update present state

    //start new display list
    TFT.DLstart();                                                      // start a new display command list
    TFT.DL(CLEAR_COLOR_RGB(0,0,0));                               // set the clear color to dark grey
    TFT.DL(CLEAR(1,1,1));                                               // clear buffers -> color buffer,stencil buffer, tag buffer

    TFT.DL(TAG(0));                                                         // assign TAG value 0 to everything else
    //plot grid
    TFT.DL(COLOR_RGB(75,75,75));
    TFT.DL(BEGIN(LINES));
    TFT.DL(LINE_WIDTH(1 * 16));
    
    //draw plot grid
    //draw row lines
    for(row = 0;row < row_div-1; row++)
    {
        TFT.DL(VERTEX2F(0,row_step*16 + row*row_step*16));
        TFT.DL(VERTEX2F(800*16,row_step*16 + row*row_step*16));
    }
    
    //draw col lines
    for(col = 0;col < col_div-1; col++)
    {
        TFT.DL(VERTEX2F(col_step*16 + col*col_step*16, 0));
        TFT.DL(VERTEX2F(col_step*16 + col*col_step*16, 480*16));
    }

//    //bitmaps
//    TFT.DL(TAG(0));     // assign TAG value 0 to bitmap
//    TFT.DL(BITMAP_SOURCE(RAM_G));
//    TFT.DL(BITMAP_LAYOUT(RGB565,x_size*2,y_size));
//    TFT.DL(BITMAP_SIZE(NEAREST,BORDER,BORDER,x_size,y_size));
//
//    TFT.DL(BEGIN(BITMAPS));     //start drawing bitmaps
//    TFT.DL(VERTEX2II(10,10,0,0));                                      // draw logo image with bit handle 0
//    TFT.DL(END());

    //buttons
    TFT.DL(COLOR_RGB(0xFF,  0xFF,   0xFF)); //this sets the button text color
    TFT.FgColor(0x007CC4);  // this sets the button color

    //back button
    TFT.Track( 50, 355, 100, 75, 4);    // track aree of button back
    TFT.DL(TAG(4));
    TFT.Button(50, 355, 100, 75, font_button, 4, text_back);                // assign TAG value 4 to text_back button

    ////plot square button
//    TFT.Track( 200, 355, 100, 75, 6);    // track aree of button square
//    TFT.DL(TAG(6));
//    TFT.Button(200, 355, 100, 75, font_button, 6, "square");                // assign TAG value 1 to text_back button//back button
//
//    //plot triangle button
//    TFT.Track( 350, 355, 100, 75, 2);    // track aree of button triangle
//    TFT.DL(TAG(2));
//    TFT.Button(350, 355, 100, 75, font_button, 2, "triangle");              // assign TAG value 2 to text_back button//back button

    //plot sine button
    TFT.Track( 500, 355, 100, 75, 3);    // track aree of button sine
    TFT.DL(TAG(3));
    TFT.Button(500, 355, 100, 75, font_button, 3, "adc in");                  // assign TAG value 3 to text_back button//back button

    //plot nothing button
    TFT.Track( 650, 355, 100, 75, 5);    // track aree of button square
    TFT.DL(TAG(5));
    TFT.Button(650, 355, 100, 75, font_button, 5, "clear");                 // assign TAG value 1 to text_back button//back button

    
    TFT.DL(TAG(0));                                                         // assign TAG value 0 to everything else
    //author text
    TFT.DL(COLOR_RGB(0xFF,  0xFF,   0xFF));
    TFT.Text((TFT.DispWidth*7/8)+9, (TFT.DispHeight*15/16), font_button, OPT_CENTERX, text_author);            // draw Text with font 31
    //plot algorithm
    //    TFT.DL( BEGIN(LINE_STRIP) );    //works okay..
    TFT.DL( BEGIN(LINE_STRIP) );

    TFT.DL(LINE_WIDTH(3 * 16));
    (*plot[plot_state])();                // use present_state as operand to draw screen function

    //display the screen
    TFT.DL(DISPLAY());                     // display the image
    TFT.Swap();                            // swap the current display list
    TFT.Flush_Co_Buffer();                 // download the command list into fifo
    TFT.WaitCmdfifo_empty();               // wait till coprocessor completes the operation
}



/************************
    function: mode b
    description: draws mode b screen.
*************************/
void mode_b(uint32_t tracker, uint8_t tag)
{
    static uint8_t val = 50;
    
    //start new display list
    TFT.DLstart();                                                      // start a new display command list
    TFT.DL(CLEAR_COLOR_RGB(255,255,255));                               // set the clear color to white
    TFT.DL(CLEAR(1,1,1));                                               // clear buffers -> color buffer,stencil buffer, tag buffer


    //bitmaps
    TFT.DL(TAG(0));     // assign TAG value 0 to bitmap
    TFT.DL(BITMAP_SOURCE(RAM_G));
    TFT.DL(BITMAP_LAYOUT(RGB565,x_size*2,y_size));
    TFT.DL(BITMAP_SIZE(NEAREST,BORDER,BORDER,x_size,y_size));

    TFT.DL(BEGIN(BITMAPS));     //start drawing bitmaps
    TFT.DL(VERTEX2II(10,10,0,0));                                      // draw logo image with bit handle 0
    TFT.DL(END());

    //buttons
    TFT.DL(COLOR_RGB(0xFF,  0xFF,   0xFF)); //this sets the button text color
    TFT.FgColor(0x007CC4);  // this sets the button color


    //back button
    TFT.Track( 50, 355, 100, 75, 4);    // track aree of button back
    TFT.DL(TAG(4));
    TFT.Button(50, 355, 100, 75, font_button, 0, text_back);                         // assign TAG value 4 to text_back button

    //backlight slider
    TFT.Track( (TFT.DispWidth*1/3), 200, (TFT.DispWidth*1/3), 50, 1);    // track aree of backlight slider
    TFT.DL(TAG(1));
    TFT.Slider((TFT.DispWidth*1/3), 200, (TFT.DispWidth*1/3), 50,0, val , 127 );                         // assign TAG value 5 to text_back button

//check if slider is touched
    if (tag == 1) {
        
        val = tracker/512;
        
        //limit the backlight to <91
       // if(val > 90)
//        {
//            val = 90;    
//        }
        
        TFT.Wr16(REG_PWM_DUTY, (val));
    }
   
    
    TFT.DL(TAG(0));     // assign TAG value 0 to bitmap
    //backlight text
    TFT.DL(COLOR_RGB(0x00, 0x00, 0x00));
    TFT.Text((TFT.DispWidth/2), (TFT.DispHeight*5/8), font_button, OPT_CENTERX, "backlight pwm value:");        // draw Text with font 31
    
    //sprintf(buffer, "%.1i", val+1);                     // write val to buffer
//    TFT.Text((TFT.DispWidth*3/4)-20, (TFT.DispHeight*5/8), font_title, OPT_CENTERX, buffer);        // draw Text with font 31
    
    TFT.Number((TFT.DispWidth*3/4)-40, (TFT.DispHeight*5/8)-15,font_title,0, val);


    TFT.DL(TAG(0));                         // assign TAG value 0 to everything else

    //title text
    TFT.DL(COLOR_RGB(0x00, 0x7C, 0xC4));
    TFT.Text((TFT.DispWidth/2), TFT.DispHeight/8, font_title, OPT_CENTERX, text_b);        // draw Text with font 31

    //author text
    TFT.DL(COLOR_RGB(0x00,  0x00,   0x00));
    TFT.Text((TFT.DispWidth*7/8)+9, (TFT.DispHeight*15/16), font_button, OPT_CENTERX, text_author);            // draw Text with font 31

    //display the screen
    TFT.DL(DISPLAY());                     // display the image
    TFT.Swap();                            // swap the current display list
    TFT.Flush_Co_Buffer();                 // download the command list into fifo
    TFT.WaitCmdfifo_empty();               // wait till coprocessor completes the operation
}



/************************
    function: mode_c
    description: draws mode C screen
*************************/
void mode_c(uint32_t tracker, uint8_t tag)
{
    //ISE40 STATUS DISPLAY
    static uint16_t data_raw_a9 = 0x0000;     
    data_raw_a9 = ((analog_in9.read_u16()>>7) & 0x1FF );
    
    //start new display list
    TFT.DLstart();                                                      // start a new display command list
    
    TFT.DL(TAG(0));                                                         // assign TAG value 0 to everything else

    
    if (data_raw_a9 > 0xFF)     // if greater than half of max input
    {
        //then ise40 "switch on!"    
        TFT.DL(CLEAR_COLOR_RGB(0,0xFF,0));      // set the clear color to green
        TFT.DL(CLEAR(1,1,1));                   // clear buffers -> color buffer,stencil buffer, tag buffer
        
        //message text
        TFT.DL(COLOR_RGB(0,0,0)); // set text color to white
        TFT.Text((TFT.DispWidth/2), (TFT.DispHeight/2), font_title, OPT_CENTERX, "SWITCH ON!");            // draw Text with font 31
    }
    else
    {
        //then ies40 "switch off!"  

        TFT.DL(CLEAR_COLOR_RGB(0xFF,0,0));      // set the clear color to green
        TFT.DL(CLEAR(1,1,1));                   // clear buffers -> color buffer,stencil buffer, tag buffer
        
        //message text
        TFT.DL(COLOR_RGB(0,0,0)); // set text color to white
        TFT.Text((TFT.DispWidth/2), (TFT.DispHeight/2), font_title, OPT_CENTERX, "SWITCH OFF!");            // draw Text with font 31  
    }
 
    TFT.Text((TFT.DispWidth/2), (TFT.DispHeight/8), font_title, OPT_CENTERX, "ISE40A");            // draw Text with font 31
    
    //buttons
    TFT.DL(COLOR_RGB(0xFF,  0xFF,   0xFF)); //this sets the button text color
    TFT.FgColor(0x007CC4);  // this sets the button color

    //back button
    TFT.Track( 50, 355, 100, 75, 4);    // track aree of button back
    TFT.DL(TAG(4));
    TFT.Button(50, 355, 100, 75, font_button, 4, text_back);                // assign TAG value 4 to text_back button

    TFT.DL(TAG(0));                                                         // assign TAG value 0 to everything else
    //author text
    TFT.DL(COLOR_RGB(0,0,0));
    TFT.Text((TFT.DispWidth*7/8)+9, (TFT.DispHeight*15/16), font_button, OPT_CENTERX, text_author);            // draw Text with font 31

    //display the screen
    TFT.DL(DISPLAY());                     // display the image
    TFT.Swap();                            // swap the current display list
    TFT.Flush_Co_Buffer();                 // download the command list into fifo
    TFT.WaitCmdfifo_empty();               // wait till coprocessor completes the operation
}



/************************
    function: plot_nothing
    description: clears plot area
*************************/
void plot_nothing(void)
{
    TFT.DL(COLOR_RGB(0x00,  0x00,   0x00));

}



/************************
    function: plot_square
    description: plots square wave
*************************/
void plot_square(void)
{
    static const int16_t square_wave_lookup[40] = { -100,-100,-100,-100,-100,   100,100,100,100,100,
            -100,-100,-100,-100,-100,   100,100,100,100,100,
            -100,-100,-100,-100,-100,   100,100,100,100,100,
            -100,-100,-100,-100,-100,   100,100,100,100,100
                                                  };

    TFT.DL(COLOR_RGB(0xFF,  0x00,   0x00));
    for(i = 0; i<40; i++) {
        TFT.DL( VERTEX2F(50*16 +18*i*16, 250*16 - (square_wave_lookup[i])*16) );
    }
}



/************************
    function: plot_triangle
    description: plots triangle wave
*************************/
void plot_triangle(void)
{
    static const int16_t tri_wave_lookup[40] = {    100,0,-100,0,   100,0,-100,0,
            100,0,-100,0,   100,0,-100,0,
            100,0,-100,0,   100,0,-100,0,
            100,0,-100,0,   100,0,-100,0,
            100,0,-100,0,   100,0,-100,0
                                               };
    TFT.DL(COLOR_RGB(0xFF,  0xFF,   0x00));
    for(i = 0; i<40; i++) {
        TFT.DL( VERTEX2F(50*16 +18*i*16, 250*16 - (tri_wave_lookup[i])*16) );
    }
}



///************************
//    function: plot_sine
//    description: plots sine wave
//*************************/
//void plot_sine(void)
//{
//    static const int16_t sine_wave_lookup[40] = {   0,  16,     31,     45,     59,     71,     81,     89,     95,     99,
//            100,  99,     95,     89,     81,     71,     59,     45,     31,     16,
//            -0, -16,    -31,    -45,    -59,    -71,    -81,    -89,    -95,    -99,
//            -100, -99,    -95,    -89,    -81,    -71,    -59,    -45,    -31,    -16,
//                                                };
//
//    TFT.DL(COLOR_RGB(0xFF,  0x00,   0xFF));
//    for(i = 0; i<40; i++) {
//        TFT.DL( VERTEX2F(50*16 +18*i*16, 250*16 + (sine_wave_lookup[i])*16) );
//    }
//}



/************************
    function: plot_data1
    description: plots analog_in data1
*************************/
void plot_sine(void)
{
    static const int16_t sine_wave_lookup[40] = {   0,  16,     31,     45,     59,     71,     81,     89,     95,     99,
            100,  99,     95,     89,     81,     71,     59,     45,     31,     16,
            -0, -16,    -31,    -45,    -59,    -71,    -81,    -89,    -95,    -99,
            -100, -99,    -95,    -89,    -81,    -71,    -59,    -45,    -31,    -16,
                                                };

    TFT.DL(COLOR_RGB(0xFF,  0x00,   0xFF));
    for(i = 0; i<40; i++) {
        TFT.DL( VERTEX2F(50*16 +18*i*16, 250*16 - (sine_wave_lookup[i])*16) );
    }
}



/************************
    function: collect_data
    description:  store ADC samples
*************************/
void collect_data(void)
{

    static const uint16_t num_pts = 100;
    
    static float    collect_time = 0,
                    plot_time = 0,
                    trigger_time = 0;
    static uint16_t i = 0x00,
                    j = 0x00;
    static Timer    timer;
    static float    begin[3]    = {0,0,0},  // {trigger, sample, plot}
                    end[3]      = {0,0,0};  // {trigger, sample, plot}

//    static const uint16_t threshold = 10;

    // data sample from ADC
    static uint16_t data_raw_a9[num_pts] = {};     
    
    timer.start();
    begin[0] = timer.read_us(); // trigger_begin


    //do while trigger[0] < trigger[1]
    //essentially find falling edge
    do
    {
        data_raw_a9[0] = ((analog_in9.read_u16()>>7) & 0x1FF );
        data_raw_a9[1] = ((analog_in9.read_u16()>>7) & 0x1FF );
    }
    while(data_raw_a9[1] < data_raw_a9[0]);
    
    do
    {
        data_raw_a9[0] = ((analog_in9.read_u16()>>7) & 0x1FF );
        data_raw_a9[1] = ((analog_in9.read_u16()>>7) & 0x1FF );
    }
    while(data_raw_a9[1] > data_raw_a9[0]); 
        //do
//    {
//        data_raw_a9[0] = ((analog_in9.read_u16()>>7) & 0x1FF );
//        data_raw_a9[1] = ((analog_in9.read_u16()>>7) & 0x1FF );
//    }
//    while(data_raw_a9[1] < data_raw_a9[0]);
//    
//    do
//    {
//        data_raw_a9[0] = ((analog_in9.read_u16()>>7) & 0x1FF );
//        data_raw_a9[1] = ((analog_in9.read_u16()>>7) & 0x1FF );
//    }
//    while(data_raw_a9[1] > data_raw_a9[0]);
    
    timer.stop();
    end[0] = timer.read_us();   //trigger_end
    
    
    timer.start();
    begin[1] = timer.read_us(); // sample_begin
    //collect data
    for(i=2;i<num_pts;i++)
    {
        //collect adc 9 most significant bits
        data_raw_a9[i] = ((analog_in9.read_u16()>>7) & 0x1FF );
    }
    timer.stop();
    end[1] = timer.read_us();   // sample_end
    
    
    timer.start();
    begin[2] = timer.read_us(); // plot_begin
    //green plot
    TFT.DL(COLOR_RGB(0x00,  0xFF,   0x00));
    //plot data
    for(i=0;i<num_pts;i++)
    {
        TFT.DL(VERTEX2F(8*i*16, TFT.DispHeight*16 - data_raw_a9[i]*15));    //  15/16 *512 = 480 which is disp_height
    }
    timer.stop();
    end[2] = timer.read_us();   // plot_end            
    
    
    trigger_time =(end[0]-begin[0])/1000;
    collect_time =(end[1]-begin[1])/1000;
    plot_time = (end[2]-begin[2])/1000;

    TFT.DL(COLOR_RGB(0xFF, 0xFF, 0xFF));
    //data points
    TFT.Text((TFT.DispWidth*7/8), TFT.DispHeight/32, font_button, OPT_RIGHTX, " data points: ");        // draw Text with font 31
    TFT.Number((TFT.DispWidth*7/8), (TFT.DispHeight/32),font_button,0,num_pts);
    
    //trigger time
    TFT.Text((TFT.DispWidth*7/8), TFT.DispHeight*3/32, font_button, OPT_RIGHTX, " trigger time [ms]: ");        // draw Text with font 31
    TFT.Number((TFT.DispWidth*7/8), (TFT.DispHeight*3/32),font_button,0,trigger_time);
    
    //sample time
    TFT.Text((TFT.DispWidth*7/8), TFT.DispHeight*5/32, font_button, OPT_RIGHTX, "sample time [ms]: ");        // draw Text with font 31
    TFT.Number((TFT.DispWidth*7/8), (TFT.DispHeight*5/32),font_button,0,collect_time);
    
    //plot time
    TFT.Text((TFT.DispWidth*7/8), TFT.DispHeight*7/32, font_button, OPT_RIGHTX, "plot time [ms]: ");        // draw Text with font 31
    TFT.Number((TFT.DispWidth*7/8), (TFT.DispHeight*7/32),font_button,0,plot_time);
        
}



/****************************************************************************/
/* Checks for an SD card and reads/writes the calibration data              */
/* from the SD card                                                         */
/****************************************************************************/
uint8_t read_calibration()
{
    static ft_char8_t temp_string[32] = {"\0"};
    static uint8_t tcal[24];                                                                                                   // touch screen Calibration

    while(sdcard_present.read()) {
        // checks for SD card, if not present hold program until a SD card is inserted
        error_screen("Error!","Insert SD card!");               // write error screen to LCD
    }

    sd.mount();
    FILE *fp= fopen("/sd/TCal/TCalData.txt", "r");
    if (fp != NULL) {
        //file successfully opened for reading!
        fread(temp_string,24,1,fp);
        for (i = 0; i < 24; i++) {
            // convert from int to char
            tcal[i] = temp_string[i];
        }
        TFT.write_calibrate(tcal);                              // write cal data to screen
        fclose(fp);
        sd.unmount();
        return 0;
    } else {
        //file not opened. might not exist.
        TFT.Calibrate();                                        // calibrate the touch screen
        TFT.read_calibrate(tcal);                               // read in cal data from screen
        mkdir("/sd/TCal", 0777);                                // open file for saving calibration data
        FILE *fp = fopen("/sd/TCal/TCalData.txt", "w");
        for (i = 0; i < 25; i++) {                              // save integers one at a time to the file
            fprintf(fp, "%c", tcal[i] );
        }
        fclose(fp);
        sd.unmount();
        return 1;

    }
}



/****************************************************************************
* tests functional operation of sd card by read/write
****************************************************************************/
uint8_t sd_test(void)
{
    while(sdcard_present.read()) {
        // checks for SD card, if not present hold program until a SD card is inserted
        error_screen("Error!","Insert SD card!");               // write error screen to LCD
    }

    TFT.DLstart();                         // start a new display command list
    TFT.DL(CLEAR_COLOR_RGB(0,0,0));    // clear the screen and set the background to green
    TFT.DL(CLEAR(1,1,1));                  // clear buffers -> color buffer,stencil buffer, tag buffer
    TFT.DL(COLOR_RGB(255,255,255));

    sd.mount();

    //Perform a write test
//    printf("\nWriting to SD card...");
    TFT.Text((TFT.DispWidth/2), 150, 28, OPT_CENTERX, "Writing to SD card...\0");                // draw Text with font 31
    FILE *fp = fopen("/sd/sdtest.txt", "w");
    if (fp != NULL) {
        //file successfully opened for writing!
        fprintf(fp, "We're writing to an SD card!");
        fclose(fp);
        TFT.DL(CLEAR_COLOR_RGB(0,255,0));    // clear the screen and set the background to green
        TFT.DL(CLEAR(1,1,1));                  // clear buffers -> color buffer,stencil buffer, tag buffer
        TFT.Text((TFT.DispWidth/2), 200, 28, OPT_CENTERX,"open file success\0" );                // draw Text with font 31

    } else {
//        printf("failed!\n");
        TFT.DL(CLEAR_COLOR_RGB(255,0,0));    // clear the screen and set the background to green
        TFT.DL(CLEAR(1,1,1));                  // clear buffers -> color buffer,stencil buffer, tag buffer
        TFT.Text((TFT.DispWidth/2), 200, 28, OPT_CENTERX, "open file failed!\0");                // draw Text with font 31
    }

    //Perform a read test
//    fprintf("Reading from SD card...");
    TFT.Text((TFT.DispWidth/2), 250, 28, OPT_CENTERX, "Reading from SD card...\0");                // draw Text with font 31

    fp = fopen("/sd/sdtest.txt", "r");
    if (fp != NULL) {
        char c = fgetc(fp);
        if (c == 'W') {
//            printf("success!\n");
            TFT.Text((TFT.DispWidth/2), 300, 28, OPT_CENTERX, "correct char!\0");                // draw Text with font 31

        } else {
//            fprintf("incorrect char (%c)!\n", c);
            TFT.Text((TFT.DispWidth/2), 300, 28, OPT_CENTERX, "incorrect char!\0");                // draw Text with font 31
        }
        sd.unmount();
        TFT.DL(DISPLAY());                                                      // display the image
        TFT.Swap();                                                             // swap the current display list
        TFT.Flush_Co_Buffer();                                                  // download the command list into fifo
        TFT.WaitCmdfifo_empty();                                                // wait till coprocessor completes the operation
        TFT.Sleep(2000);
        fclose(fp);
        return 0;

    } else {
//        printf("failed!\n");
        TFT.Text((TFT.DispWidth/2), 250, 31, OPT_CENTERX, "failed to read!\0");                // draw Text with font 31
        sd.unmount();
        TFT.DL(DISPLAY());                                                      // display the image
        TFT.Swap();                                                             // swap the current display list
        TFT.Flush_Co_Buffer();                                                  // download the command list into fifo
        TFT.WaitCmdfifo_empty();                                                // wait till coprocessor completes the operation
        TFT.Sleep(2000);
        return 1;
    }
}



uint8_t bitmap_test(uint16_t x,uint16_t y)
{

    TFT.DLstart();                         // start a new display command list
    TFT.DL(CLEAR_COLOR_RGB(0,0,0));    // clear the screen and set the background to green
    TFT.DL(CLEAR(1,1,1));                  // clear buffers -> color buffer,stencil buffer, tag buffer
    TFT.DL(COLOR_RGB(255,255,255));

    //following 3 lines can be replaced with TFT.set_bitmap(0x00, 0, )
    TFT.DL(BITMAP_SOURCE(RAM_G));
    TFT.DL(BITMAP_LAYOUT(ARGB1555,2*x_size,y_size));
    TFT.DL(BITMAP_SIZE(NEAREST,BORDER,BORDER,x_size,y_size));

    TFT.DL(BEGIN(BITMAPS));     //start drawing bitmaps
    TFT.DL(VERTEX2II(25,25,0,0));                                      // draw logo image with bit handle 0
    TFT.DL(END());

    TFT.Text((TFT.DispWidth/2), 170, 31, OPT_CENTERX, "bitmap_test() called\0");                // draw Text with font 31

    TFT.DL(DISPLAY());                                                      // display the image
    TFT.Swap();                                                             // swap the current display list
    TFT.Flush_Co_Buffer();                                                  // download the command list into fifo
    TFT.WaitCmdfifo_empty();                                                // wait till coprocessor completes the operation

    return 0;
}



/* function to load jpg file from filesystem */
/* return 0 if jpg is ok                     */
/* also return x_size and y_size of jpg      */

int8_t Load_jpg(char* filename, ft_int16_t* x_size, ft_int16_t* y_size, ft_uint32_t address)
{
    static int8_t rval = 0x00;

    while(sdcard_present.read()) {
        // checks for SD card, if not present hold program until a SD card is inserted
        error_screen("Error!","Insert SD card!");               // write error screen to LCD
    }

    sd.mount();
    rval = TFT.Load_jpg( filename, x_size, y_size, address);
    sd.unmount();
    return rval;
}



/************************
    function: display_message
    description: draws a pass/fail type message to the screen.
    useful for debugging return values from functions.
*************************/
void display_message(uint8_t error, const ft_char8_t *pass, const ft_char8_t *fail)
{
    TFT.DLstart();                         // start a new display command list
    TFT.DL(COLOR_RGB(0,0,0));

    if(error == 0x00) {
        TFT.DL(CLEAR_COLOR_RGB(0,255,0));    // clear the screen and set the background to green
        TFT.DL(CLEAR(1,1,1));                  // clear buffers -> color buffer,stencil buffer, tag buffer
        TFT.Text((TFT.DispWidth/2), 170, 31, OPT_CENTERX, pass);                // draw Text with font 31
    } else {
        TFT.DL(CLEAR_COLOR_RGB(255,0,0));    // clear the screen and set the background to red
        TFT.DL(CLEAR(1,1,1));                  // clear buffers -> color buffer,stencil buffer, tag buffer
        TFT.Text((TFT.DispWidth/2), 170, 31, OPT_CENTERX, fail);                // draw Text with font 31
    }

    TFT.DL(DISPLAY());                                                      // display the image
    TFT.Swap();                                                             // swap the current display list
    TFT.Flush_Co_Buffer();                                                  // download the command list into fifo
    TFT.WaitCmdfifo_empty();                                                // wait till coprocessor completes the operation
}



/************************
    function: Start_Screen
    description: draws boot up screen as images are loaded from the sd card
*************************/
ft_void_t start_screen(ft_char8_t *str)
{
    TFT.DLstart();                                             // start a new display command list
    TFT.DL(CLEAR_COLOR_RGB(255,255,255));                      // set the clear color to white
    TFT.DL(CLEAR(1,1,1));                                      // clear buffers -> color buffer,stencil buffer, tag buffer

    TFT.DL(COLOR_RGB(0x00,  0x7C,   0xC4));    // generate border in SMC blue, all functions are in 1/16 pixel format
    TFT.DL(BEGIN(LINES));
    TFT.DL(LINE_WIDTH(8 * 16));
    TFT.DL(VERTEX2F(0*16,0*16));
    TFT.DL(VERTEX2F(799*16,0*16));
    TFT.DL(VERTEX2F(799*16,0*16));
    TFT.DL(VERTEX2F(799*16,479*16));
    TFT.DL(VERTEX2F(799*16,479*16));
    TFT.DL(VERTEX2F(0*16,479*16));
    TFT.DL(VERTEX2F(0*16,479*16));
    TFT.DL(VERTEX2F(0*16,0*16));
    TFT.DL(END());

    TFT.Text((TFT.DispWidth/2), TFT.DispHeight/2, 31, OPT_CENTERX, str);        // draw Text with font 31
    TFT.Text((TFT.DispWidth/2), 350, 31, OPT_CENTERX, text_author);            // draw Text with font 31
    TFT.Spinner((TFT.DispWidth/2),TFT.DispHeight/4, 0,0);                       // draw a animated spinner

    TFT.DL(DISPLAY());                   // Display the image
    TFT.Swap();                          // Swap the current display list
    TFT.Flush_Co_Buffer();               // Download the command list into fifo
    TFT.WaitCmdfifo_empty();             // Wait till coprocessor completes the operation
}



/****************************************************************************/
/* This function displays the error screen                                  */
/* the function causes the screen to flash yellow and writes text to the LCD*/
/****************************************************************************/
void error_screen(ft_char8_t *str1, ft_char8_t *str2)
{
    TFT.DLstart();                         // start a new display command list
    TFT.DL(CLEAR_COLOR_RGB(255,242,0));    // clear the screen and set the background to yellow
    TFT.DL(CLEAR(1,1,1));                  // clear buffers -> color buffer,stencil buffer, tag buffer
    TFT.DL(COLOR_RGB(0,0,0));
    TFT.Text((TFT.DispWidth/2), 170, 31, OPT_CENTERX, str1);                // draw Text with font 31
    TFT.Text((TFT.DispWidth/2), 215, 31, OPT_CENTERX, str2);                // draw Text with font 31
    TFT.DL(DISPLAY());                                                      // display the image
    TFT.Swap();                                                             // swap the current display list
    TFT.Flush_Co_Buffer();                                                  // download the command list into fifo
    TFT.WaitCmdfifo_empty();                                                // wait till coprocessor completes the operation
    TFT.Sleep(500);

    TFT.DLstart();                                                          // start a new display command list
    TFT.DL(CLEAR_COLOR_RGB(255,255,255));                                   // set clear color to white
    TFT.DL(CLEAR(1,1,1));                                                   // clear buffers -> color buffer,stencil buffer, tag buffer;
    TFT.DL(COLOR_RGB(0,0,0));
    TFT.Text((TFT.DispWidth/2), 170, 31, OPT_CENTERX, str1);                // draw Text with font 31
    TFT.Text((TFT.DispWidth/2), 215, 31, OPT_CENTERX, str2);                // draw Text with font 31
    TFT.DL(DISPLAY());                                                      // display the image
    TFT.Swap();                                                             // swap the current display list
    TFT.Flush_Co_Buffer();                                                  // download the command list into fifo
    TFT.WaitCmdfifo_empty();                                                // wait till coprocessor completes the operation
    TFT.Sleep(500);
}