#ifndef SIMPLE_FIFO_H
#define SIMPLE_FIFO_H

/*
SIMPLE_FIFO was designed as an example of a FIFO construct using "char" data.
The defined type structure called SIMPLE_FIFO uses an array of type "char" and
two integer pointers "write_pointer" and "read_pointer" to form a circuular buffer.
The number of elements in the circular buffer is defined by SIMPLE_FIFO_SIZE.

The following functions are defined for the SIMPLE_FIFO type:
void simple_fifo_init(SIMPLE_FIFO *f)
int simple_fifo_writeable(SIMPLE_FIFO *f)
int simple_fifo_readable(SIMPLE_FIFO *f)
int simple_fifo_write(SIMPLE_FIFO *f, char c)
int simple_fifo_read(SIMPLE_FIFO *f, char *c)
char simple_fifo_putc(SIMPLE_FIFO *f, char c)
char simple_fifo_getc(SIMPLE_FIFO *f, char *c)
void simple_fifo_puts(SIMPLE_FIFO *f, char str[])
void simple_fifo_gets(SIMPLE_FIFO *f, char str[], int count)

The function simple_fifo_init initializes both pointers to 0 and initializes all data in the fifo to 0.
The function simple_fifo_writeable returns true if space is available in the fifo and returns false if the fifo is full.
The function simple_fifo_readable returns true if data is available in the fifo and returns false if the fifo is empty.
The function simple_fifo_write returns true if if data was successfully written to the fifo and returns false if the fifo is full.
The function simple_fifo_read returns true if if data was successfully read from the fifo and returns false if the fifo is empty.
The function simple_fifo_putc blocks until the fifo is writeable then writes a single character to the fifo.  The return value is the character written.
The function simple_fifo_getc blocks until the fifo is readable then reads a single character from the fifo.  The return value is the character read.
The function simple_fifo_puts blocks until the fifo is writeable then writes a single character to the fifo.  This is repeated until all characters in the null-terminated string out written to the fifo.  
The function simple_fifo_getc blocks until the fifo is readable then reads a single character from the fifo.  This is repeated until the carriage return character is encountered.  The resulting string is returned as null-terminated without the carriage return caracter included.
*/

#ifndef TRUE
#define TRUE ( 1 == 1 )
#endif

#ifndef FALSE
#define FALSE ( 1 == 0 )
#endif

#ifndef SIMPLE_FIFO_SIZE 
#define SIMPLE_FIFO_SIZE 16
#endif

#ifndef SIMPLE_FIFO
typedef struct simple_fifo
{
char buffer[SIMPLE_FIFO_SIZE];
int write_pointer;
int read_pointer;
} SIMPLE_FIFO;
#endif

void simple_fifo_init(SIMPLE_FIFO *f)
{
    f->write_pointer=0;
    f->read_pointer=0;
    for(int i=0; i<SIMPLE_FIFO_SIZE; i++)
    { 
        f->buffer[i]=0; 
    }
}

int simple_fifo_writeable(SIMPLE_FIFO *f)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    return(TRUE);
}

int simple_fifo_readable(SIMPLE_FIFO *f)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    return(TRUE);
}

int simple_fifo_write(SIMPLE_FIFO *f, char c)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    f->buffer[f->write_pointer] = c;
    f->write_pointer = (f->write_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}

int simple_fifo_read(SIMPLE_FIFO *f, char *c)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    *c = f->buffer[f->read_pointer];
    f->read_pointer = (f->read_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}

char simple_fifo_putc(SIMPLE_FIFO *f, char c)
{
    while(!simple_fifo_writeable(f)) {}
    simple_fifo_write(f, c);
    return(c);
}

char simple_fifo_getc(SIMPLE_FIFO *f, char *c)
{
    while(!simple_fifo_readable(f)) {}
    simple_fifo_read(f, c);
    return(*c);

}

void simple_fifo_puts(SIMPLE_FIFO *f, char str[])
{
    for(int i=0; str[i]!='\0';i++)
    {
        simple_fifo_putc(f, str[i]);
    }
}

void simple_fifo_gets(SIMPLE_FIFO *f, char str[], int count)
{
    for(int i=0; i<count; i++)
    {
        simple_fifo_getc(f, &str[i]);
        if(str[i] == '\r' || str[i] == '\n')
        {
            str[i]='\0';
            break;
        }
    }
    str[count-1]='\0';
}

/*
In addition to the SIMPLE_FIFO type, a number of similar fifo types are constructed for various data types.
The following fifo types are defined:
FIFO_CHAR   //for character data, similar to SIMPLE_FIFO above
FIFO_INT   //for default integer type
FIFO_U8     //for uint8_t data
FIFO_U16    //for uint16_t data
FIFO_U32    //for uint32_t data
FIFO_I8     //for int8_t data
FIFO_I16    //for int16_t data
FIFO_I32    //for int32_t data
FIFO_FLOAT  //for float data
FIFO_DOUBLE  //for double data

The following functions are defined for the FIFO_?? types:
void fifo_??_init(FIFO_?? *f)
int fifo_??_writeable(FIFO_?? *f)
int fifo_??_readable(FIFO_?? *f)
int fifo_??_write(FIFO_?? *f, char c)
int fifo_??_read(FIFO_?? *f, char *c)

The function fifo_??_init initializes both pointers to 0 and initializes all data in the fifo to 0.
The function fifo_??_writeable returns true if space is available in the fifo and returns false if the fifo is full.
The function fifo_??_readable returns true if data is available in the fifo and returns false if the fifo is empty.
The function fifo_??_write returns true if if data was successfully written to the fifo and returns false if the fifo is full.
The function sfifo_??_read returns true if if data was successfully read from the fifo and returns false if the fifo is empty.
*/

#include <stdint.h>


#ifndef FIFO_CHAR
typedef struct fifo_char
{
char buffer[SIMPLE_FIFO_SIZE];
int write_pointer;
int read_pointer;
} FIFO_CHAR;
#endif

void fifo_char_init(FIFO_CHAR *f)
{
    f->write_pointer=0;
    f->read_pointer=0;
    for(int i=0; i<SIMPLE_FIFO_SIZE; i++)
    { 
        f->buffer[i]=0; 
    }
}

int fifo_char_writeable(FIFO_CHAR *f)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    return(TRUE);
}

int fifo_char_readable(FIFO_CHAR *f)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    return(TRUE);
}

int fifo_char_write(FIFO_CHAR *f, char c)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    f->buffer[f->write_pointer] = c;
    f->write_pointer = (f->write_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}

int fifo_char_read(FIFO_CHAR *f, char *c)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    *c = f->buffer[f->read_pointer];
    f->read_pointer = (f->read_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}

char fifo_char_putc(FIFO_CHAR *f, char c)
{
    while(!fifo_char_writeable(f)) {}
    fifo_char_write(f, c);
    return(c);
}

char fifo_char_getc(FIFO_CHAR *f, char *c)
{
    while(!fifo_char_readable(f)) {}
    fifo_char_read(f, c);
    return(*c);

}

void fifo_char_puts(FIFO_CHAR *f, char str[])
{
    for(int i=0; str[i]!='\0';i++)
    {
        fifo_char_putc(f, str[i]);
    }
}

void fifo_char_gets(FIFO_CHAR *f, char str[], int count)
{
    for(int i=0; i<count; i++)
    {
        fifo_char_getc(f, &str[i]);
        if(str[i] == '\r' || str[i] == '\n')
        {
            str[i]='\0';
            break;
        }
    }
    str[count-1]='\0';
}



#ifndef FIFO_INT
typedef struct fifo_int
{
int buffer[SIMPLE_FIFO_SIZE];
int write_pointer;
int read_pointer;
} FIFO_INT;
#endif

void fifo_int_init(FIFO_INT *f)
{
    f->write_pointer=0;
    f->read_pointer=0;
    for(int i=0; i<SIMPLE_FIFO_SIZE; i++)
    { 
        f->buffer[i]=0; 
    }
}

int fifo_int_writeable(FIFO_INT *f)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    return(TRUE);
}

int fifo_int_readable(FIFO_INT *f)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    return(TRUE);
}

int fifo_int_write(FIFO_INT *f, int c)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    f->buffer[f->write_pointer] = c;
    f->write_pointer = (f->write_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}

int fifo_int_read(FIFO_INT *f, int *c)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    *c = f->buffer[f->read_pointer];
    f->read_pointer = (f->read_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}

#ifndef FIFO_U8
typedef struct fifo_u8
{
uint8_t buffer[SIMPLE_FIFO_SIZE];
int write_pointer;
int read_pointer;
} FIFO_U8;
#endif

void fifo_u8_init(FIFO_U8 *f)
{
    f->write_pointer=0;
    f->read_pointer=0;
    for(int i=0; i<SIMPLE_FIFO_SIZE; i++)
    { 
        f->buffer[i]=0; 
    }
}

int fifo_u8_writeable(FIFO_U8 *f)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    return(TRUE);
}

int fifo_u8_readable(FIFO_U8 *f)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    return(TRUE);
}

int fifo_u8_write(FIFO_U8 *f, uint8_t c)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    f->buffer[f->write_pointer] = c;
    f->write_pointer = (f->write_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}

int fifo_u8_read(FIFO_U8 *f, uint8_t *c)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    *c = f->buffer[f->read_pointer];
    f->read_pointer = (f->read_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}

#ifndef FIFO_U16
typedef struct fifo_u16
{
uint16_t buffer[SIMPLE_FIFO_SIZE];
int write_pointer;
int read_pointer;
} FIFO_U16;
#endif

void fifo_u16_init(FIFO_U16 *f)
{
    f->write_pointer=0;
    f->read_pointer=0;
    for(int i=0; i<SIMPLE_FIFO_SIZE; i++)
    { 
        f->buffer[i]=0; 
    }
}

int fifo_u16_writeable(FIFO_U16 *f)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    return(TRUE);
}

int fifo_u16_readable(FIFO_U16 *f)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    return(TRUE);
}

int fifo_u16_write(FIFO_U16 *f, uint16_t c)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    f->buffer[f->write_pointer] = c;
    f->write_pointer = (f->write_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}

int fifo_u16_read(FIFO_U16 *f, uint16_t *c)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    *c = f->buffer[f->read_pointer];
    f->read_pointer = (f->read_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}

#ifndef FIFO_U32
typedef struct fifo_u32
{
uint32_t buffer[SIMPLE_FIFO_SIZE];
int write_pointer;
int read_pointer;
} FIFO_U32;
#endif

void fifo_u32_init(FIFO_U32 *f)
{
    f->write_pointer=0;
    f->read_pointer=0;
    for(int i=0; i<SIMPLE_FIFO_SIZE; i++)
    { 
        f->buffer[i]=0; 
    }
}

int fifo_u32_writeable(FIFO_U32 *f)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    return(TRUE);
}

int fifo_u32_readable(FIFO_U32 *f)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    return(TRUE);
}

int fifo_u32_write(FIFO_U32 *f, uint32_t c)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    f->buffer[f->write_pointer] = c;
    f->write_pointer = (f->write_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}

int fifo_u32_read(FIFO_U32 *f, uint32_t *c)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    *c = f->buffer[f->read_pointer];
    f->read_pointer = (f->read_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}

#ifndef FIFO_I8
typedef struct fifo_i8
{
int8_t buffer[SIMPLE_FIFO_SIZE];
int write_pointer;
int read_pointer;
} FIFO_I8;
#endif

void fifo_i8_init(FIFO_I8 *f)
{
    f->write_pointer=0;
    f->read_pointer=0;
    for(int i=0; i<SIMPLE_FIFO_SIZE; i++)
    { 
        f->buffer[i]=0; 
    }
}

int fifo_i8_writeable(FIFO_I8 *f)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    return(TRUE);
}

int fifo_i8_readable(FIFO_I8 *f)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    return(TRUE);
}

int fifo_i8_write(FIFO_I8 *f, int8_t c)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    f->buffer[f->write_pointer] = c;
    f->write_pointer = (f->write_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}

int fifo_i8_read(FIFO_I8 *f, int8_t *c)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    *c = f->buffer[f->read_pointer];
    f->read_pointer = (f->read_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}


#ifndef FIFO_I16
typedef struct fifo_i16
{
int16_t buffer[SIMPLE_FIFO_SIZE];
int write_pointer;
int read_pointer;
} FIFO_I16;
#endif

void fifo_i16_init(FIFO_I16 *f)
{
    f->write_pointer=0;
    f->read_pointer=0;
    for(int i=0; i<SIMPLE_FIFO_SIZE; i++)
    { 
        f->buffer[i]=0; 
    }
}

int fifo_i16_writeable(FIFO_I16 *f)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    return(TRUE);
}

int fifo_i16_readable(FIFO_I16 *f)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    return(TRUE);
}

int fifo_i16_write(FIFO_I16 *f, int16_t c)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    f->buffer[f->write_pointer] = c;
    f->write_pointer = (f->write_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}

int fifo_i16_read(FIFO_I16 *f, int16_t *c)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    *c = f->buffer[f->read_pointer];
    f->read_pointer = (f->read_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}

#ifndef FIFO_I32
typedef struct fifo_i32
{
int32_t buffer[SIMPLE_FIFO_SIZE];
int write_pointer;
int read_pointer;
} FIFO_I32;
#endif

void fifo_i32_init(FIFO_I32 *f)
{
    f->write_pointer=0;
    f->read_pointer=0;
    for(int i=0; i<SIMPLE_FIFO_SIZE; i++)
    { 
        f->buffer[i]=0; 
    }
}

int fifo_i32_writeable(FIFO_I32 *f)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    return(TRUE);
}

int fifo_i32_readable(FIFO_I32 *f)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    return(TRUE);
}

int fifo_i32_write(FIFO_I32 *f, int32_t c)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    f->buffer[f->write_pointer] = c;
    f->write_pointer = (f->write_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}

int fifo_i32_read(FIFO_I32 *f, int32_t *c)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    *c = f->buffer[f->read_pointer];
    f->read_pointer = (f->read_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}


#ifndef FIFO_FLOAT
typedef struct fifo_float
{
float buffer[SIMPLE_FIFO_SIZE];
int write_pointer;
int read_pointer;
} FIFO_FLOAT;
#endif

void fifo_float_init(FIFO_FLOAT *f)
{
    f->write_pointer=0;
    f->read_pointer=0;
    for(int i=0; i<SIMPLE_FIFO_SIZE; i++)
    { 
        f->buffer[i]=0; 
    }
}

int fifo_float_writeable(FIFO_FLOAT *f)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    return(TRUE);
}

int fifo_float_readable(FIFO_FLOAT *f)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    return(TRUE);
}

int fifo_float_write(FIFO_FLOAT *f, float c)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    f->buffer[f->write_pointer] = c;
    f->write_pointer = (f->write_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}

int fifo_float_read(FIFO_FLOAT *f, float *c)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    *c = f->buffer[f->read_pointer];
    f->read_pointer = (f->read_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}


#ifndef FIFO_DOUBLE
typedef struct fifo_double
{
double buffer[SIMPLE_FIFO_SIZE];
int write_pointer;
int read_pointer;
} FIFO_DOUBLE;
#endif

void fifo_double_init(FIFO_DOUBLE *f)
{
    f->write_pointer=0;
    f->read_pointer=0;
    for(int i=0; i<SIMPLE_FIFO_SIZE; i++)
    { 
        f->buffer[i]=0; 
    }
}

int fifo_double_writeable(FIFO_DOUBLE *f)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    return(TRUE);
}

int fifo_double_readable(FIFO_DOUBLE *f)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    return(TRUE);
}

int fifo_double_write(FIFO_DOUBLE *f, double c)
{
    if(((f->write_pointer+1)% SIMPLE_FIFO_SIZE) == f->read_pointer)
    {
        return(FALSE);     /* buffer full, can't write */  
    }
    f->buffer[f->write_pointer] = c;
    f->write_pointer = (f->write_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}

int fifo_double_read(FIFO_DOUBLE *f, double *c)
{
    if(f->read_pointer == f->write_pointer)
    {
        return(FALSE); /* buffer empty, can't read */
    }
    *c = f->buffer[f->read_pointer];
    f->read_pointer = (f->read_pointer+1)% SIMPLE_FIFO_SIZE;
    return(TRUE);
}






#endif /* #ifndef SIMPLE_FIFO_H */
