/**
*   @file   Tracker.h
*
*   @brief  Tracker class implements API for sensing the position of the
*           dump truck.
*
*   @author Terrabots Team
*
*/

#ifndef TRACKER_H
#define TRACKER_H

#include "mbed.h"
#include "QEI.h"

/**
*   @class Tracker
*
*   @brief  Interface for tracking the position of a Dump Truck configured
*   "robot."
*/

#define STRAIGHT    0
#define CCW         1
#define CW          2
#define PI          3.14159265

class Tracker {
    public:
        /**
        *   Constructor for the Tracker object.
        *
        *   @param[in]  _chA     The interrupt pin used for channel A
        *   @param[in]  _chB     The interrupt pin used for channel B
        *   @param[in]  _pot     The AnalogIn pin used for the potentiometer
        */
        Tracker(PinName _chA, PinName _chB, PinName _pot);
        /**
        *   Set initial position and distance traveled to 0.
        */
        void clear();
        /**
        *   Set a reference "zero angle" for the potentiometer and wait five
        *   seconds for the user to set a turning angle.
        */
        void potSetup();
        /**
        *   Set a direction, either clockwise or counterclockwise, for the
        *   dump truck, based on the turning angle.
        */
        void setDirection();
        /**
        *   Check if the dump truck has moved before starting position-tracking
        *   calculations.
        */
        void checkToStart();
        /**
        *   Calculate the x and y position of the dump truck based on sensor
        *   measurements.
        */
        void calcDisplacement();
    private:
        /**
        *   The QEI object created with the QEI library to interpret encoder
        *   measurements.
        */
        QEI *wheel;
        /**
        *   The interrupt pin used for channel A.
        */
        PinName chA;
        /**
        *   The interrupt pin used for channel B.
        */
        PinName chB;
        /**
        *   The AnalogIn pin used for the potentiometer.
        */
        AnalogIn pot;
        /**
        *   The number of pulses recorded by the encoder.
        */
        float pulseCount;
        /**
        *   The type of encoding (X2 or X4) used for the encoder.
        */
        float encoding;
        /**
        *   The circumference of the front dump truck wheel.
        */
        float circumference;
        /**
        *   Equal to 1360. This constant is used to convert pulses counted to
        *   inches traveled.
        */
        float constant;
        /**
        *   The distance traveled by the dump truck.
        */
        float distance;
        /**
        *   The horizontal position of the dump truck.
        */
        float x;
        /**
        *   The vertical position of the dump truck.
        */
        float y;
        /**
        *   The raw analog value of the potentiometer
        */
        float value;
        /**
        *   The angle at which the dump truck is turning.
        */
        float turnAngle;
        /**
        *   The angle at which the dump truck is driving straight forward.
        */
        float zeroAngle;
        /**
        *   The direction at which the dump truck is turning.
        */
        int dir;
        /**
        *   The distance from the front wheel to the turning axle.
        */
        float df;
        /**
        *   The distance from the rear wheel to the turning axle.
        */
        float db;
        float W;
        float R;
        float dispAngle;

};
#endif