/**
*   @file   DumpTruck.h
*
*   @brief  DumpTruck class integrates multiple sensors into one API.
*
*   @author Terrabots Team
*
*/

#ifndef MBED_DUMP_TRUCK_H
#define MBED_DUMP_TRUCK_H

#include "mbed.h"
#include "Tracker.h"
#include "Motor.h"
#include "nRF24L01P.h"
#include "SRF05.h"
#include "commands.h"

#define TRANSFER_SIZE   8

/**
*   @class DumpTruck
*
*   @brief  Interface for controlling a Dump truck.
*/

class DumpTruck {
public:
    /**
    *   Constructor for the Tracker object.
    *
    *   @param[in]  truckId     The dump truck's identifier
    */
    DumpTruck(int truckId);

    /**
    *   Process command, parsing it and calling the associated function.
    */
    void processCommand();

    /**
    *   Send not acknowledge to base station.
    */

    void sendNack();

    /**
    *   Send acknowledge to base station.
    */

    void sendAck();
    
    /**
    *   Initialize transceiver and other miscellaneous devices.
    */
    void startComms();
    
    /**
    *   Receive a command from base station.
    */
    void getCommand();
    
    /**
    *   Send important data back to the base station.
    */
    void reportData();

    /**
    *   Drive a certain distance at a desired speed.
    */
    void driveDistance();// frontMotor
    
    /**
    *   Drive at a desired speed.
    */
    void drive();
    
    /**
    *   Turn the rear wheels a certain angle.
    */
    void turn();                 // turnMotor
    
    /**
    *   Move the dump truck's bed up/down a certain angle.
    */
    void moveBed();  // bedMotor
    
    /**
    *   Stop driving.
    */
    void stopDrive();                            // all Motors
    
    /**
    *   Stop turning.
    */
    void stopTurn();                            // all Motors
    
    /**
    *   Stop all motors if an object comes within range of the
    *   ultrasonic sensor
    */
    void detectStop();

    /**
    *   Stop bed.
    */
    void stopBed();                            // all Motors

    /**
    *   Read from the ultrasonic sensor and determine the dump truck's
    *   proximity from objects.
    */
    bool detect();                          // returns bool for object too close
    
    int txDataCnt;
    int rxDataCnt;
    char txData[TRANSFER_SIZE], rxData[TRANSFER_SIZE];
    char acked[TRANSFER_SIZE];
    char nacked[TRANSFER_SIZE]; 
    bool received;
    int numCommands;
    int commandValue;
    typedef void (DumpTruck::*funcArray)(void);
    funcArray commArr[sizeof(commands)/sizeof(*commands)];
    float zeroAngle;
    bool interrupted;
    
protected:    
    Tracker *track;
    //IMU *bed;
    Motor *frontMotor;
    Motor *turnMotor;
    Motor *bedMotor;
    nRF24L01P *nrf; //Transceiver *nrf;
    SRF05 *srf;
    AnalogIn *pot;

    // add direct control for motor, switch, and ultrasonic sensor.

    int truckNumber;

    // motor variables
    float speed;                            // drive
    float distance;                         // drive
    float potAngle;                         // turn
    float bedAngle;                         // bed

    // bed-specific variables
    bool switchState;   // BusIn the two limit switches

    // ultrasonic-specific variables
    bool tooClose;
    float proximity;
};
#endif
