/* Copyright (C) 2016 Telesensorics Inc, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
 

#ifndef __DYNAMIXEL_BUS_H__
#define __DYNAMIXEL_BUS_H__

#include "mbed.h"

#include <vector>
using namespace std;


enum InstructionTable
{
    instructionPing            = 0x1,          // no action. obtain status packet.
    instructionRead            = 0x2,          // write values in control table
    instructionWrite           = 0x3,          // read values in control table
    instructionWriteAction     = 0x4,          // write actions to be performed upon triggering
    instructionTriggerAction   = 0x5,          // trigger stored actions
    instructionReset           = 0x6,          // changes ALL control table vaalues to factory default
    instructionSyncWrite       = 0x83          // write to many servos at once
};


enum StatusCodes
{
    statusValid       = 0x80,       // REVIEW: Not in spec. I am overloading this for Ping, such that a 0 return value = unit not found
    instructionError  = 0x40,
    overloadError     = 0x20,
    checksumError     = 0x10,
    rangeError        = 0x08,
    overheatingError  = 0x04,
    angleLimitError   = 0x02,
    inputVoltageError = 0x01
};

typedef unsigned char StatusCode;

typedef vector<unsigned char> CommBuffer;

typedef unsigned char ServoId;

typedef pair< ServoId, CommBuffer > SyncIdDataPair;
    
class DynamixelBus
{
public:
    DynamixelBus( PinName txUartPin, PinName rxUartPin, PinName txSelectPin, PinName rxSelectPin, int baud );
    
    StatusCode  Ping( ServoId id );
    StatusCode  Read( ServoId id, unsigned char controlTableStart, unsigned char bytesToRead, CommBuffer& data );
    StatusCode  Write( ServoId id, unsigned char controlTableStart, const CommBuffer& dataToWrite );
    StatusCode  WriteAction( ServoId id, unsigned char controlTableStart, const CommBuffer& dataToWrite );
    void        TriggerAction( ServoId id );
    StatusCode  HardReset( ServoId id );
    StatusCode  SyncWrite( unsigned char controlTableStart, unsigned char bytesToWrite, vector<SyncIdDataPair>& data );

private:
    unsigned char CalculateTxPacketChecksum( CommBuffer& packet );
    StatusCode send_ping( ServoId id );
    StatusCode send_write( ServoId id, unsigned char controlTableStart, const CommBuffer& dataToWrite );
    StatusCode send_read( ServoId id, unsigned char controlTableStart, unsigned char bytesToRead, CommBuffer& data );
        
private:
    Serial      _uart;      // The uart the servos live on.
    DigitalOut _txSelect;   // Turn on to enable the tri-state buffer before sending data over the bus. Turn off when data has been sent.
    DigitalOut _rxSelect;   // Turn on to enable the tri-state buffer before receiving data over the bus. Turn off when data has been received.
    
    float _replyDelay;      // The delay time between command being sent and response being sent by the servo.
    int _baud;              // baud of all connected servos
    float _responseTimeout;
};

#endif // __DYNAMIXEL_BUS_H__