/* Copyright (C) 2016 Telesensorics Inc, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
 * and associated documentation files (the "Software"), to deal in the Software without restriction, 
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, 
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or 
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING 
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef MBED_AX12_H
#define MBED_AX12_H

#include "mbed.h"
#include "DynamixelBus.h"


enum ControlTable
{
    ctModelNumberLow      = 0,        // RD
    ctModelNumberHigh     = 1,        // RD
    ctFirmwareVersion     = 2,        // RD
    ctID                  = 3,        // RD/WR
    ctBaudRate            = 4,        // RD/WR
    ctReturnDelayTime     = 5,        // RD/WR
    ctCWAngleLimitL       = 6,        // RD/WR
    ctCWAngleLimitH       = 7,        // RD/WR
    ctCCWAngleLimitL      = 8,        // RD/WR
    ctCCWAngleLimitH      = 9,        // RD/WR
    ctReserved1           = 10,
    ctTemperatureLimit    = 11,       // RD/WR
    ctVoltageLow          = 12,       // RD/WR
    ctVoltageHigh         = 13,       // RD/WR
    ctMaxTorqueL          = 14,       // RD/WR
    ctMaxTorqueH          = 15,       // RD/WR
    ctStatusReturnLevel   = 16,       // RD/WR
    ctAlarmLED            = 17,       // RD/WR
    ctAlarmShutdown       = 18,       // RD/WR
    ctReserved2           = 19,
    ctDownCalibrationL    = 20,       // RR
    ctDownCalibrationH    = 21,       // RD
    ctUpCalibrationL      = 22,       // RD
    ctUpCalibrationH      = 23,       // RD
    ctTorqueEnable        = 24,       // RD/WR
    ctLED                 = 25,       // RD/WR
    ctCWComplianceMargin  = 26,       // RD/WR
    ctCCWComplianceMargin = 27,       // RD/WR
    ctCWComplianceSlope   = 28,       // RD/WR
    ctCCWComplianceSlope  = 29,       // RD/WR
    ctGoalPositionL       = 30,       // RD/WR
    ctGoalPositionH       = 31,       // RD/WR
    ctMovingSpeedL        = 32,       // RD/WR
    ctMovingSpeedH        = 33,       // RD/WR
    ctTorqueLimitL        = 34,       // RD/WR
    ctTorqueLimitH        = 35,       // RD/WR
    ctPresentPositionL    = 36,       // RD
    ctPresentPositionH    = 37,       // RD
    ctPresentSpeedL       = 38,       // RD
    ctPresentSpeedH       = 39,       // RD
    ctPresentLoadL        = 40,       // RD
    ctPresentLoadH        = 41,       // RD
    ctPresentVoltage      = 42,       // RD
    ctPresentTemperature  = 43,       // RD
    ctRegisteredInst      = 44,       // RD/WR
    ctReserved3           = 45,       
    ctMoving              = 46,       // RD
    ctLock                = 47,       // RD/WR
    ctPunchL              = 48,       // RD/WR
    ctPunchH              = 49        // RD/WR
};

// to reduce time reading values from servo, 
// we read values as a block from PresentPosition through Moving
// these are cached until ClearCache call
enum CachedControlTable
{
    cctPresentPositionL    = 36 - 36,
    cctPresentPositionH    = 37 - 36,
    cctPresentSpeedL       = 38 - 36,
    cctPresentSpeedH       = 39 - 36,
    cctPresentLoadL        = 40 - 36,
    cctPresentLoadH        = 41 - 36,
    cctPresentVoltage      = 42 - 36,
    cctPresentTemperature  = 43 - 36,
    cctRegisteredInst      = 44 - 36,
    cctReserved3           = 45 - 36,
    cctMoving              = 46 - 36
};

const unsigned char AX12ReadBlockStart  = ctPresentPositionL;
const unsigned char AX12ReadBlockLength = ctLock - ctPresentPositionL;

class AX12
{

public:
    AX12(DynamixelBus* pbus, ServoId = 1);

    // clear cache to force actual read from servo
    void ClearCache();
    
    // sets the servo ID
    StatusCode SetServoId(char newId);

    // returns the status of the servo (connected/not connected + errors)
    StatusCode Ping(void);

    // Sets the goal position of the servo in degrees
    StatusCode SetGoal(float degrees);

    // determines if the servo is moving
    bool IsMoving(void);

    // gets the position of the servo in degrees
    float GetPosition(void);

    // gets the internal temperature of the servo
    int GetTemperature(void);

    // gets the servo power supply voltage
    float GetSupplyVoltage(void);
    
    // Sets the replay delay for dynamixel protocol
    StatusCode SetReplyDelay(int us);

    // gets the servo load
    float GetLoad(void);
    
    // enables/disables holding torque 
    StatusCode TorqueEnable( bool );

    // get last error
    virtual int GetLastError();
   
    // check if has error
    virtual bool HasError();

    
private :
    ServoId _ID;
    DynamixelBus* _pbus;
    
    unsigned char _LastError;
    
    unsigned char _ReadCache[AX12ReadBlockLength];
    bool _HasReadCache;

    StatusCode LoadReadCache();
};
#endif
