/* mbed MCP4822 Library, for driving the 12-Bit DAC with internal Vref and SPI interface
 * Copyright (c) 2015, Created by Steen Joergensen (stjo2809)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
#include "mbed.h"

#ifndef MBED_MCP4822_H
#define MBED_MCP4822_H

//=============================================================================
// Declaration of variables & custom #defines
//=============================================================================

#define CB_OUTPUT_A     0x00       // commad bit for output A
#define CB_OUTPUT_B     0x80       // commad bit for output A
#define CB_GAIN_1X      0x20       // commad bit for gain 1x
#define CB_GAIN_2X      0x00       // commad bit for gain 2x
#define CB_SHDN         0x10       // commad bit for Output enabled
#define CB_NSHDN        0x00       // commad bit for Output buffer disabled, output is high-impedance

//=============================================================================
// Functions Declaration
//=============================================================================

/** Interface to the 12-Bit DAC with internal Vref and SPI interface
 *
  *  Using the driver:
 *   - remenber to setup SPI in main routine or use pins instance.
 *
 *  Defaults in this driver on start up:
 *   - Datasheet start up
 */
class MCP4822 {
public:
    /** Create an instance of the MCP4822 connected via specfied SPI instance.
     *
     * @param spi The mbed SPI instance (make in main routine)
     * @param nLDAC The Latch DAC Synchronization pin 
     * @param nCs The SPI chip select pin.
     */
    MCP4822(SPI& spi, PinName nLDAC, PinName nCs);
    
    /** Create an instance of the MCP4261 connected with SPI pins.
     *
     * @param nLDAC The Latch DAC Synchronization pin 
     * @param mosi The SPI Master Output, Slave Input pin.
     * @param miso The SPI Master Input, Slave Output pin. 
     * @param sck The SPI Serial Clock pin.
     * @param nCs The SPI chip select pin.
     */
    MCP4822(PinName nLDAC, PinName mosi, PinName miso,PinName sck, PinName nCs);
    

    /** Write to output A.
     *
     * @param gain the gain is 2x = '0' and 1x = '1'.
     * @param data The 12 bits value to write to the output.
     */
    void a(bool gain, int data);
    
    /** Write to output B.
     *
     * @param gain the gain is 2x = '0' and 1x = '1'.
     * @param data The 12 bits value to write to the output.
     */
    void b(bool gain, int data);
    
    /** Output Synchronization.
     *
     * @param act The LDAC is Active = true and Inactive = false.
     */
    void ldac(bool act);
    
    /** Output Shutdown.
     *
     * @param output A = '0' and B = '1'.
     * @param act The SHDN is Active = true and Inactive = false.
     */
    void shdn(bool output, bool act);
    
  

private:
    SPI& _spi;
    DigitalOut _nCs;
    DigitalOut _nLDAC;
    
    char _upper_half;
    char _lower_half;
    
    char _make_upper_half(bool output, bool gain, bool shdn, int data);
    char _make_lower_half(int data);
    void _write(bool output, bool gain, bool shdn, int data);             

};

#endif