/*
  ______                              _
 / _____)             _              | |
( (____  _____ ____ _| |_ _____  ____| |__
 \____ \| ___ |    (_   _) ___ |/ ___)  _ \
 _____) ) ____| | | || |_| ____( (___| | | |
(______/|_____)_|_|_| \__)_____)\____)_| |_|
    (C)2016 Semtech

Description: Display demo menus and manage touch sensor.

Maintainer: Gregory Cristian & Gilbert Menth
*/

#include "mbed.h"
#include "Menu.h"
#include "DisplayDriver.h"
#include "Eeprom.h"
#include "DemoApplication.h"


/*!
 * \brief Maximum character count on the same line with function DrawText
 * This include needed space for the outline (at the begin and the end of line).
 * MAX_CHAR_PER_BTN is the same, but for a button.
*/
#define MAX_CHAR_PER_LINE       28


/*!
 * \brief Bitmap of Semtech Logo stored here (copied from a C file).
*/
extern uint8_t SemtechLogo[];

/*!
 * \brief Local copy of the Device which is fitted on the HW board
*/
uint8_t DeviceConnected = 0;

/*!
 * \brief Local copy of the Matching Frequency fitted on the HW
*/
uint8_t MatchingFreq = 0;

/*!
 * \brief This ticker give the rythme to check X and Y coordinates of activated
 * touched region (if pressed). CheckScreenFlag keep the status if Touchscreen
 * has been checked or not, until checked.
*/
Ticker CheckScreenTouch;
volatile bool CheckScreenFlag = false;

/*!
 * \brief This ticker give the rythme to avoid quick menu change
 */
Ticker DebouncedScreenTouch;
volatile bool ScreenBeenDebounced = false;

/*!
 * \brief List og graphical object in the application.
*/
enum MenuObjectsId
{
    PAGE_BORDER     = 0,
    TITLE_LINE,
    TITLE_TEXT,
    FIRM_LOGO,
    BTN0_BORDER,       // to accord with DrawActiveButton( ), each button should
    BTN0_MENU_TEXT,    // have consecutively MENU_TEXT (full large text),
    BTN0_LABEL_TEXT,   // LABEL_TEXT (left side of the button) and VALUE_TEXT
    BTN0_VALUE_TEXT,   // (right side of the button).
    BTN0b_BORDER,
    BTN0b_MENU_TEXT,
    BTN0b_LABEL_TEXT,
    BTN0b_VALUE_TEXT,
    BTN1_BORDER,
    BTN1_MENU_TEXT,
    BTN1_LABEL_TEXT,
    BTN1_VALUE_TEXT,
    BTN2_BORDER,
    BTN2_MENU_TEXT,
    BTN2_LABEL_TEXT,
    BTN2_VALUE_TEXT,
    BTN3_BORDER,
    BTN3_MENU_TEXT,
    BTN3_LABEL_TEXT,
    BTN3_VALUE_TEXT,
    BTN4_BORDER,
    BTN4_MENU_TEXT,
    BTN4_LABEL_TEXT,
    BTN4_VALUE_TEXT,
    BTN5_BORDER,
    BTN5_MENU_TEXT,
    BTN5_LABEL_TEXT,
    BTN5_VALUE_TEXT,
    BTN6_BORDER,
    BTN6_MENU_TEXT,
    BTN6_LABEL_TEXT,
    BTN6_VALUE_TEXT,
    BTN7_BORDER,
    BTN7_MENU_TEXT,
    BTN8_BORDER,
    BTN8_MENU_TEXT,
    BTN9_BORDER,
    BTN9_MENU_TEXT,
    BTN9_LABEL_TEXT,
    BTN9_VALUE_TEXT,
    LINE0_TEXT,
    LINE1_TEXT,
    LINE2_TEXT,
    LINE3_TEXT,
    LINE3_COL0_TEXT,
    LINE3_COL1_TEXT,
    LINE3_COL2_TEXT,
    LINE4_TEXT,         // to accord with DrawText( ), each text line should
    LINE4_COL0_TEXT,    // have consecutively TEXT (full large text),
    LINE4_COL1_TEXT,    // COL0_TEXT (left side of the line), COL1_TEXT (center
    LINE4_COL2_TEXT,    // region of the line) and COL2_TEXT (right side of the
    LINE5_COL0_TEXT,    // line). COL0..2 are optionals if doesn't exist.
    LINE5_COL1_TEXT,
    LINE5_COL2_TEXT,
    LINE6_COL0_TEXT,
    LINE6_COL1_TEXT,
    LINE6_COL2_TEXT,
    LINE7_COL0_TEXT,
    LINE7_COL1_TEXT,
    LINE7_COL2_TEXT,
    LINE8_COL0_TEXT,
    LINE8_COL1_TEXT,
    LINE8_COL2_TEXT,
    LINE9_COL0_TEXT,
    LINE9_COL1_TEXT,
    LINE9_COL2_TEXT,
};
#define GRAPH_OBJECTS_COUNT     70
GraphObject_t MenuObjects[GRAPH_OBJECTS_COUNT] =
{
    // { Id,           Type,      Xpos, Ypos, Height, Width, LineW, BackColor, FrontColor, DoFill, FillCol, Source, TouchActive }
    { PAGE_BORDER,     GO_RECTANGLE, 0,   0,   320,    240,   1, BACK_COLOR, PAGE_COLOR,    false, NULL, NULL,        false },
    { TITLE_LINE,      GO_LINE,      0,   30,  1,      240,   1, BACK_COLOR, PAGE_COLOR,    false, NULL, NULL,        false },
    { TITLE_TEXT,      GO_TEXT,      15,  9,   NULL,   208,   1, BACK_COLOR, WHITE,         false, NULL, NULL,        false },
    { FIRM_LOGO,       GO_IMAGE,     20,  37,  95,     215,   1, BACK_COLOR, WHITE,         false, NULL, SemtechLogo, false },
    { BTN0_BORDER,     GO_RECTANGLE, 10,  40,  30,     106,   1, BACK_COLOR, BUTTON_BORDER, false, NULL, NULL,        false },
    { BTN0_MENU_TEXT,  GO_TEXT,      15,  48,  NULL,   96,    1, BACK_COLOR, MENU_TEXT,     false, NULL, NULL,        false },
    { BTN0_LABEL_TEXT, GO_TEXT,      15,  48,  NULL,   32,    1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { BTN0_VALUE_TEXT, GO_TEXT,      47,  48,  NULL,   64,    1, BACK_COLOR, TEXT_VALUE,    false, NULL, NULL,        false },
    { BTN0b_BORDER,    GO_RECTANGLE, 10,  40,  30,     222,   1, BACK_COLOR, BUTTON_BORDER, false, NULL, NULL,        false },
    { BTN0b_MENU_TEXT, GO_TEXT,      15,  48,  NULL,   200,   1, BACK_COLOR, MENU_TEXT,     false, NULL, NULL,        false },
    { BTN0b_LABEL_TEXT,GO_TEXT,      15,  48,  NULL,   64,    1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { BTN0b_VALUE_TEXT,GO_TEXT,      79,  48,  NULL,   152,   1, BACK_COLOR, TEXT_VALUE,    false, NULL, NULL,        false },
    { BTN1_BORDER,     GO_RECTANGLE, 10,  75,  30,     222,   1, BACK_COLOR, BUTTON_BORDER, false, NULL, NULL,        false },
    { BTN1_MENU_TEXT,  GO_TEXT,      15,  83,  NULL,   200,   1, BACK_COLOR, MENU_TEXT,     false, NULL, NULL,        false },
    { BTN1_LABEL_TEXT, GO_TEXT,      15,  83,  NULL,   64,    1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { BTN1_VALUE_TEXT, GO_TEXT,      79,  83,  NULL,   152,   1, BACK_COLOR, TEXT_VALUE,    false, NULL, NULL,        false },
    { BTN2_BORDER,     GO_RECTANGLE, 10,  110, 30,     222,   1, BACK_COLOR, BUTTON_BORDER, false, NULL, NULL,        false },
    { BTN2_MENU_TEXT,  GO_TEXT,      15,  118, NULL,   200,   1, BACK_COLOR, MENU_TEXT,     false, NULL, NULL,        false },
    { BTN2_LABEL_TEXT, GO_TEXT,      15,  118, NULL,   64,    1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { BTN2_VALUE_TEXT, GO_TEXT,      79,  118, NULL,   152,   1, BACK_COLOR, TEXT_VALUE,    false, NULL, NULL,        false },
    { BTN3_BORDER,     GO_RECTANGLE, 10,  145, 30,     222,   1, BACK_COLOR, BUTTON_BORDER, false, NULL, NULL,        false },
    { BTN3_MENU_TEXT,  GO_TEXT,      15,  153, NULL,   200,   1, BACK_COLOR, MENU_TEXT,     false, NULL, NULL,        false },
    { BTN3_LABEL_TEXT, GO_TEXT,      15,  153, NULL,   64,    1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { BTN3_VALUE_TEXT, GO_TEXT,      79,  153, NULL,   152,   1, BACK_COLOR, TEXT_VALUE,    false, NULL, NULL,        false },
    { BTN4_BORDER,     GO_RECTANGLE, 10,  180, 30,     222,   1, BACK_COLOR, BUTTON_BORDER, false, NULL, NULL,        false },
    { BTN4_MENU_TEXT,  GO_TEXT,      15,  188, NULL,   200,   1, BACK_COLOR, MENU_TEXT,     false, NULL, NULL,        false },
    { BTN4_LABEL_TEXT, GO_TEXT,      15,  188, NULL,   64,    1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { BTN4_VALUE_TEXT, GO_TEXT,      79,  188, NULL,   152,   1, BACK_COLOR, TEXT_VALUE,    false, NULL, NULL,        false },
    { BTN5_BORDER,     GO_RECTANGLE, 10,  215, 30,     222,   1, BACK_COLOR, BUTTON_BORDER, false, NULL, NULL,        false },
    { BTN5_MENU_TEXT,  GO_TEXT,      15,  223, NULL,   200,   1, BACK_COLOR, MENU_TEXT,     false, NULL, NULL,        false },
    { BTN5_LABEL_TEXT, GO_TEXT,      15,  223, NULL,   64,    1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { BTN5_VALUE_TEXT, GO_TEXT,      79,  223, NULL,   152,   1, BACK_COLOR, TEXT_VALUE,    false, NULL, NULL,        false },
    { BTN6_BORDER,     GO_RECTANGLE, 10,  250, 30,     222,   1, BACK_COLOR, BUTTON_BORDER, false, NULL, NULL,        false },
    { BTN6_MENU_TEXT,  GO_TEXT,      15,  258, NULL,   200,   1, BACK_COLOR, MENU_TEXT,     false, NULL, NULL,        false },
    { BTN6_LABEL_TEXT, GO_TEXT,      15,  258, NULL,   64,    1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { BTN6_VALUE_TEXT, GO_TEXT,      79,  258, NULL,   152,   1, BACK_COLOR, TEXT_VALUE,    false, NULL, NULL,        false },
    { BTN7_BORDER,     GO_RECTANGLE, 10,  285, 30,     106,   1, BACK_COLOR, BUTTON_BORDER, false, NULL, NULL,        false },
    { BTN7_MENU_TEXT,  GO_TEXT,      15,  293, NULL,   96,    1, BACK_COLOR, MENU_TEXT,     false, NULL, NULL,        false },
    { BTN8_BORDER,     GO_RECTANGLE, 125, 285, 30,     107,   1, BACK_COLOR, BUTTON_BORDER, false, NULL, NULL,        false },
    { BTN8_MENU_TEXT,  GO_TEXT,      130, 293, NULL,   96,    1, BACK_COLOR, MENU_TEXT,     false, NULL, NULL,        false },
    { BTN9_BORDER,     GO_RECTANGLE, 125, 40,  30,     107,   1, BACK_COLOR, BUTTON_BORDER, false, NULL, NULL,        false },
    { BTN9_MENU_TEXT,  GO_TEXT,      130, 48,  NULL,   96,    1, BACK_COLOR, MENU_TEXT,     false, NULL, NULL,        false },
    { BTN9_LABEL_TEXT, GO_TEXT,      130, 48,  NULL,   32,    1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { BTN9_VALUE_TEXT, GO_TEXT,      162, 48,  NULL,   64,    1, BACK_COLOR, TEXT_VALUE,    false, NULL, NULL,        false },
    { LINE0_TEXT,      GO_TEXT,      5,   80,  NULL,   232,   1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { LINE1_TEXT,      GO_TEXT,      5,   100, NULL,   232,   1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { LINE2_TEXT,      GO_TEXT,      5,   120, NULL,   232,   1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { LINE3_TEXT,      GO_TEXT,      5,   140, NULL,   232,   1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { LINE3_COL0_TEXT, GO_TEXT,      5,   140, NULL,   104,   1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { LINE3_COL1_TEXT, GO_TEXT,      110, 140, NULL,   56,    1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { LINE3_COL2_TEXT, GO_TEXT,      170, 140, NULL,   64,    1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { LINE4_TEXT,      GO_TEXT,      5,   160, NULL,   232,   1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { LINE4_COL0_TEXT, GO_TEXT,      5,   160, NULL,   104,   1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { LINE4_COL1_TEXT, GO_TEXT,      110, 160, NULL,   56,    1, BACK_COLOR, TEXT_VALUE,    false, NULL, NULL,        false },
    { LINE4_COL2_TEXT, GO_TEXT,      170, 160, NULL,   64,    1, BACK_COLOR, TEXT_VALUE,    false, NULL, NULL,        false },
    { LINE5_COL0_TEXT, GO_TEXT,      5,   180, NULL,   104,   1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { LINE5_COL1_TEXT, GO_TEXT,      110, 180, NULL,   56,    1, BACK_COLOR, TEXT_VALUE,    false, NULL, NULL,        false },
    { LINE5_COL2_TEXT, GO_TEXT,      170, 180, NULL,   64,    1, BACK_COLOR, TEXT_VALUE,    false, NULL, NULL,        false },
    { LINE6_COL0_TEXT, GO_TEXT,      5,   200, NULL,   104,   1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { LINE6_COL1_TEXT, GO_TEXT,      110, 200, NULL,   56,    1, BACK_COLOR, TEXT_VALUE,    false, NULL, NULL,        false },
    { LINE6_COL2_TEXT, GO_TEXT,      170, 200, NULL,   64,    1, BACK_COLOR, TEXT_VALUE,    false, NULL, NULL,        false },
    { LINE7_COL0_TEXT, GO_TEXT,      5,   220, NULL,   104,   1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { LINE7_COL1_TEXT, GO_TEXT,      110, 220, NULL,   56,    1, BACK_COLOR, TEXT_VALUE,    false, NULL, NULL,        false },
    { LINE7_COL2_TEXT, GO_TEXT,      170, 220, NULL,   64,    1, BACK_COLOR, TEXT_VALUE,    false, NULL, NULL,        false },
    { LINE8_COL0_TEXT, GO_TEXT,      5,   240, NULL,   104,   1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { LINE8_COL1_TEXT, GO_TEXT,      110, 240, NULL,   56,    1, BACK_COLOR, TEXT_VALUE,    false, NULL, NULL,        false },
    { LINE8_COL2_TEXT, GO_TEXT,      170, 240, NULL,   64,    1, BACK_COLOR, TEXT_VALUE,    false, NULL, NULL,        false },
    { LINE9_COL0_TEXT, GO_TEXT,      5,   260, NULL,   104,   1, BACK_COLOR, TEXT_COLOR,    false, NULL, NULL,        false },
    { LINE9_COL1_TEXT, GO_TEXT,      110, 260, NULL,   56,    1, BACK_COLOR, TEXT_VALUE,    false, NULL, NULL,        false },
    { LINE9_COL2_TEXT, GO_TEXT,      170, 260, NULL,   64,    1, BACK_COLOR, TEXT_VALUE,    false, NULL, NULL,        false }
};

/*!
 * \brief DrawText( ) need char* to display a text. It can display until
 * 3 different texts on the same line. To avoid recursive use of the same
 * ressource, 3 temporary string are declared.
 */
char StringText[MAX_CHAR_PER_LINE + 1];  // don't forget the /0 (end of string)
char StringText2[MAX_CHAR_PER_LINE + 1];
char StringText3[MAX_CHAR_PER_LINE + 1];

/*!
 * \brief CurrentPage store current displayed page (cf.: list of availlable
 * pages in Menu.h). PeviousPage is used only when navigate through "Radio
 * Config" to be able to restore the page before "Radio Congig", witch can be
 * PingPong, PER or START_PAGE. The goal is to speed menu navigation.
*/
static uint8_t CurrentPage  = START_PAGE;
static uint8_t PreviousPage = START_PAGE;

/*!
 * \brief In "Radio Config Freq", we can update the central frequency. To avoid 
 * keyboard, we use (+) et (-) keys. CurrentFreqBase is the offset to be applied
 * when we increase or decrease the frequency.
*/
static FreqBase CurrentFreqBase = FB100K;

/*!
 * \brief If radio params does't respect bandwidth > fdev + bitrate/2, the radio
 * couldn't make a proper modulation. Parameters will be printed in red.
*/
static bool ErrorParam = false;


void CheckScreen( void );
void DebouncedScreen( void );
void DrawActiveButton( uint8_t buttonId, uint8_t* text1, uint8_t* text2 );
void ButtonChangeText( uint8_t buttonId, uint8_t* text1, uint8_t* text2 );
void DrawText( uint8_t lineId, uint8_t* text0, uint8_t* text1, uint8_t* text2 );
void RadioSettingsCheckError( void );
void RadioSettingsClearError( void );


void MenuInit( uint8_t deviceConnected, uint8_t matchingFreq )
{
    DeviceConnected = deviceConnected;
    MatchingFreq = matchingFreq;
    DisplayDriverInit( );
    GraphObjectDraw( &( MenuObjects[PAGE_BORDER] ), NULL, false, false);
    GraphObjectDraw( &( MenuObjects[TITLE_LINE] ), NULL, false, false);
    MenuSetPage( START_PAGE );
    CheckScreenTouch.attach_us( &CheckScreen, 100000 );     // every 100 ms
}

uint8_t MenuHandler( bool refresh )
{
    uint8_t graphObjectTouched = 0xFF;

    if( CheckScreenFlag == true )
    {
        if( ScreenBeenDebounced == false )
        {
            ScreenBeenDebounced = true;
            CheckScreenFlag = false;
            DebouncedScreenTouch.attach_us( &DebouncedScreen, 150000 ); // every 150 ms
            GraphObjectTouched( MenuObjects, GRAPH_OBJECTS_COUNT, &graphObjectTouched );
        }
    }

    if( ( refresh == true ) && ( graphObjectTouched == 0xFF ) )
    {
        MenuSetPage( CurrentPage );
    }
    else if( ( graphObjectTouched != 0xFF ) || ( refresh == true ) )
    {
        switch( CurrentPage )
        {
            case START_PAGE:
                if( graphObjectTouched == BTN2_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    EepromLoadSettings( ( RadioPacketTypes_t )Eeprom.EepromData.DemoSettings.ModulationType );
                    MenuSetPage( PAGE_PING_PONG );
                }
                else if( graphObjectTouched == BTN3_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    EepromLoadSettings( ( RadioPacketTypes_t )Eeprom.EepromData.DemoSettings.ModulationType );
                    MenuSetPage( PAGE_PER );
                }
                else if( graphObjectTouched == BTN4_BORDER )
                {
                    EepromLoadSettings( ( RadioPacketTypes_t )Eeprom.EepromData.DemoSettings.ModulationType );
                    MenuSetPage( PAGE_RADIO_TEST );
                }    
                else if( graphObjectTouched == BTN5_BORDER )
                {
                    MenuSetPage( PAGE_RADIO_PARAM );
                }
                else if( graphObjectTouched == BTN6_BORDER )
                {
                    MenuSetPage( PAGE_UTILITIES );
                }
                break;

            case PAGE_PING_PONG:
                if( graphObjectTouched == BTN0_BORDER )
                {
                    ButtonChangeText( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                    if( Eeprom.EepromData.DemoSettings.Entity == MASTER )
                    {
                        Eeprom.EepromData.DemoSettings.Entity = SLAVE;
                        ButtonChangeText( BTN0_BORDER, ( uint8_t* )"   SLAVE", NULL );
                        DrawText( LINE3_COL0_TEXT, NULL, NULL, ( uint8_t* )" " );
                        DrawText( LINE4_COL0_TEXT, NULL, NULL, ( uint8_t* )" " );
                        DrawText( LINE5_COL0_TEXT, NULL, NULL, ( uint8_t* )" " );
                        DrawText( LINE6_COL0_TEXT, NULL, NULL, ( uint8_t* )" " );
                        DrawText( LINE7_COL0_TEXT, NULL, NULL, ( uint8_t* )" " );
                        DrawText( LINE8_COL0_TEXT, NULL, NULL, ( uint8_t* )" " );
                        DrawText( LINE9_COL0_TEXT, NULL, NULL, ( uint8_t* )" " );
                    }
                    else
                    {
                        Eeprom.EepromData.DemoSettings.Entity = MASTER;
                        ButtonChangeText( BTN0_BORDER, ( uint8_t* )"   MASTER", NULL );
                        DrawText( LINE3_COL0_TEXT, NULL, NULL, ( uint8_t* )"MASTER" );
                    }
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    EepromSaveSettings( DEMO_SETTINGS );
                    MenuSetPage( PAGE_PING_PONG );
                }
                else if( graphObjectTouched == BTN7_BORDER )
                {
                    StopDemoApplication( );
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    PreviousPage = START_PAGE; // clear Previous page
                    MenuSetPage( START_PAGE );
                }
                else if( graphObjectTouched == BTN8_BORDER )
                {
                    if( Eeprom.EepromData.DemoSettings.HoldDemo == true )
                    {
                        Eeprom.EepromData.DemoSettings.HoldDemo = false;
                        ButtonChangeText( BTN8_BORDER, ( uint8_t* )"    STOP", NULL );
                        Eeprom.EepromData.DemoSettings.CntPacketTx          = 0;
                        Eeprom.EepromData.DemoSettings.CntPacketRxOK        = 0;
                        Eeprom.EepromData.DemoSettings.CntPacketRxOKSlave   = 0;
                        Eeprom.EepromData.DemoSettings.CntPacketRxKO        = 0;
                        Eeprom.EepromData.DemoSettings.CntPacketRxKOSlave   = 0;
                        Eeprom.EepromData.DemoSettings.RxTimeOutCount       = 0;
                        StopDemoApplication( );
                        RunDemoApplicationPingPong( );
                    }
                    else
                    {
                        Eeprom.EepromData.DemoSettings.HoldDemo = true;
                        ButtonChangeText( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                        StopDemoApplication( );
                        wait_ms( 200 ); // debouncing of the touch screen
                    }
                }
                else if( graphObjectTouched == BTN9_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    StopDemoApplication( );
                    // "CANCEL" or "OK & SAVE" button of PAGE_RADIO_PARAM use
                    // PreviousPage to set next page. Speed up the navigation.
                    PreviousPage = PAGE_PING_PONG;
                    MenuSetPage( PAGE_RADIO_PARAM );
                }
                else if( refresh == true )
                {
                    MenuSetPage( PAGE_PING_PONG );
                }
                break;

            case PAGE_PER:
                if( graphObjectTouched == BTN0_BORDER )
                {
                    ButtonChangeText( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                    if( Eeprom.EepromData.DemoSettings.Entity == MASTER )
                    {
                        Eeprom.EepromData.DemoSettings.Entity = SLAVE;
                        ButtonChangeText( BTN0_BORDER, ( uint8_t* )"   SLAVE", NULL );
                        DrawText( LINE4_COL0_TEXT, ( uint8_t* )"Rx OK", NULL, NULL );
                        DrawText( LINE5_COL0_TEXT, ( uint8_t* )"Rx KO", NULL, NULL );
                        DrawText( LINE6_COL0_TEXT, ( uint8_t* )"PER",   NULL, NULL );
                        DrawText( LINE7_COL0_TEXT, ( uint8_t* )"Freq Err", NULL, NULL );
                        DrawText( LINE8_COL0_TEXT, ( uint8_t* )"Last Rssi", NULL, NULL );
                        DrawText( LINE9_COL0_TEXT, ( uint8_t* )"Last SNR", NULL, NULL );
                    }
                    else
                    {
                        Eeprom.EepromData.DemoSettings.Entity = MASTER;
                        ButtonChangeText( BTN0_BORDER, ( uint8_t* )"   MASTER", NULL );
                        DrawText( LINE4_COL0_TEXT, ( uint8_t* )"Transmitted", ( uint8_t* )GetMenuDemoNumSentPacket( ), NULL );
                        DrawText( LINE5_COL0_TEXT, ( uint8_t* )" ", ( uint8_t* )" ", NULL );
                        DrawText( LINE6_COL0_TEXT, ( uint8_t* )" ", ( uint8_t* )" ", NULL );
                        DrawText( LINE7_COL0_TEXT, ( uint8_t* )" ", ( uint8_t* )" ", NULL );
                        DrawText( LINE8_COL0_TEXT, ( uint8_t* )" ", ( uint8_t* )" ", NULL );
                        DrawText( LINE9_COL0_TEXT, ( uint8_t* )" ", ( uint8_t* )" ", NULL );
                    }

                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    EepromSaveSettings( DEMO_SETTINGS );
                    MenuSetPage( PAGE_PER );
                }
                else if( graphObjectTouched == BTN7_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    PreviousPage = START_PAGE; // clear Previous page
                    MenuSetPage( START_PAGE );
                }
                else if( graphObjectTouched == BTN8_BORDER )
                {
                    if( Eeprom.EepromData.DemoSettings.HoldDemo == true )
                    {
                        Eeprom.EepromData.DemoSettings.HoldDemo = false;
                        ButtonChangeText( BTN8_BORDER, ( uint8_t* )"    STOP", NULL );
                        StopDemoApplication( );
                        Eeprom.EepromData.DemoSettings.CntPacketTx          = 0;
                        Eeprom.EepromData.DemoSettings.CntPacketRxOK        = 0;
                        Eeprom.EepromData.DemoSettings.CntPacketRxKO        = 0;
                        Eeprom.EepromData.DemoSettings.RxTimeOutCount       = 0;
                        RunDemoApplicationPer( );
                    }
                    else
                    {
                        Eeprom.EepromData.DemoSettings.HoldDemo = true;
                        ButtonChangeText( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                        StopDemoApplication( );
                        wait_ms( 200 ); // debouncing of the touch screen
                    }
                }
                else if( graphObjectTouched == BTN9_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    StopDemoApplication( );
                    // "CANCEL" or "OK & SAVE" button of PAGE_RADIO_PARAM use
                    // PreviousPage to set next page. Speed up the navigation.
                    PreviousPage = PAGE_PER;
                    MenuSetPage( PAGE_RADIO_PARAM );
                }
                else if( refresh == true )
                {
                    MenuSetPage( PAGE_PER );
                }
                break;

            case PAGE_RADIO_PARAM:
                if( graphObjectTouched == BTN0_BORDER )
                {
                    if( Eeprom.EepromData.DemoSettings.ModulationType == PACKET_TYPE_LORA )
                    {
                        Eeprom.EepromData.DemoSettings.ModulationType = PACKET_TYPE_GFSK;
                    }
                    else// if( Eeprom.EepromData.DemoSettings.ModulationType == PACKET_TYPE_GFSK )
                    {
                        Eeprom.EepromData.DemoSettings.ModulationType = PACKET_TYPE_LORA;
                    }
                    EepromLoadSettings( ( RadioPacketTypes_t )Eeprom.EepromData.DemoSettings.ModulationType );
                    RadioSettingsCheckError( );
                    ButtonChangeText( BTN0_BORDER, NULL, ( uint8_t* )GetMenuRadioFrameType( ) );
                    ButtonChangeText( BTN3_BORDER, NULL, ( uint8_t* )GetRadioModulationParameters1( ) );
                    ButtonChangeText( BTN4_BORDER, NULL, ( uint8_t* )GetRadioModulationParameters2( ) );
                    ButtonChangeText( BTN5_BORDER, NULL, ( uint8_t* )GetRadioModulationParameters3( ) );
                    ButtonChangeText( BTN6_BORDER, NULL, ( uint8_t* )GetRadioModulationParameters4( ) );
                    ButtonChangeText( BTN9_BORDER, NULL, ( uint8_t* )GetRadioPayloadLength( ) );
                    RadioSettingsClearError( );
                    //MenuSetPage( PAGE_RADIO_PARAM ); // _GMH_ still need ?
                }
                else if( graphObjectTouched == BTN1_BORDER )
                {
                    MenuSetPage( PAGE_RADIO_PARAM_FREQ );
                }
                else if( graphObjectTouched == BTN2_BORDER )
                {
                    if( DeviceConnected == SX1261 )
                    {
                        Eeprom.EepromData.DemoSettings.TxPower += 1;
                        if( Eeprom.EepromData.DemoSettings.TxPower > SX1261_POWER_TX_MAX )
                        {
                            Eeprom.EepromData.DemoSettings.TxPower = SX1261_POWER_TX_MIN;
                        }
                    }
                    else
                    {
                        Eeprom.EepromData.DemoSettings.TxPower += 1;
                        if( Eeprom.EepromData.DemoSettings.TxPower > SX1262_POWER_TX_MAX )
                        {
                            Eeprom.EepromData.DemoSettings.TxPower = SX1262_POWER_TX_MIN;
                        }
                    }
                    ButtonChangeText( BTN2_BORDER, NULL, ( uint8_t* )GetRadioTxPower( ) );
                }
                else if( graphObjectTouched == BTN3_BORDER )
                {
                    if( Eeprom.EepromData.DemoSettings.ModulationType == PACKET_TYPE_LORA )
                    {
                        if( Eeprom.EepromData.DemoSettings.ModulationParam1 < LORA_SF12 )
                        {
                            Eeprom.EepromData.DemoSettings.ModulationParam1++;
                        }
                        else
                        {
                            Eeprom.EepromData.DemoSettings.ModulationParam1 = LORA_SF5;
                        }
                    }
                    else // PACKET_TYPE_GFSK
                    {
                        switch( Eeprom.EepromData.DemoSettings.ModulationParam1 )
                        {
                            case DEMO_BR_100:
                                Eeprom.EepromData.DemoSettings.ModulationParam1 = DEMO_BR_600;
                                break;

                            case DEMO_BR_600:
                                Eeprom.EepromData.DemoSettings.ModulationParam1 = DEMO_BR_4800;
                                break;
                            default:
                            case DEMO_BR_4800:
                                Eeprom.EepromData.DemoSettings.ModulationParam1 = DEMO_BR_9600;
                                break;

                            case DEMO_BR_9600:
                                Eeprom.EepromData.DemoSettings.ModulationParam1 = DEMO_BR_19200;
                                break;

                            case DEMO_BR_19200:
                                Eeprom.EepromData.DemoSettings.ModulationParam1 = DEMO_BR_57600;
                                break;

                            case DEMO_BR_57600:
                                Eeprom.EepromData.DemoSettings.ModulationParam1 = DEMO_BR_100000;
                                break;

                            case DEMO_BR_100000:
                                Eeprom.EepromData.DemoSettings.ModulationParam1 = DEMO_BR_250000;
                                break;

                            case DEMO_BR_250000:
                                Eeprom.EepromData.DemoSettings.ModulationParam1 = DEMO_BR_100;
                                break;
                        }
                        RadioSettingsCheckError( );
                    }
                    ButtonChangeText( BTN3_BORDER, NULL, ( uint8_t* )GetRadioModulationParameters1( ) );
                    RadioSettingsClearError( );
                }
                else if( graphObjectTouched == BTN4_BORDER )
                {
                    if( Eeprom.EepromData.DemoSettings.ModulationType == PACKET_TYPE_LORA )
                    {
                        switch( Eeprom.EepromData.DemoSettings.ModulationParam2 )
                        {
                            case LORA_BW_007:
                                Eeprom.EepromData.DemoSettings.ModulationParam2 = LORA_BW_010;
                                break;
                            case LORA_BW_010:
                                Eeprom.EepromData.DemoSettings.ModulationParam2 = LORA_BW_015;
                                break;
                            case LORA_BW_015:
                                Eeprom.EepromData.DemoSettings.ModulationParam2 = LORA_BW_020;
                                break;
                            case LORA_BW_020:
                                Eeprom.EepromData.DemoSettings.ModulationParam2 = LORA_BW_031;
                                break;
                            case LORA_BW_031:
                                Eeprom.EepromData.DemoSettings.ModulationParam2 = LORA_BW_041;
                                break;
                            case LORA_BW_041:
                                Eeprom.EepromData.DemoSettings.ModulationParam2 = LORA_BW_062;
                                break;
                            
                            default:
                            case LORA_BW_062:
                                Eeprom.EepromData.DemoSettings.ModulationParam2 = LORA_BW_125;
                                break;
                            case LORA_BW_125:
                                Eeprom.EepromData.DemoSettings.ModulationParam2 = LORA_BW_250;
                                break;
                            case LORA_BW_250:
                                Eeprom.EepromData.DemoSettings.ModulationParam2 = LORA_BW_500;
                                break;
                            case LORA_BW_500:
                                Eeprom.EepromData.DemoSettings.ModulationParam2 = LORA_BW_007;
                                break;
                        }
                    }
                    else //Eeprom.EepromData.DemoSettings.ModulationType == PACKET_TYPE_GFSK
                    {
                        switch( Eeprom.EepromData.DemoSettings.ModulationParam2 )
                        {
                            default:
                            case DEMO_FDEV_5000:
                                Eeprom.EepromData.DemoSettings.ModulationParam2 = DEMO_FDEV_10000;
                                break;
                            
                            case DEMO_FDEV_10000:
                                Eeprom.EepromData.DemoSettings.ModulationParam2 = DEMO_FDEV_25000;
                                break;

                            case DEMO_FDEV_25000:
                                Eeprom.EepromData.DemoSettings.ModulationParam2 = DEMO_FDEV_50000;
                                break;

                            case DEMO_FDEV_50000:
                                Eeprom.EepromData.DemoSettings.ModulationParam2 = DEMO_FDEV_75000;
                                break;

                            case DEMO_FDEV_75000:
                                Eeprom.EepromData.DemoSettings.ModulationParam2 = DEMO_FDEV_100000;
                                break;

                            case DEMO_FDEV_100000:
                                Eeprom.EepromData.DemoSettings.ModulationParam2 = DEMO_FDEV_150000;
                                break;

                            case DEMO_FDEV_150000:
                                Eeprom.EepromData.DemoSettings.ModulationParam2 = DEMO_FDEV_5000;
                                break;
                        }
                        RadioSettingsCheckError( );
                    }
                    ButtonChangeText( BTN4_BORDER, NULL, ( uint8_t* )GetRadioModulationParameters2( ) );
                    RadioSettingsClearError( );
                }    
                else if( graphObjectTouched == BTN5_BORDER )
                {
                    if( Eeprom.EepromData.DemoSettings.ModulationType == PACKET_TYPE_LORA )
                    {
                        if( Eeprom.EepromData.DemoSettings.ModulationParam3 < LORA_CR_4_8 )
                        {
                            Eeprom.EepromData.DemoSettings.ModulationParam3++;
                        }
                        else
                        {
                            Eeprom.EepromData.DemoSettings.ModulationParam3 = LORA_CR_4_5;
                        }
                    }
                    else // PACKET_TYPE_GFSK
                    {
                        switch( Eeprom.EepromData.DemoSettings.ModulationParam3 )
                        {
                            case MOD_SHAPING_OFF :
                                Eeprom.EepromData.DemoSettings.ModulationParam3 = MOD_SHAPING_G_BT_03;
                                break;
                            case MOD_SHAPING_G_BT_03 :
                                Eeprom.EepromData.DemoSettings.ModulationParam3 = MOD_SHAPING_G_BT_05;
                                break;
                            case MOD_SHAPING_G_BT_05 :
                                Eeprom.EepromData.DemoSettings.ModulationParam3 = MOD_SHAPING_G_BT_07;
                                break;
                            case MOD_SHAPING_G_BT_07 :
                                Eeprom.EepromData.DemoSettings.ModulationParam3 = MOD_SHAPING_G_BT_1;
                                break;
                            case MOD_SHAPING_G_BT_1 :
                                Eeprom.EepromData.DemoSettings.ModulationParam3 = MOD_SHAPING_OFF;
                                break;
                        }
                    }
                    ButtonChangeText( BTN5_BORDER, NULL, ( uint8_t* )GetRadioModulationParameters3( ) );
                }
                else if( graphObjectTouched == BTN6_BORDER )
                {
                    if( Eeprom.EepromData.DemoSettings.ModulationType == PACKET_TYPE_LORA )
                    {
                        if( Eeprom.EepromData.DemoSettings.PacketParam2 == LORA_PACKET_FIXED_LENGTH )
                        {
                            Eeprom.EepromData.DemoSettings.PacketParam2 = LORA_PACKET_VARIABLE_LENGTH;
                        }
                        else
                        {
                            Eeprom.EepromData.DemoSettings.PacketParam2 = LORA_PACKET_FIXED_LENGTH;
                        }
                    }
                    else // Eeprom.EepromData.DemoSettings.ModulationType == PACKET_TYPE_GFSK
                    {
                        switch( Eeprom.EepromData.DemoSettings.ModulationParam4 )
                        {
                            case RX_BW_4800:
                                Eeprom.EepromData.DemoSettings.ModulationParam4 = RX_BW_5800;
                                break;

                            case RX_BW_5800:
                                Eeprom.EepromData.DemoSettings.ModulationParam4 = RX_BW_7300;
                                break;

                            case RX_BW_7300:
                                Eeprom.EepromData.DemoSettings.ModulationParam4 = RX_BW_9700;
                                break;

                            case RX_BW_9700:
                                Eeprom.EepromData.DemoSettings.ModulationParam4 = RX_BW_11700;
                                break;

                            case RX_BW_11700:
                                Eeprom.EepromData.DemoSettings.ModulationParam4 = RX_BW_14600;
                                break;

                            case RX_BW_14600:
                                Eeprom.EepromData.DemoSettings.ModulationParam4 = RX_BW_19500;
                                break;

                            case RX_BW_19500:
                                Eeprom.EepromData.DemoSettings.ModulationParam4 = RX_BW_23400;
                                break;

                            case RX_BW_23400:
                                Eeprom.EepromData.DemoSettings.ModulationParam4 = RX_BW_29300;
                                break;

                            case RX_BW_29300:
                                Eeprom.EepromData.DemoSettings.ModulationParam4 = RX_BW_39000;
                                break;

                            case RX_BW_39000:
                                Eeprom.EepromData.DemoSettings.ModulationParam4 = RX_BW_58600;
                                break;

                            case RX_BW_58600:
                                Eeprom.EepromData.DemoSettings.ModulationParam4 = RX_BW_93800;
                                break;

                            case RX_BW_93800:
                                Eeprom.EepromData.DemoSettings.ModulationParam4 = RX_BW_117300;
                                break;

                            case RX_BW_117300:
                                Eeprom.EepromData.DemoSettings.ModulationParam4 = RX_BW_156200;
                                break;

                            case RX_BW_156200:
                                Eeprom.EepromData.DemoSettings.ModulationParam4 = RX_BW_187200;
                                break;

                            case RX_BW_187200:
                                Eeprom.EepromData.DemoSettings.ModulationParam4 = RX_BW_234300;
                                break;

                            case RX_BW_234300:
                                Eeprom.EepromData.DemoSettings.ModulationParam4 = RX_BW_312000;
                                break;

                            case RX_BW_312000:
                                Eeprom.EepromData.DemoSettings.ModulationParam4 = RX_BW_373600;
                                break;

                            case RX_BW_373600:
                                Eeprom.EepromData.DemoSettings.ModulationParam4 = RX_BW_467000;
                                break;

                            case RX_BW_467000:
                                Eeprom.EepromData.DemoSettings.ModulationParam4 = RX_BW_4800;
                                break;
                        }
                        RadioSettingsCheckError( );
                    }
                    ButtonChangeText( BTN6_BORDER, NULL, ( uint8_t* )GetRadioModulationParameters4( ) );
                    RadioSettingsClearError( );
                }
                else if( graphObjectTouched == BTN7_BORDER )
                {
                    uint8_t modTmp = Eeprom.EepromData.DemoSettings.ModulationType;

                    EepromMcuReadBuffer( 0, Eeprom.Buffer, EEPROM_BUFFER_SIZE );
                    EepromLoadGeneralSettings( );
                    EepromLoadSettings( ( RadioPacketTypes_t )modTmp );
                    MenuSetPage( PreviousPage );
                }
                else if( graphObjectTouched == BTN8_BORDER )
                {
                    switch( Eeprom.EepromData.DemoSettings.ModulationType )
                    {
                        case PACKET_TYPE_LORA:
                            EepromSaveSettings( RADIO_LORA_PARAMS );
                            break;

                        case PACKET_TYPE_GFSK:
                            EepromSaveSettings( RADIO_GFSK_PARAMS );
                            break;

                        default:

                            break;
                    }
                    EepromSaveSettings( DEMO_SETTINGS );
                    MenuSetPage( PreviousPage );
                }
                else if( graphObjectTouched == BTN9_BORDER )
                {
                    if( Eeprom.EepromData.DemoSettings.ModulationType == PACKET_TYPE_LORA )
                    {
                        if( Eeprom.EepromData.DemoSettings.PacketParam3 < 255 )
                        {
                            Eeprom.EepromData.DemoSettings.PacketParam3 += 1;
                        }
                        else
                        {
                            Eeprom.EepromData.DemoSettings.PacketParam3 = 1;
                        }
                    }
                    else // PACKET_TYPE_GFSK
                    {
                        if( Eeprom.EepromData.DemoSettings.PacketParam6 < 255 )
                        {
                            Eeprom.EepromData.DemoSettings.PacketParam6 += 1;
                        }
                        else
                        {
                            Eeprom.EepromData.DemoSettings.PacketParam6 = 1;
                        }
                    }
                    ButtonChangeText( BTN9_BORDER, NULL, ( uint8_t* )GetRadioPayloadLength( ) );
                }
                break;

            case PAGE_RADIO_PARAM_FREQ:
                if( graphObjectTouched == BTN0_BORDER )
                {
                    MenuSetPage( PAGE_RADIO_PARAM );
                }
                else if( graphObjectTouched == BTN3_BORDER )
                {
                    if( DeviceConnected == SX1268 )
                    {
                        Eeprom.EepromData.DemoSettings.Frequency = DEMO_CENTRAL_FREQ_PRESET4;
                    }
                    else
                    {
                        Eeprom.EepromData.DemoSettings.Frequency = DEMO_CENTRAL_FREQ_PRESET6;
                    }
                    MenuSetPage( PAGE_RADIO_PARAM );
                }
                else if( graphObjectTouched == BTN4_BORDER )
                {
                    if( DeviceConnected == SX1268 )
                    {
                        Eeprom.EepromData.DemoSettings.Frequency = DEMO_CENTRAL_FREQ_PRESET3;
                    }
                    else
                    {
                        Eeprom.EepromData.DemoSettings.Frequency = DEMO_CENTRAL_FREQ_PRESET7;
                    }
                    MenuSetPage( PAGE_RADIO_PARAM );
                }
                else if( graphObjectTouched == BTN5_BORDER )
                {
                    if( DeviceConnected == SX1268 )
                    {
                        Eeprom.EepromData.DemoSettings.Frequency = DEMO_CENTRAL_FREQ_PRESET9;
                    }
                    else
                    {
                        Eeprom.EepromData.DemoSettings.Frequency = DEMO_CENTRAL_FREQ_PRESET8;
                    }
                    MenuSetPage( PAGE_RADIO_PARAM );
                }
                else if( graphObjectTouched == BTN6_BORDER )
                {
                    switch( CurrentFreqBase )
                    {
                        case FB1:
                            CurrentFreqBase = FB10;
                            break;
                        case FB10:
                            CurrentFreqBase = FB100;
                            break;
                        case FB100:
                            CurrentFreqBase = FB1K;
                            break;
                        case FB1K:
                            CurrentFreqBase = FB10K;
                            break;
                        case FB10K:
                            CurrentFreqBase = FB100K;
                            break;
                        case FB100K:
                            CurrentFreqBase = FB1M;
                            break;
                        case FB1M:
                            CurrentFreqBase = FB10M;
                            break;
                        case FB10M:
                            CurrentFreqBase = FB1;
                            break;
                    }
                    ButtonChangeText( BTN6_BORDER, NULL, ( uint8_t* )GetRadioFreqBase( ) );
                }
                else if( graphObjectTouched == BTN7_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.Frequency -= CurrentFreqBase;
                    if( Eeprom.EepromData.DemoSettings.Frequency < DEMO_CENTRAL_FREQ_MIN )
                    {
                        Eeprom.EepromData.DemoSettings.Frequency = DEMO_CENTRAL_FREQ_MIN;
                    }
                    ButtonChangeText( BTN2_BORDER, NULL, ( uint8_t* )GetRadioFrequency( ) );
                }
                else if( graphObjectTouched == BTN8_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.Frequency += CurrentFreqBase;
                    if( Eeprom.EepromData.DemoSettings.Frequency > DEMO_CENTRAL_FREQ_MAX )
                    {
                        Eeprom.EepromData.DemoSettings.Frequency = DEMO_CENTRAL_FREQ_MAX;
                    }
                    ButtonChangeText( BTN2_BORDER, NULL, ( uint8_t* )GetRadioFrequency( ) );
                }
                break;

            case PAGE_UTILITIES:
                if( graphObjectTouched == BTN0_BORDER )
                {
                    EepromEraseCheckSum( );
                    NVIC_SystemReset( );
                }
                else if( graphObjectTouched == BTN3_BORDER )
                {
                    if( Eeprom.EepromData.DemoSettings.BoostedRx == true )
                    {
                        Eeprom.EepromData.DemoSettings.BoostedRx = false;
                    }
                    else
                    {
                        Eeprom.EepromData.DemoSettings.BoostedRx = true;
                    }
                    ButtonChangeText( BTN3_BORDER, NULL, ( uint8_t* )GetMenuDemoRadioBoostedRx( ) );
                }
                else if( graphObjectTouched == BTN4_BORDER )
                {
                    if( Eeprom.EepromData.DemoSettings.RadioPowerMode == USE_LDO )
                    {
                        Eeprom.EepromData.DemoSettings.RadioPowerMode = USE_DCDC;
                    }
                    else
                    {
                        Eeprom.EepromData.DemoSettings.RadioPowerMode = USE_LDO;
                    }
                    ButtonChangeText( BTN4_BORDER, NULL, ( uint8_t* )GetMenuDemoRadioPowerMode( ) );
                }
                else if( graphObjectTouched == BTN5_BORDER )
                {
                    switch( Eeprom.EepromData.DemoSettings.MaxNumPacket )
                    {
                        case 0:
                            Eeprom.EepromData.DemoSettings.MaxNumPacket = 100;
                            break;
                        case 100:
                            Eeprom.EepromData.DemoSettings.MaxNumPacket = 200;
                            break;
                        case 200:
                            Eeprom.EepromData.DemoSettings.MaxNumPacket = 500;
                            break;
                        case 500:
                            Eeprom.EepromData.DemoSettings.MaxNumPacket = 1000;
                            break;
                        case 1000:
                            Eeprom.EepromData.DemoSettings.MaxNumPacket = 0;
                            break;
                    }
                    ButtonChangeText( BTN5_BORDER, NULL, ( uint8_t* )GetMenuDemoMaxNumPacket( ) );
                }
                else if( graphObjectTouched == BTN7_BORDER )
                {
                    uint8_t modTmp = Eeprom.EepromData.DemoSettings.ModulationType;
                    
                    EepromMcuReadBuffer( 0, Eeprom.Buffer, EEPROM_BUFFER_SIZE );
                    EepromLoadGeneralSettings( );
                    EepromLoadSettings( ( RadioPacketTypes_t )modTmp );
                    MenuSetPage( START_PAGE );
                }
                else if( graphObjectTouched == BTN8_BORDER )
                {
                    EepromSaveSettings( DEMO_SETTINGS );
                    MenuSetPage( START_PAGE );
                }
                else if( refresh == true )
                {
                    MenuSetPage( PAGE_UTILITIES );
                }
                break;

            case PAGE_RADIO_TEST:
                if( graphObjectTouched == BTN0b_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    MenuSetPage( PAGE_SLEEP_MODE );
                }
                else if( graphObjectTouched == BTN1_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    MenuSetPage( PAGE_STBY_RC_MODE );
                }
                else if( graphObjectTouched == BTN2_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    MenuSetPage( PAGE_STBY_XOSC_MODE );
                }
                else if( graphObjectTouched == BTN3_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    MenuSetPage( PAGE_TX_CW );
                }
                else if( graphObjectTouched == BTN4_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    MenuSetPage( PAGE_CONT_MODULATION );
                }
                else if( graphObjectTouched == BTN5_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    MenuSetPage( PAGE_CONT_RX );
                }
                else if( graphObjectTouched == BTN6_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    MenuSetPage( PAGE_RSSI_TEST );
                }
                else if( graphObjectTouched == BTN7_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    MenuSetPage( START_PAGE );
                }
                break;

            case PAGE_RSSI_TEST:
                if( graphObjectTouched == BTN7_BORDER )
                {
                    StopDemoApplication( );
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    PreviousPage = PAGE_RADIO_TEST; // clear Previous page
                    MenuSetPage( PAGE_RADIO_TEST );
                }
                else if( graphObjectTouched == BTN8_BORDER )
                {
                    if( Eeprom.EepromData.DemoSettings.HoldDemo == true )
                    {
                        Eeprom.EepromData.DemoSettings.HoldDemo = false;
                        StopDemoApplication( );
                        ButtonChangeText( BTN8_BORDER, ( uint8_t* )"    STOP", NULL );
                        RunDemoTestRssi( );
                    }
                    else
                    {
                        Eeprom.EepromData.DemoSettings.HoldDemo = true;
                        StopDemoApplication( );
                        ButtonChangeText( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                    }
                }
                else if( graphObjectTouched == BTN9_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    StopDemoApplication( );
                    // "CANCEL" or "OK & SAVE" button of PAGE_RADIO_PARAM use
                    // PreviousPage to set next page. Speed up the navigation.
                    PreviousPage = PAGE_RSSI_TEST;
                    MenuSetPage( PAGE_RADIO_PARAM );
                }
                break;

            case PAGE_SLEEP_MODE:
                if( graphObjectTouched == BTN7_BORDER )
                {
                    StopDemoApplication( );
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    PreviousPage = PAGE_RADIO_TEST; // clear Previous page
                    MenuSetPage( PAGE_RADIO_TEST );
                }
                else if( graphObjectTouched == BTN8_BORDER )
                {
                    if( Eeprom.EepromData.DemoSettings.HoldDemo == true )
                    {
                        Eeprom.EepromData.DemoSettings.HoldDemo = false;
                        StopDemoApplication( );
                        ButtonChangeText( BTN8_BORDER, ( uint8_t* )"    STOP", NULL );
                        RunDemoSleepMode( );
                    }
                    else
                    {
                        Eeprom.EepromData.DemoSettings.HoldDemo = true;
                        StopDemoApplication( );
                        ButtonChangeText( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                        MenuSetPage( PAGE_SLEEP_MODE );
                    }
                }
                else if( graphObjectTouched == BTN9_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    StopDemoApplication( );
                    // "CANCEL" or "OK & SAVE" button of PAGE_RADIO_PARAM use
                    // PreviousPage to set next page. Speed up the navigation.
                    PreviousPage = PAGE_SLEEP_MODE;
                    MenuSetPage( PAGE_RADIO_PARAM );
                }
                break;

            case PAGE_STBY_RC_MODE:
                if( graphObjectTouched == BTN7_BORDER )
                {
                    StopDemoApplication( );
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    PreviousPage = PAGE_RADIO_TEST; // clear Previous page
                    MenuSetPage( PAGE_RADIO_TEST );
                }
                else if( graphObjectTouched == BTN8_BORDER )
                {
                    if( Eeprom.EepromData.DemoSettings.HoldDemo == true )
                    {
                        Eeprom.EepromData.DemoSettings.HoldDemo = false;
                        StopDemoApplication( );
                        ButtonChangeText( BTN8_BORDER, ( uint8_t* )"    STOP", NULL );
                        RunDemoStandbyRcMode( );
                    }
                    else
                    {
                        Eeprom.EepromData.DemoSettings.HoldDemo = true;
                        StopDemoApplication( );
                        ButtonChangeText( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                    }
                }
                else if( graphObjectTouched == BTN9_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    StopDemoApplication( );
                    // "CANCEL" or "OK & SAVE" button of PAGE_RADIO_PARAM use
                    // PreviousPage to set next page. Speed up the navigation.
                    PreviousPage = PAGE_STBY_RC_MODE;
                    MenuSetPage( PAGE_RADIO_PARAM );
                }
                break;

            case PAGE_STBY_XOSC_MODE:
                if( graphObjectTouched == BTN7_BORDER )
                {
                    StopDemoApplication( );
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    PreviousPage = PAGE_RADIO_TEST; // clear Previous page
                    MenuSetPage( PAGE_RADIO_TEST );
                }
                else if( graphObjectTouched == BTN8_BORDER )
                {
                    if( Eeprom.EepromData.DemoSettings.HoldDemo == true )
                    {
                        Eeprom.EepromData.DemoSettings.HoldDemo = false;
                        StopDemoApplication( );
                        ButtonChangeText( BTN8_BORDER, ( uint8_t* )"    STOP", NULL );
                        RunDemoStandbyXoscMode( );
                    }
                    else
                    {
                        Eeprom.EepromData.DemoSettings.HoldDemo = true;
                        StopDemoApplication( );
                        ButtonChangeText( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                    }
                }
                else if( graphObjectTouched == BTN9_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    StopDemoApplication( );
                    // "CANCEL" or "OK & SAVE" button of PAGE_RADIO_PARAM use
                    // PreviousPage to set next page. Speed up the navigation.
                    PreviousPage = PAGE_STBY_XOSC_MODE;
                    MenuSetPage( PAGE_RADIO_PARAM );
                }
                break;

            case PAGE_TX_CW:
                if( graphObjectTouched == BTN7_BORDER )
                {
                    StopDemoApplication( );
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    PreviousPage = PAGE_RADIO_TEST; // clear Previous page
                    MenuSetPage( PAGE_RADIO_TEST );
                }
                else if( graphObjectTouched == BTN8_BORDER )
                {
                    if( Eeprom.EepromData.DemoSettings.HoldDemo == true )
                    {
                        Eeprom.EepromData.DemoSettings.HoldDemo = false;
                        StopDemoApplication( );
                        ButtonChangeText( BTN8_BORDER, ( uint8_t* )"    STOP", NULL );
                        RunDemoTxCw( );
                    }
                    else
                    {
                        Eeprom.EepromData.DemoSettings.HoldDemo = true;
                        StopDemoApplication( );
                        ButtonChangeText( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                    }
                }
                else if( graphObjectTouched == BTN9_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    StopDemoApplication( );
                    // "CANCEL" or "OK & SAVE" button of PAGE_RADIO_PARAM use
                    // PreviousPage to set next page. Speed up the navigation.
                    PreviousPage = PAGE_TX_CW;
                    MenuSetPage( PAGE_RADIO_PARAM );
                }
                break;

            case PAGE_CONT_MODULATION:
                if( graphObjectTouched == BTN7_BORDER )
                {
                    StopDemoApplication( );
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    PreviousPage = PAGE_RADIO_TEST; // clear Previous page
                    MenuSetPage( PAGE_RADIO_TEST );
                }
                else if( graphObjectTouched == BTN8_BORDER )
                {
                    if( Eeprom.EepromData.DemoSettings.HoldDemo == true )
                    {
                        Eeprom.EepromData.DemoSettings.HoldDemo = false;
                        StopDemoApplication( );
                        ButtonChangeText( BTN8_BORDER, ( uint8_t* )"    STOP", NULL );
                        RunDemoTxContinuousModulation( );
                    }
                    else
                    {
                        Eeprom.EepromData.DemoSettings.HoldDemo = true;
                        StopDemoApplication( );
                        ButtonChangeText( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                        wait_ms( 200 ); // debouncing of the touch screen
                    }
                }
                else if( graphObjectTouched == BTN9_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    StopDemoApplication( );
                    // "CANCEL" or "OK & SAVE" button of PAGE_RADIO_PARAM use
                    // PreviousPage to set next page. Speed up the navigation.
                    PreviousPage = PAGE_CONT_MODULATION;
                    MenuSetPage( PAGE_RADIO_PARAM );
                }
                break;
            case PAGE_CONT_RX:
                if( graphObjectTouched == BTN7_BORDER )
                {
                    StopDemoApplication( );
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    PreviousPage = PAGE_RADIO_TEST; // clear Previous page
                    MenuSetPage( PAGE_RADIO_TEST );
                }
                else if( graphObjectTouched == BTN8_BORDER )
                {
                    if( Eeprom.EepromData.DemoSettings.HoldDemo == true )
                    {
                        Eeprom.EepromData.DemoSettings.HoldDemo = false;
                        StopDemoApplication( );
                        ButtonChangeText( BTN8_BORDER, ( uint8_t* )"    STOP", NULL );
                        RunDemoRxContinuous( );
                    }
                    else
                    {
                        Eeprom.EepromData.DemoSettings.HoldDemo = true;
                        StopDemoApplication( );
                        ButtonChangeText( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                    }
                }
                else if( graphObjectTouched == BTN9_BORDER )
                {
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    StopDemoApplication( );
                    // "CANCEL" or "OK & SAVE" button of PAGE_RADIO_PARAM use
                    // PreviousPage to set next page. Speed up the navigation.
                    PreviousPage = PAGE_CONT_RX;
                    MenuSetPage( PAGE_RADIO_PARAM );
                }
                break;
        }
    }
    return CurrentPage;
}

void MenuSetPage( uint8_t page )
{
    if( page != CurrentPage )
    {
        // remove all button and text (except title, page border and line)
        // and disable all button (touchActive = false).
        GraphObject_t clearRect = { NULL, GO_RECTANGLE, 3, 35, 282, 230, 1, \
                                    BACK_COLOR, BACK_COLOR, true, BACK_COLOR, \
                                    NULL, false };

        GraphObjectClear( &clearRect, true );
        for( uint8_t i = 0; i < GRAPH_OBJECTS_COUNT; i++ )
        {
            MenuObjects[i].TouchActive = false;
        }
    }
    
    switch( page )
    {
        case START_PAGE:
            if( DeviceConnected == SX1261 )
            {
                if( MatchingFreq == MATCHING_FREQ_915 )
                {
                    DrawText( TITLE_TEXT, ( uint8_t* )"  SX1261 915MHz Demo Kit", NULL, NULL );
                }
                else if( MatchingFreq == MATCHING_FREQ_868 )
                {
                    DrawText( TITLE_TEXT, ( uint8_t* )"  SX1261 868MHz Demo Kit", NULL, NULL );
                }
            }
            else if( DeviceConnected == SX1262 )
            {
                if( MatchingFreq == MATCHING_FREQ_915 )
                {
                    DrawText( TITLE_TEXT, ( uint8_t* )"  SX1262 915MHz Demo Kit", NULL, NULL );
                }
                else if( MatchingFreq == MATCHING_FREQ_868 )
                {
                    DrawText( TITLE_TEXT, ( uint8_t* )"  SX1262 868MHz Demo Kit", NULL, NULL );
                }
            }
            else if( DeviceConnected == SX1268 )
            {
                if( MatchingFreq == MATCHING_FREQ_780 )
                {
                    DrawText( TITLE_TEXT, ( uint8_t* )"  SX1268 780MHz Demo Kit", NULL, NULL );
                }
                else if( MatchingFreq == MATCHING_FREQ_490 )
                {
                    DrawText( TITLE_TEXT, ( uint8_t* )"  SX1268 490MHz Demo Kit", NULL, NULL );
                }
            }

            DisplayDriverDrawLogo( SemtechLogo, 20, 37 );
            DrawActiveButton( BTN2_BORDER, ( uint8_t* )"DEMO Ping Pong", NULL );
            DrawActiveButton( BTN3_BORDER, ( uint8_t* )"DEMO PER", NULL );
            DrawActiveButton( BTN4_BORDER, ( uint8_t* )"Radio Test Modes", NULL );
            DrawActiveButton( BTN5_BORDER, ( uint8_t* )"Radio Settings", NULL );
            DrawActiveButton( BTN6_BORDER, ( uint8_t* )"Utilities", NULL );
            CurrentPage = START_PAGE;
            break;

        case PAGE_PING_PONG:
            if( CurrentPage == PAGE_PING_PONG )
            {
                if( Eeprom.EepromData.DemoSettings.HoldDemo == true )
                {
                    ButtonChangeText( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                }
                if( Eeprom.EepromData.DemoSettings.Entity == MASTER )
                {
                    DrawText( LINE4_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoRxOkSlave( ), ( uint8_t* )GetMenuDemoRxOk( ) );
                    DrawText( LINE5_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoRxKoSlave( ), ( uint8_t* )GetMenuDemoRxKo( ) );
                    DrawText( LINE6_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoResultPerCent1( Eeprom.EepromData.DemoSettings.CntPacketRxOKSlave, Eeprom.EepromData.DemoSettings.CntPacketRxOKSlave + Eeprom.EepromData.DemoSettings.CntPacketRxKOSlave ), \
                              ( uint8_t* )GetMenuDemoResultPerCent2( Eeprom.EepromData.DemoSettings.CntPacketRxOK, Eeprom.EepromData.DemoSettings.CntPacketRxOK + Eeprom.EepromData.DemoSettings.CntPacketRxKO + Eeprom.EepromData.DemoSettings.RxTimeOutCount ) );
                    DrawText( LINE7_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoResultPerCent1( Eeprom.EepromData.DemoSettings.CntPacketRxKOSlave, Eeprom.EepromData.DemoSettings.CntPacketRxOKSlave + Eeprom.EepromData.DemoSettings.CntPacketRxKOSlave ), \
                              ( uint8_t* )GetMenuDemoResultPerCent2( Eeprom.EepromData.DemoSettings.CntPacketRxKO + Eeprom.EepromData.DemoSettings.RxTimeOutCount, Eeprom.EepromData.DemoSettings.CntPacketRxOK + Eeprom.EepromData.DemoSettings.CntPacketRxKO + Eeprom.EepromData.DemoSettings.RxTimeOutCount ) );
                    DrawText( LINE8_COL0_TEXT, NULL, ( uint8_t* )"     ", ( uint8_t* )GetMenuDemoRssi( ) );
                    DrawText( LINE9_COL0_TEXT, NULL, ( uint8_t* )"     ", ( uint8_t* )GetMenuDemoSnr( ) );
                }
                else
                {  
                    DrawText( LINE4_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoRxOk( ), NULL );
                    DrawText( LINE5_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoRxKo( ), NULL );
                    DrawText( LINE6_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoResultPerCent1( Eeprom.EepromData.DemoSettings.CntPacketRxOK, Eeprom.EepromData.DemoSettings.CntPacketRxOK + Eeprom.EepromData.DemoSettings.CntPacketRxKO + Eeprom.EepromData.DemoSettings.RxTimeOutCount ), NULL );
                    DrawText( LINE7_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoResultPerCent1( Eeprom.EepromData.DemoSettings.CntPacketRxKO + Eeprom.EepromData.DemoSettings.RxTimeOutCount, Eeprom.EepromData.DemoSettings.CntPacketRxOK + Eeprom.EepromData.DemoSettings.CntPacketRxKO + Eeprom.EepromData.DemoSettings.RxTimeOutCount ), NULL );
                    DrawText( LINE8_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoRssi( ), NULL );
                    DrawText( LINE9_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoSnr( ), NULL );
                }
            }
            else
            {
                DrawText( TITLE_TEXT, ( uint8_t* )"      PING PONG Demo", NULL, NULL );
                DisplayCurrentRadioParams( PAGE_PING_PONG );
                DrawText( LINE3_COL0_TEXT, ( uint8_t* )GetTotalPackets( ), ( uint8_t* )" SLAVE", NULL );
                DrawText( LINE4_COL0_TEXT, ( uint8_t* )"Rx OK", ( uint8_t* )"     0", NULL );
                DrawText( LINE5_COL0_TEXT, ( uint8_t* )"Rx KO", ( uint8_t* )"     0", NULL );
                DrawText( LINE6_COL0_TEXT, ( uint8_t* )"Rx PSR", ( uint8_t* )"  0.00", NULL );
                DrawText( LINE7_COL0_TEXT, ( uint8_t* )"Rx PER", ( uint8_t* )"  0.00", NULL );

                if( Eeprom.EepromData.DemoSettings.Entity == MASTER )
                { 
                    DrawActiveButton( BTN0_BORDER, ( uint8_t* )"   MASTER", NULL );
                    DrawText( LINE3_COL0_TEXT, NULL, NULL, ( uint8_t* )"MASTER" );
                    DrawText( LINE4_COL0_TEXT, NULL, NULL, ( uint8_t* )"     0" );
                    DrawText( LINE5_COL0_TEXT, NULL, NULL, ( uint8_t* )"     0" );
                    DrawText( LINE6_COL0_TEXT, NULL, NULL, ( uint8_t* )"  0.00" );
                    DrawText( LINE7_COL0_TEXT, NULL, NULL, ( uint8_t* )"  0.00" );
                    DrawText( LINE8_COL0_TEXT, ( uint8_t* )"Last Rssi", ( uint8_t* )"    ", ( uint8_t* )"  0" );
                    DrawText( LINE9_COL0_TEXT, ( uint8_t* )"Last SNR", ( uint8_t* )"    ", ( uint8_t* )"  0" );
                }
                else
                {
                    DrawText( LINE8_COL0_TEXT, ( uint8_t* )"Last Rssi", ( uint8_t* )"     0", ( uint8_t* )"    ");
                    DrawText( LINE9_COL0_TEXT, ( uint8_t* )"Last SNR", ( uint8_t* )"     0", ( uint8_t* )"    " );
                    DrawActiveButton( BTN0_BORDER, ( uint8_t* )"   SLAVE", NULL );
                }
                DrawActiveButton( BTN7_BORDER, ( uint8_t* )"    BACK", NULL );
                DrawActiveButton( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                DrawActiveButton( BTN9_BORDER, ( uint8_t* )"  SETTINGS", NULL );
            }
            CurrentPage = PAGE_PING_PONG;
            break;
        case PAGE_PER:
            if( CurrentPage == PAGE_PER )
            {
                if( Eeprom.EepromData.DemoSettings.HoldDemo == true )
                {
                    ButtonChangeText( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                }
                if( Eeprom.EepromData.DemoSettings.Entity == MASTER )
                { 
                    DrawText( LINE4_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoNumSentPacket( ), NULL );
                }
                else
                {  
                    DrawText( LINE4_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoRxOk( ), NULL );
                    DrawText( LINE5_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoRxKo( ), NULL );
                    DrawText( LINE6_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoResultPerCent1( Eeprom.EepromData.DemoSettings.CntPacketRxKO + Eeprom.EepromData.DemoSettings.RxTimeOutCount, Eeprom.EepromData.DemoSettings.CntPacketRxOK + Eeprom.EepromData.DemoSettings.CntPacketRxKO + Eeprom.EepromData.DemoSettings.RxTimeOutCount ), NULL );
                    DrawText( LINE7_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoFreqError( ), NULL );
                    DrawText( LINE8_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoRssi( ), NULL );
                    DrawText( LINE9_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoSnr( ), NULL );
                }
            }
            else
            {
                DrawText( TITLE_TEXT, ( uint8_t* )"         PER Demo", NULL, NULL );
                DrawText( LINE3_COL0_TEXT, ( uint8_t* )"Test Total", ( uint8_t* )GetMenuDemoMaxNumPacket( ), NULL );
                DisplayCurrentRadioParams( PAGE_PER );

                if( Eeprom.EepromData.DemoSettings.Entity == MASTER )
                { 
                    DrawActiveButton( BTN0_BORDER, ( uint8_t* )"   MASTER", NULL );
                    DrawText( LINE4_COL0_TEXT, ( uint8_t* )"Transmitted", ( uint8_t* )GetMenuDemoNumSentPacket( ), NULL );
                }
                else
                {
                    DrawActiveButton( BTN0_BORDER, ( uint8_t* )"   SLAVE", NULL );
                    DrawText( LINE4_COL0_TEXT, ( uint8_t* )"Rx OK", ( uint8_t* )"     0", NULL );
                    DrawText( LINE5_COL0_TEXT, ( uint8_t* )"Rx KO", ( uint8_t* )"     0", NULL );
                    DrawText( LINE6_COL0_TEXT, ( uint8_t* )"PER",   ( uint8_t* )"  0.00", NULL );
                    DrawText( LINE7_COL0_TEXT, ( uint8_t* )"Freq Err", ( uint8_t* )"     0", NULL );
                    DrawText( LINE8_COL0_TEXT, ( uint8_t* )"Last Rssi", ( uint8_t* )"     0", NULL );
                    DrawText( LINE9_COL0_TEXT, ( uint8_t* )"Last SNR", ( uint8_t* )"     0", NULL );
                }
                DrawActiveButton( BTN7_BORDER, ( uint8_t* )"    BACK", NULL );
                DrawActiveButton( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                DrawActiveButton( BTN9_BORDER, ( uint8_t* )"  SETTINGS", NULL );
            }
            CurrentPage = PAGE_PER;
            break;

        case PAGE_RADIO_PARAM:
            if( ErrorParam == true )
            {
                MenuObjects[BTN3_VALUE_TEXT].FrontColor = OBJECT_ERROR;
                MenuObjects[BTN4_VALUE_TEXT].FrontColor = OBJECT_ERROR;
                MenuObjects[BTN6_VALUE_TEXT].FrontColor = OBJECT_ERROR;
            }
            if( CurrentPage != PAGE_RADIO_PARAM )
            {
                DrawText( TITLE_TEXT, ( uint8_t* )"      Radio Settings", NULL, NULL );
                DrawActiveButton( BTN0_BORDER, ( uint8_t* )"Mod:", ( uint8_t* )GetMenuRadioFrameType( ) );
                DrawActiveButton( BTN1_BORDER, ( uint8_t* )"Freq:", ( uint8_t* )GetRadioFrequency( ) );
                DrawActiveButton( BTN2_BORDER, ( uint8_t* )"Tx Pow:", ( uint8_t* )GetRadioTxPower( ) );
                DrawActiveButton( BTN3_BORDER, ( uint8_t* )"Param 1:", ( uint8_t* )GetRadioModulationParameters1( ) );
                DrawActiveButton( BTN4_BORDER, ( uint8_t* )"Param 2:", ( uint8_t* )GetRadioModulationParameters2( ) );
                DrawActiveButton( BTN5_BORDER, ( uint8_t* )"Param 3:", ( uint8_t* )GetRadioModulationParameters3( ) );
                DrawActiveButton( BTN6_BORDER, ( uint8_t* )"Param 4:", ( uint8_t* )GetRadioModulationParameters4( ) );
                DrawActiveButton( BTN7_BORDER, ( uint8_t* )"   CANCEL", NULL );
                DrawActiveButton( BTN8_BORDER, ( uint8_t* )" OK & SAVE", NULL );
                DrawActiveButton( BTN9_BORDER, ( uint8_t* )"PL:", ( uint8_t* )GetRadioPayloadLength( ) );
            }
            MenuObjects[BTN3_VALUE_TEXT].FrontColor = TEXT_VALUE;
            MenuObjects[BTN4_VALUE_TEXT].FrontColor = TEXT_VALUE;
            MenuObjects[BTN6_VALUE_TEXT].FrontColor = TEXT_VALUE;
            CurrentPage = PAGE_RADIO_PARAM;
            break;

        case PAGE_RADIO_PARAM_FREQ:
            if( CurrentPage != PAGE_RADIO_PARAM_FREQ )
            {
                DrawText( TITLE_TEXT, ( uint8_t* )"   Radio Settings : Freq", NULL, NULL );
                DrawActiveButton( BTN0_BORDER, ( uint8_t* )"     OK", NULL );
                DrawActiveButton( BTN2_BORDER, ( uint8_t* )"Freq:", ( uint8_t* )GetRadioFrequency( ) );
                DrawActiveButton( BTN3_BORDER, ( uint8_t* )"Preset1:", ( uint8_t* )GetRadioFreqBasePS1( ) );
                DrawActiveButton( BTN4_BORDER, ( uint8_t* )"Preset2:", ( uint8_t* )GetRadioFreqBasePS2( ) );
                DrawActiveButton( BTN5_BORDER, ( uint8_t* )"Preset3:", ( uint8_t* )GetRadioFreqBasePS3( ) );
                DrawActiveButton( BTN6_BORDER, ( uint8_t* )"Step:", ( uint8_t* )GetRadioFreqBase( ) );
                DrawActiveButton( BTN7_BORDER, ( uint8_t* )"     -", NULL );
                DrawActiveButton( BTN8_BORDER, ( uint8_t* )"     +", NULL );
            }
            CurrentPage = PAGE_RADIO_PARAM_FREQ;
            break;

        case PAGE_UTILITIES:
            if( CurrentPage != PAGE_UTILITIES )
            {
                DrawText( TITLE_TEXT, ( uint8_t* )"         Utilities", NULL, NULL );
                // FIRMWARE_VERSION is defined in DemoApplication.h
                DrawText( LINE0_TEXT, ( uint8_t* )FIRMWARE_VERSION, NULL, NULL );

                DrawActiveButton( BTN0_BORDER, ( uint8_t* )"Fact. Reset", NULL );
                DrawActiveButton( BTN3_BORDER, ( uint8_t* )"RxBoost:", ( uint8_t* )GetMenuDemoRadioBoostedRx( ) );
                DrawActiveButton( BTN4_BORDER, ( uint8_t* )"DC Reg: ", ( uint8_t* )GetMenuDemoRadioPowerMode( ) );
                DrawActiveButton( BTN5_BORDER, ( uint8_t* )"Packets:", ( uint8_t* )GetMenuDemoMaxNumPacket( ) );
                DrawActiveButton( BTN7_BORDER, ( uint8_t* )"   CANCEL", NULL );
                DrawActiveButton( BTN8_BORDER, ( uint8_t* )" OK & SAVE", NULL );
            }
            CurrentPage = PAGE_UTILITIES;
            break;

        case PAGE_RADIO_TEST:
            DrawText( TITLE_TEXT, ( uint8_t* )"     Radio Test Modes", NULL, NULL );
            DrawActiveButton( BTN0b_BORDER, ( uint8_t* )"        Sleep Mode", NULL );
            DrawActiveButton( BTN1_BORDER, ( uint8_t* )"      Standby RC Mode", NULL );
            DrawActiveButton( BTN2_BORDER, ( uint8_t* )"     Standby XOSC Mode", NULL );
            DrawActiveButton( BTN3_BORDER, ( uint8_t* )"        Set Tx CW", NULL );
            DrawActiveButton( BTN4_BORDER, ( uint8_t* )"    Set Cont. Modulation", NULL );
            DrawActiveButton( BTN5_BORDER, ( uint8_t* )"     Set Cont. Receive", NULL );
            DrawActiveButton( BTN6_BORDER, ( uint8_t* )"         Test Rssi ", NULL );
            DrawActiveButton( BTN7_BORDER, ( uint8_t* )"   BACK", NULL );
            CurrentPage = PAGE_RADIO_TEST;
            break;

        case PAGE_RSSI_TEST:
            if( CurrentPage == PAGE_RSSI_TEST )
            {
                if( Eeprom.EepromData.DemoSettings.HoldDemo == true )
                {
                    ButtonChangeText( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                }
                DrawText( LINE6_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoRssi( ), NULL );
            }
            else
            {
                DrawText( TITLE_TEXT, ( uint8_t* )"       Test Rssi   ", NULL, NULL );
                DisplayCurrentRadioParams( PAGE_CONT_RX );
                DrawText( LINE6_COL0_TEXT, ( uint8_t* )"Rssi",   ( uint8_t* )"  0.00", NULL );
                DrawActiveButton( BTN7_BORDER, ( uint8_t* )"    BACK", NULL );
                DrawActiveButton( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                DrawActiveButton( BTN9_BORDER, ( uint8_t* )"  SETTINGS", NULL );
            }
            CurrentPage = PAGE_RSSI_TEST;
            break;
        
        case PAGE_SLEEP_MODE:
            if( CurrentPage != PAGE_SLEEP_MODE )
            {
                DrawText( TITLE_TEXT, ( uint8_t* )"        Sleep Mode", NULL, NULL );
                DisplayCurrentRadioParams( PAGE_SLEEP_MODE );
                DrawActiveButton( BTN7_BORDER, ( uint8_t* )"    BACK", NULL );
                DrawActiveButton( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                DrawActiveButton( BTN9_BORDER, ( uint8_t* )"  SETTINGS", NULL );
                CurrentPage = PAGE_SLEEP_MODE;
            }
            break;

        case PAGE_STBY_RC_MODE:
            if( CurrentPage != PAGE_STBY_RC_MODE )
            {
                DrawText( TITLE_TEXT, ( uint8_t* )"     Standby RC Mode", NULL, NULL );
                DisplayCurrentRadioParams( PAGE_STBY_RC_MODE );
                DrawActiveButton( BTN7_BORDER, ( uint8_t* )"    BACK", NULL );
                DrawActiveButton( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                DrawActiveButton( BTN9_BORDER, ( uint8_t* )"  SETTINGS", NULL );
                CurrentPage = PAGE_STBY_RC_MODE;
            }
            break;

        case PAGE_STBY_XOSC_MODE:
            if( CurrentPage != PAGE_STBY_XOSC_MODE )
            {
                DrawText( TITLE_TEXT, ( uint8_t* )"     Standby XOSC Mode", NULL, NULL );
                DisplayCurrentRadioParams( PAGE_STBY_XOSC_MODE );
                DrawActiveButton( BTN7_BORDER, ( uint8_t* )"    BACK", NULL );
                DrawActiveButton( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                DrawActiveButton( BTN9_BORDER, ( uint8_t* )"  SETTINGS", NULL );
                CurrentPage = PAGE_STBY_XOSC_MODE;
            }
            break;

        case PAGE_TX_CW:
            if( CurrentPage != PAGE_TX_CW )
            {
                DrawText( TITLE_TEXT, ( uint8_t* )"        Set Tx CW", NULL, NULL );
                DisplayCurrentRadioParams( PAGE_TX_CW );
                DrawActiveButton( BTN7_BORDER, ( uint8_t* )"    BACK", NULL );
                DrawActiveButton( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                DrawActiveButton( BTN9_BORDER, ( uint8_t* )"  SETTINGS", NULL );
                CurrentPage = PAGE_TX_CW;
            }
            break;

        case PAGE_CONT_MODULATION:
            if( CurrentPage != PAGE_CONT_MODULATION )
            {
                DrawText( TITLE_TEXT, ( uint8_t* )"   Set Cont. Modulation", NULL, NULL );
                DisplayCurrentRadioParams( PAGE_CONT_MODULATION );
                DrawActiveButton( BTN7_BORDER, ( uint8_t* )"    BACK", NULL );
                DrawActiveButton( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                DrawActiveButton( BTN9_BORDER, ( uint8_t* )"  SETTINGS", NULL );
                CurrentPage = PAGE_CONT_MODULATION;
            }
            break;

        case PAGE_CONT_RX:
            if( CurrentPage == PAGE_CONT_RX )
            {
                if( Eeprom.EepromData.DemoSettings.HoldDemo == true )
                {
                    ButtonChangeText( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                }
                DrawText( LINE4_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoRxOk( ), NULL );
                DrawText( LINE5_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoRxKo( ), NULL );
                DrawText( LINE6_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoResultPerCent1( Eeprom.EepromData.DemoSettings.CntPacketRxKO, Eeprom.EepromData.DemoSettings.CntPacketRxOK + Eeprom.EepromData.DemoSettings.CntPacketRxKO ), NULL );
                DrawText( LINE7_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoFreqError( ), NULL );
                DrawText( LINE8_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoRssi( ), NULL );
                DrawText( LINE9_COL0_TEXT, NULL, ( uint8_t* )GetMenuDemoSnr( ), NULL );
            }
            else
            {
                DrawText( TITLE_TEXT, ( uint8_t* )"       Set Cont. Rx   ", NULL, NULL );
                DisplayCurrentRadioParams( PAGE_CONT_RX );
                DrawText( LINE4_COL0_TEXT, ( uint8_t* )"Rx OK", ( uint8_t* )"     0", NULL );
                DrawText( LINE5_COL0_TEXT, ( uint8_t* )"Rx KO", ( uint8_t* )"     0", NULL );
                DrawText( LINE6_COL0_TEXT, ( uint8_t* )"PER",   ( uint8_t* )"  0.00", NULL );
                DrawText( LINE7_COL0_TEXT, ( uint8_t* )"Freq Err", ( uint8_t* )"     0", NULL );
                DrawText( LINE8_COL0_TEXT, ( uint8_t* )"Last Rssi", ( uint8_t* )"     0", NULL );
                DrawText( LINE9_COL0_TEXT, ( uint8_t* )"Last SNR", ( uint8_t* )"     0", NULL );
                DrawActiveButton( BTN7_BORDER, ( uint8_t* )"    BACK", NULL );
                DrawActiveButton( BTN8_BORDER, ( uint8_t* )"   START", NULL );
                DrawActiveButton( BTN9_BORDER, ( uint8_t* )"  SETTINGS", NULL );
            }
            CurrentPage = PAGE_CONT_RX;
            break;

        default:
            break;
    }
}

void DisplayCurrentRadioParams( uint8_t page )
{
    switch( page )
    {
        case PAGE_SLEEP_MODE:
        case PAGE_STBY_RC_MODE:
        case PAGE_STBY_XOSC_MODE:
        case PAGE_TX_CW:
            // Do display different without packet settings (no modulation here)
            // 2 + (9) + 4 + (7) = 22 (=<28 : ok)
            sprintf( StringText ,"f:%s, P:%s", GetRadioFrequencyMHz( ), GetRadioTxPower( ) );           // StringText3 + StringText2
            DrawText( LINE0_TEXT, ( uint8_t* )StringText, NULL, NULL);
            sprintf( StringText ,"Radio Power Mode: %s", ( uint8_t* )GetMenuDemoRadioPowerMode( ) + 2 );
            DrawText( LINE1_TEXT, ( uint8_t* )StringText, NULL, NULL);
            sprintf( StringText ,"Press START to launch test" );
            DrawText( LINE3_TEXT, ( uint8_t* )StringText, NULL, NULL);
            break;

        case PAGE_CONT_MODULATION:
            // 2 + (9) + 4 + (7) + 2 + (4) = 28 (=<28 : ok)
            sprintf( StringText ,"f:%s, P:%s, %s", GetRadioFrequencyMHz( ), GetRadioTxPower( ), \
                                                   ( uint8_t* )GetMenuDemoRadioPowerMode( ) + 2 );      // StringText3 + StringText2
            DrawText( LINE0_TEXT, ( uint8_t* )StringText, NULL, NULL);
            if( Eeprom.EepromData.DemoSettings.ModulationType == PACKET_TYPE_LORA )
            {
                // (5) + 2 + (4) + 2 + (12) = 25 (=<28 : ok)
                sprintf( StringText ,"%s: %s, %s", GetMenuRadioFrameType( ) + 1, \
                                                   GetRadioModulationParameters1( ),
                                                   GetRadioModulationParameters4( ) );                  // StringText2
            }
            else
            {
                if( ErrorParam == true )
                {
                    MenuObjects[LINE1_TEXT].FrontColor = OBJECT_ERROR;
                    MenuObjects[LINE2_TEXT].FrontColor = OBJECT_ERROR;
                }
                // (4) + 2 + (10) + 1 + (11) = 28 (=<28 : ok)
                sprintf( StringText ,"%s: %s/%s", GetMenuRadioFrameType( ) + 1, \
                                                  GetRadioModulationParameters1( ), \
                                                  GetRadioModulationParameters4( ) );                   // StringText2
            }
            DrawText( LINE1_TEXT, ( uint8_t* )StringText, NULL, NULL);
            // (10) + 2 + (10) = 22 (=<28 : ok)
            sprintf( StringText ,"%s, %s", GetRadioModulationParameters2( ), \
                                           GetRadioModulationParameters3( ) );
            DrawText( LINE2_TEXT, ( uint8_t* )StringText, NULL, NULL);
            sprintf( StringText ,"Press START to launch test" );
            DrawText( LINE4_TEXT, ( uint8_t* )StringText, NULL, NULL);
            MenuObjects[LINE1_TEXT].FrontColor = TEXT_COLOR;
            MenuObjects[LINE2_TEXT].FrontColor = TEXT_COLOR;
            break;

        case PAGE_CONT_RX:      // standart display
        case PAGE_PING_PONG:
        case PAGE_PER:
            // 2 + (9) + 4 + (7) = 22 (=<28 : ok)
            sprintf( StringText ,"f:%s, P:%s", GetRadioFrequencyMHz( ), GetRadioTxPower( ) );           // StringText3 + StringText2
            DrawText( LINE0_TEXT, ( uint8_t* )StringText, NULL, NULL);
            if( Eeprom.EepromData.DemoSettings.ModulationType == PACKET_TYPE_LORA )
            {
                // (5) + 2 + (4) + 2 + (12) = 25 (=<28 : ok)
                sprintf( StringText ,"%s: %s, %s", GetMenuRadioFrameType( ) + 1, \
                                                   GetRadioModulationParameters1( ),
                                                   GetRadioModulationParameters4( ) );                  // StringText2
            }
            else
            {
                if( ErrorParam == true )
                {
                    MenuObjects[LINE1_TEXT].FrontColor = OBJECT_ERROR;
                    MenuObjects[LINE2_TEXT].FrontColor = OBJECT_ERROR;
                }
                // (4) + 2 + (10) + 1 + (11) = 28 (=<28 : ok)
                sprintf( StringText ,"%s: %s/%s", GetMenuRadioFrameType( ) + 1, \
                                                  GetRadioModulationParameters1( ), \
                                                  GetRadioModulationParameters4( ) );                   // StringText2
            }
            DrawText( LINE1_TEXT, ( uint8_t* )StringText, NULL, NULL);
            // 3 + (3) + 1 + (10) + 1 + (10) = 28 (=<28 : ok)
            sprintf( StringText ,"PL:%s/%s/%s", GetRadioPayloadLength( ), \
                                                  GetRadioModulationParameters2( ), \
                                                  GetRadioModulationParameters3( ) );                   // StringText + StringText2 + StringText3
            DrawText( LINE2_TEXT, ( uint8_t* )StringText, NULL, NULL);
            MenuObjects[LINE1_TEXT].FrontColor = TEXT_COLOR;
            MenuObjects[LINE2_TEXT].FrontColor = TEXT_COLOR;
            break;
    }
}

char* GetMenuRadioFrameType( void )
{
    switch( Eeprom.EepromData.DemoSettings.ModulationType )
    {
        case PACKET_TYPE_GFSK: return ( char* )" GFSK";
        case PACKET_TYPE_LORA:
        default:               return ( char* )" LORA";
    }
}

char* GetRadioModulationParameters1( void )
{
    if( Eeprom.EepromData.DemoSettings.ModulationType == PACKET_TYPE_LORA )
    {
        switch( Eeprom.EepromData.DemoSettings.ModulationParam1 )
        {
            case LORA_SF5:  return ( char* )"SF5";
            case LORA_SF6:  return ( char* )"SF6";
            case LORA_SF7:  return ( char* )"SF7";
            case LORA_SF8:  return ( char* )"SF8";
            case LORA_SF9:  return ( char* )"SF9";
            case LORA_SF10: return ( char* )"SF10";
            case LORA_SF11: return ( char* )"SF11";
            case LORA_SF12: return ( char* )"SF12";
            default:        return ( char* )"X";
        }
    }
    else // PACKET_TYPE_GFSK
    {
        // quicker than using a float (which is not optimized in MBED)
        sprintf( StringText2, "%3d.%1d kbps", \
                 Eeprom.EepromData.DemoSettings.ModulationParam1 / 1000, \
                 ( Eeprom.EepromData.DemoSettings.ModulationParam1 % 1000) / 100 );
        return StringText2;
    }
}

char* GetRadioModulationParameters2( void )
{
    if( Eeprom.EepromData.DemoSettings.ModulationType == PACKET_TYPE_LORA )
    {
        switch( Eeprom.EepromData.DemoSettings.ModulationParam2 )
        {
            case LORA_BW_500: return ( char* )"BW 500 kHz";
            case LORA_BW_250: return ( char* )"BW 250 kHz";
            case LORA_BW_125: return ( char* )"BW 125 kHz";
            case LORA_BW_062: return ( char* )"BW 62 kHz";
            case LORA_BW_041: return ( char* )"BW 41 kHz";
            case LORA_BW_031: return ( char* )"BW 31 kHz";
            case LORA_BW_020: return ( char* )"BW 20 kHz";
            case LORA_BW_015: return ( char* )"BW 15 kHz";
            case LORA_BW_010: return ( char* )"BW 10 kHz";
            case LORA_BW_007: return ( char* )"BW 7 kHz";
            default:          return ( char* )"X";
        }
    }
    else // Eeprom.EepromData.DemoSettings.ModulationType == PACKET_TYPE_GFSK
    {
        // quicker than using a float (which is not optimized in MBED)
        sprintf( StringText3, "Fd %3d kHz", \
                 Eeprom.EepromData.DemoSettings.ModulationParam2 / 1000 );
        return StringText3;
    }
}

char* GetRadioModulationParameters3( void )
{
    if( Eeprom.EepromData.DemoSettings.ModulationType == PACKET_TYPE_LORA )
    {
        switch( Eeprom.EepromData.DemoSettings.ModulationParam3 )
        {
            case LORA_CR_4_5:       return ( char* )"CR 4/5";
            case LORA_CR_4_6:       return ( char* )"CR 4/6";
            case LORA_CR_4_7:       return ( char* )"CR 4/7";
            case LORA_CR_4_8:       return ( char* )"CR 4/8";
            default:                return ( char* )"X";
        }
    }
    else // PACKET_TYPE_GFSK
    {
        switch( Eeprom.EepromData.DemoSettings.ModulationParam3 )
        {
            case MOD_SHAPING_OFF:          return ( char* )"NONE   ";
            case MOD_SHAPING_G_BT_03:      return ( char* )"G BT .3";
            case MOD_SHAPING_G_BT_05:      return ( char* )"G BT .5";
            case MOD_SHAPING_G_BT_07:      return ( char* )"G BT .7";
            case MOD_SHAPING_G_BT_1:       return ( char* )"G BT 1 ";
            default:                       return ( char* )"X";
        }
    }
}

char* GetRadioModulationParameters4( void )
{
    if( Eeprom.EepromData.DemoSettings.ModulationType == PACKET_TYPE_LORA )
    {
        if( Eeprom.EepromData.DemoSettings.PacketParam2 == LORA_PACKET_FIXED_LENGTH )
        {
            return ( char* )"Fixed length";
        }
        else  // RADIO_PACKET_VARIABLE_LENGTH
        {
            return ( char* )"Variable len";
        }
    }
    else // Eeprom.EepromData.DemoSettings.ModulationType == PACKET_TYPE_GFSK
    {
        switch( Eeprom.EepromData.DemoSettings.ModulationParam4 )
        {
            case RX_BW_4800  :  return ( char* )"BW 4.8 kHz";
            case RX_BW_5800  :  return ( char* )"BW 5.8 kHz";
            case RX_BW_7300  :  return ( char* )"BW 7.3 kHz";
            case RX_BW_9700  :  return ( char* )"BW 9.7 kHz";
            case RX_BW_11700 :  return ( char* )"BW 11.7 kHz";
            case RX_BW_14600 :  return ( char* )"BW 14.6 kHz";
            case RX_BW_19500 :  return ( char* )"BW 19.5 kHz";
            case RX_BW_23400 :  return ( char* )"BW 23.4 kHz";
            case RX_BW_29300 :  return ( char* )"BW 29.3 kHz";
            case RX_BW_39000 :  return ( char* )"BW 39 kHz";
            case RX_BW_46900 :  return ( char* )"BW 46.9 kHz";
            case RX_BW_58600 :  return ( char* )"BW 58.6 kHz";
            case RX_BW_78200 :  return ( char* )"BW 78.2 kHz";
            case RX_BW_93800 :  return ( char* )"BW 93.8 kHz";
            case RX_BW_117300 : return ( char* )"BW 117 kHz";
            case RX_BW_156200 : return ( char* )"BW 156 kHz";
            case RX_BW_187200 : return ( char* )"BW 187 kHz";
            case RX_BW_234300 : return ( char* )"BW 234 kHz";
            case RX_BW_312000 : return ( char* )"BW 312 kHz";
            case RX_BW_373600 : return ( char* )"BW 373 kHz";
            case RX_BW_467000 : return ( char* )"BW 467 kHz";
            default:            return ( char* )"X";
        }
    }
}

char* GetRadioFrequency( void )
{
    sprintf( StringText2, "%lu Hz", \
             ( unsigned long )Eeprom.EepromData.DemoSettings.Frequency );
    return StringText2;
}

char* GetRadioFrequencyMHz( void )
{
    // quicker than using a float (which is not optimized in MBED)
    sprintf( StringText3, "%3d.%1d MHz", \
             Eeprom.EepromData.DemoSettings.Frequency / 1000000, \
             ( Eeprom.EepromData.DemoSettings.Frequency % 1000000 ) / 100000 );
    return StringText3;
}

char* GetRadioFreqBase( void )
{
    switch( CurrentFreqBase )
    {
        case FB1:    return ( char* )"1 Hz";
        case FB10:   return ( char* )"10 Hz";
        case FB100:  return ( char* )"100 Hz";
        case FB1K:   return ( char* )"1 kHz";
        case FB10K:  return ( char* )"10 kHz";
        case FB100K: return ( char* )"100 kHz";
        case FB1M:   return ( char* )"1 MHz";
        case FB10M:  return ( char* )"10 MHz";
        default:     return ( char* )"X";
    }
}

char* GetRadioFreqBasePS1( void )
{
    if( DeviceConnected != SX1268 )
    {
        sprintf( StringText, "%lu Hz", DEMO_CENTRAL_FREQ_PRESET6 );
    }
    else
    {
        sprintf( StringText, "%lu Hz", DEMO_CENTRAL_FREQ_PRESET4 );
    }
    return StringText;
}

char* GetRadioFreqBasePS2( void )
{
    if( DeviceConnected != SX1268 )
    {
        sprintf( StringText, "%lu Hz", DEMO_CENTRAL_FREQ_PRESET7 );
    }
    else
    {
        sprintf( StringText, "%lu Hz", DEMO_CENTRAL_FREQ_PRESET3 );
    }
    return StringText;
}

char* GetRadioFreqBasePS3( void )
{
    if( DeviceConnected != SX1268 )
    {
        sprintf( StringText, "%lu Hz", DEMO_CENTRAL_FREQ_PRESET8 );
    }
    else
    {
        sprintf( StringText, "%lu Hz", DEMO_CENTRAL_FREQ_PRESET9 );
    }
    return StringText;
}

char* GetRadioTxPower( void )
{
    sprintf( StringText2, "%d dBm", Eeprom.EepromData.DemoSettings.TxPower );
    return StringText2;
}

char* GetRadioPayloadLength( void )
{
    if( Eeprom.EepromData.DemoSettings.ModulationType == PACKET_TYPE_LORA )
    {
        sprintf( StringText2, "%d", Eeprom.EepromData.DemoSettings.PacketParam3 );
    }
    else // PACKET_TYPE_GFSK
    {
        sprintf( StringText2, "%d", Eeprom.EepromData.DemoSettings.PacketParam6 );
    }
    return StringText2;
}

char* GetMenuDemoMaxNumPacket( void )
{
    if( Eeprom.EepromData.DemoSettings.MaxNumPacket == 0 )
    {
        return ( char* )"Infinite";
    }
    else
    {
        sprintf( StringText, "%6d", \
                 Eeprom.EepromData.DemoSettings.MaxNumPacket );
    }
    return StringText;
}

char* GetMenuDemoNumSentPacket( void )
{
    sprintf( StringText2, "%6d", Eeprom.EepromData.DemoSettings.CntPacketTx );
    return StringText2;
}

char* GetMenuDemoFreqError( void )
{
    if( Eeprom.EepromData.DemoSettings.ModulationType == PACKET_TYPE_LORA )
    {
        sprintf( StringText2, "%6d", Eeprom.EepromData.DemoSettings.FreqErrorEst );
    }
    else
    {
        sprintf( StringText2, "N/A" );
    }
    return StringText2;
}

char* GetMenuDemoRxOk( void )
{
    sprintf( StringText2, "%6lu", \
            ( unsigned long )( Eeprom.EepromData.DemoSettings.CntPacketRxOK ) );
    return StringText2;
}

char* GetMenuDemoRxKo( void )
{
    sprintf( StringText2, "%6lu", \
             ( unsigned long )( Eeprom.EepromData.DemoSettings.CntPacketRxKO + \
             Eeprom.EepromData.DemoSettings.RxTimeOutCount ) ); 
    return StringText2;
}

char* GetMenuDemoRxOkSlave( void )
{
    sprintf( StringText3, "%6lu", \
       ( unsigned long )( Eeprom.EepromData.DemoSettings.CntPacketRxOKSlave ) );
    return StringText3;
}

char* GetMenuDemoResultPerCent1( uint32_t value, uint32_t reference )
{
    // quicker than using a float (which is not optimized in MBED)
    sprintf( StringText2, "%3d.%02d", \
             ( ( value * 10000 ) / reference ) / 100, \
             ( ( value * 10000 ) / reference ) % 100 );
    return StringText2;
}

char* GetMenuDemoResultPerCent2( uint32_t value, uint32_t reference )
{
    // quicker than using a float (which is not optimized in MBED)
    sprintf( StringText3, "%3d.%02d", \
             ( ( value * 10000 ) / reference ) / 100, \
             ( ( value * 10000 ) / reference ) % 100 );
    return StringText3;
}

char* GetMenuDemoRxKoSlave( void )
{
    sprintf( StringText3, "%6lu", \
       ( unsigned long )( Eeprom.EepromData.DemoSettings.CntPacketRxKOSlave ) );
    return StringText3;
}

char* GetMenuDemoRssi( void )
{
    sprintf( StringText2, "%6d", Eeprom.EepromData.DemoSettings.RssiValue );
    return StringText2;
}

char* GetMenuDemoSnr( void )
{
    if( Eeprom.EepromData.DemoSettings.SnrValue >= 0 )
    {
        sprintf( StringText2, "     /" ); 
    }
    else
    {
        sprintf( StringText2, "%6d", Eeprom.EepromData.DemoSettings.SnrValue );
    }
    return StringText2;
}

char* GetTotalPackets( void )
{
    if( Eeprom.EepromData.DemoSettings.MaxNumPacket == 0 )
    {
        return ( char* )"Total:  Inf.";
    }
    else
    {
        sprintf( StringText, "Total: %5lu", \
                 ( unsigned long )( Eeprom.EepromData.DemoSettings.MaxNumPacket ) );
        return StringText;
    }
}

char* GetMenuDemoRadioPowerMode( void )
{
    if( Eeprom.EepromData.DemoSettings.RadioPowerMode == USE_LDO )
    {
        return ( char* )"  LDO";
    }
    else
    {
        return ( char* )"  DCDC";
    }
}

char* GetMenuDemoRadioBoostedRx( void )
{
    if( Eeprom.EepromData.DemoSettings.BoostedRx == 0x01 )
    {
        return ( char* )"  True";
    }
    else
    {
        return ( char* )"  False";
    }
}


void CheckScreen( void )
{
    CheckScreenFlag = true;
}

void DebouncedScreen (void )
{
    ScreenBeenDebounced = false;
    DebouncedScreenTouch.detach( );
}

void RadioSettingsCheckError( void )
{
    if( Eeprom.EepromData.DemoSettings.ModulationType == PACKET_TYPE_GFSK )
    {
        switch( Eeprom.EepromData.DemoSettings.ModulationParam4 )
        {
            case RX_BW_4800:
                ErrorParam = ( (Eeprom.EepromData.DemoSettings.ModulationParam2 * 2) + Eeprom.EepromData.DemoSettings.ModulationParam1  ) >= 4800;
                break;

            case RX_BW_5800:
                ErrorParam = ( (Eeprom.EepromData.DemoSettings.ModulationParam2 * 2) + Eeprom.EepromData.DemoSettings.ModulationParam1  ) >= 5800;
                break;

            case RX_BW_7300:
                ErrorParam = ( (Eeprom.EepromData.DemoSettings.ModulationParam2 * 2) + Eeprom.EepromData.DemoSettings.ModulationParam1  ) >= 7300;
                break;

            case RX_BW_9700:
                ErrorParam = ( (Eeprom.EepromData.DemoSettings.ModulationParam2 * 2) + Eeprom.EepromData.DemoSettings.ModulationParam1  ) >= 9700;
                break;

            case RX_BW_11700:
                ErrorParam = ( (Eeprom.EepromData.DemoSettings.ModulationParam2 * 2) + Eeprom.EepromData.DemoSettings.ModulationParam1  ) >= 11700;
                break;

            case RX_BW_14600:
                ErrorParam = ( (Eeprom.EepromData.DemoSettings.ModulationParam2 * 2) + Eeprom.EepromData.DemoSettings.ModulationParam1  ) >= 14600;
                break;

            case RX_BW_19500:
                ErrorParam = ( (Eeprom.EepromData.DemoSettings.ModulationParam2 * 2) + Eeprom.EepromData.DemoSettings.ModulationParam1  ) >= 19500;
                break;

            case RX_BW_23400:
                ErrorParam = ( (Eeprom.EepromData.DemoSettings.ModulationParam2 * 2) + Eeprom.EepromData.DemoSettings.ModulationParam1  ) >= 23400;
                break;

            case RX_BW_29300:
                ErrorParam = ( (Eeprom.EepromData.DemoSettings.ModulationParam2 * 2) + Eeprom.EepromData.DemoSettings.ModulationParam1  ) >= 29300;
                break;

            case RX_BW_39000:
                ErrorParam = ( (Eeprom.EepromData.DemoSettings.ModulationParam2 * 2) + Eeprom.EepromData.DemoSettings.ModulationParam1  ) >= 39000;
                break;

            case RX_BW_46900:
                ErrorParam = ( (Eeprom.EepromData.DemoSettings.ModulationParam2 * 2) + Eeprom.EepromData.DemoSettings.ModulationParam1  ) >= 46900;
                break;

            case RX_BW_58600:
                ErrorParam = ( (Eeprom.EepromData.DemoSettings.ModulationParam2 * 2) + Eeprom.EepromData.DemoSettings.ModulationParam1  ) >= 58600;
                break;

            case RX_BW_78200:
                ErrorParam = ( (Eeprom.EepromData.DemoSettings.ModulationParam2 * 2) + Eeprom.EepromData.DemoSettings.ModulationParam1  ) >= 78200;
                break;

            case RX_BW_93800:
                ErrorParam = ( (Eeprom.EepromData.DemoSettings.ModulationParam2 * 2) + Eeprom.EepromData.DemoSettings.ModulationParam1  ) >= 93800;
                break;

            case RX_BW_117300:
                ErrorParam = ( (Eeprom.EepromData.DemoSettings.ModulationParam2 * 2) + Eeprom.EepromData.DemoSettings.ModulationParam1  ) >= 117300;
                break;

            case RX_BW_156200:
                ErrorParam = ( (Eeprom.EepromData.DemoSettings.ModulationParam2 * 2) + Eeprom.EepromData.DemoSettings.ModulationParam1  ) >= 156200;
                break;

            case RX_BW_187200:
                ErrorParam = ( (Eeprom.EepromData.DemoSettings.ModulationParam2 * 2) + Eeprom.EepromData.DemoSettings.ModulationParam1  ) >= 187200;
                break;

            case RX_BW_234300:
                ErrorParam = ( (Eeprom.EepromData.DemoSettings.ModulationParam2 * 2) + Eeprom.EepromData.DemoSettings.ModulationParam1  ) >= 234300;
                break;

            case RX_BW_312000:
                ErrorParam = ( (Eeprom.EepromData.DemoSettings.ModulationParam2 * 2) + Eeprom.EepromData.DemoSettings.ModulationParam1  ) >= 312000;
                break;

            case RX_BW_373600:
                ErrorParam = ( (Eeprom.EepromData.DemoSettings.ModulationParam2 * 2) + Eeprom.EepromData.DemoSettings.ModulationParam1  ) >= 373600;
                break;

            case RX_BW_467000:
                ErrorParam = ( (Eeprom.EepromData.DemoSettings.ModulationParam2 * 2) + Eeprom.EepromData.DemoSettings.ModulationParam1  ) >= 467000;
                break;

            default:
                ErrorParam = true;
                break;
        }
        if( ErrorParam == true )
        {
            MenuObjects[BTN3_VALUE_TEXT].FrontColor = OBJECT_ERROR;
            MenuObjects[BTN4_VALUE_TEXT].FrontColor = OBJECT_ERROR;
            MenuObjects[BTN6_VALUE_TEXT].FrontColor = OBJECT_ERROR;
        }
        else
        {
            RadioSettingsClearError( );
        }
        ButtonChangeText( BTN3_BORDER, NULL, ( uint8_t* )GetRadioModulationParameters1( ) );
        ButtonChangeText( BTN4_BORDER, NULL, ( uint8_t* )GetRadioModulationParameters2( ) );
        ButtonChangeText( BTN6_BORDER, NULL, ( uint8_t* )GetRadioModulationParameters4( ) );
    }
}

void RadioSettingsClearError( void )
{
    MenuObjects[BTN3_VALUE_TEXT].FrontColor = TEXT_VALUE;
    MenuObjects[BTN4_VALUE_TEXT].FrontColor = TEXT_VALUE;
    MenuObjects[BTN6_VALUE_TEXT].FrontColor = TEXT_VALUE;
}

/*!
 * \brief Draw button
 *
 * \param [in]  buttonId      Id of the button to draw
 * \param [in]  *text1        Text to draw as label
 * \param [in]  *text2        Text to draw as value
 */
void DrawActiveButton( uint8_t buttonId, uint8_t* text1, uint8_t* text2 )
{
    GraphObjectDraw( &( MenuObjects[buttonId] ), NULL, NULL, true );
    if( text2 == NULL )
    {
        if( text1 != NULL )
        {
            GraphObjectDraw( &( MenuObjects[buttonId + 1] ), text1, NULL, false );
        }
    }
    else
    {
        if( text1 != NULL )
        {
            GraphObjectDraw( &( MenuObjects[buttonId + 2] ), text1, NULL, false );
        }
        GraphObjectDraw( &( MenuObjects[buttonId + 3] ), text2, NULL, false );
    }
}

void ButtonChangeText( uint8_t buttonId, uint8_t* text1, uint8_t* text2 )
{
    if( text2 == NULL )
    {
        if( text1 != NULL )
        {
            GraphObjectDraw( &( MenuObjects[buttonId + 1] ), text1, NULL, false );
        }
    }
    else
    {
        if( text1 != NULL )
        {
            GraphObjectDraw( &( MenuObjects[buttonId + 2] ), text1, NULL, false );
        }
        GraphObjectDraw( &( MenuObjects[buttonId + 3] ), text2, NULL, false );
    }
}

void DrawText( uint8_t lineId, uint8_t* text0, uint8_t* text1, uint8_t* text2 )
{
    if( text0 != NULL )
    {
        GraphObjectDraw( &( MenuObjects[lineId] ), text0, NULL, false );
    }
    if( text1 != NULL )
    {
        GraphObjectDraw( &( MenuObjects[lineId + 1] ), text1, NULL, false );
    }
    if( text2 != NULL )
    {
        GraphObjectDraw( &( MenuObjects[lineId + 2] ), text2, NULL, false );
    }
}

