/*
  ______                              _
 / _____)             _              | |
( (____  _____ ____ _| |_ _____  ____| |__
 \____ \| ___ |    (_   _) ___ |/ ___)  _ \
 _____) ) ____| | | || |_| ____( (___| | | |
(______/|_____)_|_|_| \__)_____)\____)_| |_|
    (C)2016 Semtech

Description: EEPROM routines header

Maintainer: Gregory Cristian & Gilbert Menth
*/

#ifndef EEPROM_H
#define EEPROM_H


#include "Menu.h"
#include "DemoApplication.h"
#include "sx126x.h"


/*!
 * \brief Define Buffer indexes for each EepromData for copy and restore
 */
#define SCR_CAL_FLAG_EEPROM_ADDR            0   // boolean
#define SCR_CAL_POSA_EEPROM_ADDR            1   // int32
#define SCR_CAL_POSB_EEPROM_ADDR            5   // int32
#define SCR_CAL_POSC_EEPROM_ADDR            9   // int32
#define SCR_CAL_POSD_EEPROM_ADDR            13  // int32
#define SCR_CAL_POSE_EEPROM_ADDR            17  // int32
#define SCR_CAL_POSF_EEPROM_ADDR            21  // int32
#define APP_ENTITY_EEPROM_ADDR              25  // uint8
#define APP_RADIO_BOOSTED_RX_EEPROM_ADDR    26  // uint8
#define APP_FREQ_EEPROM_ADDR                27  // uint32_t Frequency;
#define APP_TXPWR_EEPROM_ADDR               31  // int8_t TxPower;
#define APP_MOD_TYPE_EEPROM_ADDR            32  // uint8_t ModulationType;
#define APP_PER_NPAK_MAX_EEPROM_ADDR        33  // uint32_t MaxNumPacket;
#define APP_RADIO_POWER_MODE_EEPROM_ADDR    37  // RadioRegulatorMode_t
#define MOD_PKET_TYPE_EEPROM_ADDR           38  // uint8_t
#define MOD_GFSK_BR_EEPROM_ADDR             39  // uint32_t Bitrate
#define MOD_GFSK_FDEV_EEPROM_ADDR           43  // uint32_t FrequencyDeviation
#define MOD_GFSK_MOD_SHAPE_EEPROM_ADDR      47  // uint8_t ModulationShape
#define MOD_GFSK_BW_EEPROM_ADDR             48  // uint8_t Bandwidth
#define RFU_1                               49
#define RFU_2                               53
#define RFU_3                               57
#define RFU_4                               58
#define RFU_5                               59
#define RFU_6                               63
#define RFU_7                               67
#define RFU_8                               68
#define MOD_LORA_SPREADF_EEPROM_ADDR        69  // uint8_t SpreadingFactor
#define MOD_LORA_BW_EEPROM_ADDR             70  // uint8_t Bandwidth
#define MOD_LORA_CODERATE_EEPROM_ADDR       71  // uint8_t CodingRate
#define PAK_PKET_TYPE_EEPROM_ADDR           72  // uint8_t

#define PAK_GFSK_PREAMBLE_LEN_EEPROM_ADDR   73  // uint16_t PreambleTxLength
#define PAK_GFSK_PR_MIN_DET_EEPROM_ADDR     75  // uint8_t PreambleMinDetect
#define PAK_GFSK_SYNC_LEN_EEPROM_ADDR       76  // uint8_t SyncWordLength
#define PAK_GFSK_ADDR_COMP_EEPROM_ADDR      77  // uint8_t AddrComp
#define PAK_GFSK_HEADERTYPE_EEPROM_ADDR     78  // uint8_t HeaderType
#define PAK_GFSK_PL_LEN_EEPROM_ADDR         79  // uint8_t PayloadLength
#define PAK_GFSK_CRC_LEN_EEPROM_ADDR        80  // uint8_t CrcLength
#define PAK_GFSK_DC_FREE_EEPROM_ADDR        81  // uint8_t DCFree

#define RFU_9                               82
#define RFU_10                              84
#define RFU_11                              85
#define RFU_12                              86
#define RFU_13                              87
#define RFU_14                              88
#define RFU_15                              89
#define RFU_16                              90
                                              
#define RFU_17                              91
#define RFU_18                              93
#define RFU_19                              94
#define RFU_20                              95
#define RFU_21                              96
#define RFU_22                              97
#define RFU_23                              98
#define RFU_24                              99

#define PAK_LORA_PREAMBLE_LEN_EEPROM_ADDR   100 // PreambleLength
#define PAK_LORA_HEADERTYPE_EEPROM_ADDR     102 // HeaderType
#define PAK_LORA_PL_LEN_EEPROM_ADDR         103 // PayloadLength
#define PAK_LORA_CRC_MODE_EEPROM_ADDR       104 // CrcMode
#define PAK_LORA_IQ_INV_EEPROM_ADDR         105 // InvertIQ
#define EEPROM_LAST_DEVICE_CONNECTED        106 // uint8
#define EEPROM_CRC_EEPROM_ADDR              107 // uint16

/*!
 * \brief Eeprom buffer size. Cf. above.
 */
#define EEPROM_BUFFER_SIZE                  109

/*!
 * \brief Part of EEPROM to save or restore
 */
typedef enum
{
    ALL_DATA,
    SCREEN_DATA,
    DEMO_SETTINGS,
    RADIO_GFSK_PARAMS,
    RADIO_LORA_PARAMS,
}EepromDataSet_t;

/*!
 * \brief EepromData structure
 */
typedef struct
{
    MenuSettings_t MenuSettings;
    DemoSettings_t DemoSettings;
    ModulationParams_t ModulationParams;
    PacketParams_t PacketParams;
    uint16_t CheckSum;
}EepromData_t;

/*!
 * \brief Eeprom structure
 */
typedef struct
{
    EepromData_t EepromData;
    // Allows for the checksum to be carried out
    uint8_t Buffer[EEPROM_BUFFER_SIZE];
}Eeprom_t;

/*!
 * \brief Local copy of Eeprom. (defined in Eeprom.cpp)
 */
extern Eeprom_t Eeprom;

/*!
 * \brief Initialises the contents of EepromData
 * 
 * \param [in]  deviceConnected  Device currently connected
 * \param [in]  matchingFreq     matching frequency
 */
void EepromInit( uint8_t deviceConnected, uint8_t matchingFreq );

/*!
 * \brief Read Eeprom from emulated EEPROM (in fact in Flash " higher address).
 *
 * \param [in]  addr          address of data (EEPROM offset not to be include)
 * \param [in]  buffer        buffer to use for copy
 * \param [in]  size          size of data to copy
 *
 * \retval      status        Status of operation (SUCCESS, ..)
 */
uint8_t EepromMcuReadBuffer( uint16_t addr, uint8_t *buffer, uint16_t size );

/*!
 * \brief Writes the EepromData to emulated EEPROM
 *
 * \param [in]  dataSet       Set of data to save or restore
 */
void EepromSaveSettings( EepromDataSet_t dataSet);

/*!
 * \brief Loads EepromData from emulated EEPROM
 */
void EepromLoadGeneralSettings ( void );

/*!
 * \brief Loads EepromData with updated modulation and packet parameters
 *
 * \param [in]  modulation    modulation type to select for mod. & packet params
 */
void EepromLoadSettings( RadioPacketTypes_t modulation );

/*!
 * \brief Initialises the contents of flash to default values & save to EEPROM
 *
 * \param [in]  deviceConnected    Device currently connected
 * \param [in]  matchingFreq       board matching frequency
 */
void EepromSetDefaultSettings( uint8_t deviceConnected, uint8_t matchingFreq );

/*!
 * \brief Erase the data checksum to reset the radio to default parameters
 *
 */
void EepromEraseCheckSum( void );

#endif //EEPROM_H
