/** Description:
 *  This File is responsible for defining the buffers that will be used in
 *  various areas of the program. The buffers that are defined here will be 
 *  refferenced in the main.cpp and will be filled with data in one of two ways:
 *  
 *  1.  The buffers will be filled with data upon receiving a successful payload
 *      transmission over LoRa. In GenericPingPong.cpp, the function OnRxDone() 
 *      will call the function fillGlobalBufsWithPayload() that is responsible for 
 *      filling the appropriate buffers with the payload data that was received
 *      by either the master or slave device by the other device that sent it.
 *      
 *  2.  The buffers will be used to fill up a payload buffer that will be sent
 *      over LoRa by either the master or slave device to the receiving device.
 *      The function fillPayloadWithGlobalBufs() will use the global buffers that
 *      are filled with data for various sensors in the main.cpp to construct a
 *      payload to deliver to the other device when SX1276PingPong() is called 
 *      in file GenericPingPong.cpp. 
 *     
 */

#ifndef __GLOBAL_BUFFERS_H__
#define __GLOBAL_BUFFERS_H__\
 
#include "mbed.h"
//#include "GenericPingPong2.h"
//#include "support.h"

/***************************************************************************
 * MASTER Device
 **************************************************************************/\
//#define MASTER 1

/***************************************************************************
 * SLAVE Device
 **************************************************************************/\
#define SLAVE 1


/***************************************************************************
 * Indexes for which byte specific data begins at in the payload buffer
 **************************************************************************/
/* size of ID data that defines what the signature of the device is */
const uint8_t size_signature = 1;

/* size of data in bytes that is acquired by the master device */
//const uint8_t size_of_ble       = 2;
//const uint8_t size_of_dum       = 2;

/* size of data in bytes that is acquired by the slave device */
//const uint8_t size_of_grid_eye  = 128; 
//const uint8_t size_of_gps       = 20;
//const uint8_t size_of_MAX17055  = 22;
//const uint8_t size_of_MAX77650  = 5;

const uint8_t size_of_MAX30208  = 2;            // temperature data
const uint8_t size_of_MAX44009  = 2;            // light data
const uint8_t size_of_MAX20361  = 4;            // AO32 data
const uint8_t size_of_other  = 0;               // other data


/* These are the sizes of each payload in bytes. Since there is different amount 
 * of data being sent in each direction, we need to declare the total size of 
 * the payload so we can instatiate the correct buffer sizes to store data that 
 * is to be delivered and for data that is received. 
 */

const uint16_t PAYLOAD_BUFFER_SIZE_MASTER_TO_SLAVE = size_signature;
//const uint16_t PAYLOAD_BUFFER_SIZE_MASTER_TO_SLAVE = 0;
const uint16_t PAYLOAD_BUFFER_SIZE_SLAVE_TO_MASTER = size_signature + size_of_MAX30208 + size_of_MAX44009 + size_of_MAX20361;

/* determine the appropriate buffer sizes */
//#if   defined(TARGET_MAX32630FTHR) // Master Device: Bluetooth Gateway
//    const uint16_t BufferSizeTx = PAYLOAD_BUFFER_SIZE_MASTER_TO_SLAVE;
//    const uint16_t BufferSizeRx = PAYLOAD_BUFFER_SIZE_SLAVE_TO_MASTER;
//#elif defined(TARGET_MAX32620FTHR) // Slave Device: Robot
//    const uint16_t BufferSizeTx = PAYLOAD_BUFFER_SIZE_SLAVE_TO_MASTER;
//    const uint16_t BufferSizeRx = PAYLOAD_BUFFER_SIZE_MASTER_TO_SLAVE;
//#endif

/* determine the appropriate buffer sizes */
#if MASTER == 1     // Master Device 
    const uint16_t BufferSizeTx = PAYLOAD_BUFFER_SIZE_MASTER_TO_SLAVE;
    const uint16_t BufferSizeRx = PAYLOAD_BUFFER_SIZE_SLAVE_TO_MASTER;
#elif SLAVE == 1    // Slave Device
    const uint16_t BufferSizeTx = PAYLOAD_BUFFER_SIZE_SLAVE_TO_MASTER;
    const uint16_t BufferSizeRx = PAYLOAD_BUFFER_SIZE_MASTER_TO_SLAVE;
#endif    


//#if   defined(TARGET_MAX32630FTHR) // Master Device: BLE-to-LoRa Gateway
//    /* These are indexs used to create the payload buffer to send to Slave */  
//    const uint8_t tx_idx_signature = 0;                                   // 1st buf in tx payload (begins at byte 0)
//    const uint8_t tx_idx_ble       = tx_idx_signature + size_signature;   // 2nd buf in tx payload
//    
//    /* These are indexs used to deconstruct received payload buffer sent by the Slave */
//    const uint8_t rx_idx_signature = 0;                                   // 1st buf in rx payload (begins at byte 0)
//    const uint8_t rx_idx_grid_eye  = rx_idx_signature + size_signature;   // 2nd buf in rx payload
//    const uint8_t rx_idx_gps       = rx_idx_grid_eye  + size_of_grid_eye; // 3rd buf in rx payload
//    const uint8_t rx_idx_MAX17055  = rx_idx_gps       + size_of_gps;      // 4th buf in rx payload
//    const uint8_t rx_idx_MAX77650  = rx_idx_MAX17055  + size_of_MAX17055; // 5th buf in rx payload
//        
//#elif defined(TARGET_MAX32620FTHR) // Client Device: LoRa Controlled Robot
//    /* These are indexs used to create the payload buffer to send to Master */
//    const uint8_t tx_idx_signature = 0;                                   // 1st buf in tx payload (begins at byte 0)
//    const uint8_t tx_idx_grid_eye  = tx_idx_signature + size_signature;   // 2nd buf in tx payload
//    const uint8_t tx_idx_gps       = tx_idx_grid_eye  + size_of_grid_eye; // 3rd buf in tx payload
//    const uint8_t tx_idx_MAX17055  = tx_idx_gps       + size_of_gps;      // 4th buf in tx payload
//    const uint8_t tx_idx_MAX77650  = tx_idx_MAX17055  + size_of_MAX17055; // 5th buf in tx payload
//    
//    /* These are indexs used to deconstruct received payload buffer sent by the Master */
//    const uint8_t rx_idx_signature = 0;                                   // 1st buf in rx payload (begins at byte 0)
//    const uint8_t rx_idx_ble       = rx_idx_signature + size_signature;   // 2nd buf in rx payload
//#endif

#if   MASTER == 1 // Master Device
    /* These are indexs used to create the payload buffer to send to Slave */  
    const uint8_t tx_idx_signature = 0;                                   // 1st buf in tx payload (begins at byte 0)
//    const uint8_t tx_idx_dum       = tx_idx_signature + size_signature;   // 2nd buf in tx payload
    
    /* These are indexs used to deconstruct received payload buffer sent by the Slave */
    const uint8_t rx_idx_signature = 0;                                   // 1st buf in rx payload (begins at byte 0)
    const uint8_t rx_idx_MAX30208  = rx_idx_signature + size_signature;   // 2nd buf in rx payload
    const uint8_t rx_idx_MAX44009  = rx_idx_MAX30208  + size_of_MAX30208; // 3rd buf in rx payload
    const uint8_t rx_idx_MAX20361  = rx_idx_MAX44009  + size_of_MAX44009;      // 4th buf in rx payload
//    const uint8_t rx_idx_other  = rx_idx_MAX20361  + size_of_MAX20361; // 5th buf in rx payload
        
#elif SLAVE == 1 // Slave Device: LoRa Controlled Robot
    /* These are indexs used to create the payload buffer to send to Master */
    const uint8_t tx_idx_signature = 0;                                   // 1st buf in tx payload (begins at byte 0)
    const uint8_t tx_idx_MAX30208  = tx_idx_signature + size_signature;   // 2nd buf in tx payload
    const uint8_t tx_idx_MAX44009  = tx_idx_MAX30208  + size_of_MAX30208; // 3rd buf in tx payload
    const uint8_t tx_idx_MAX20361  = tx_idx_MAX44009  + size_of_MAX44009 ;      // 4th buf in tx payload
//    const uint8_t tx_idx_other  = tx_idx_MAX20361  + size_of_MAX20361; // 5th buf in tx payload
    
    /* These are indexs used to deconstruct received payload buffer sent by the Master */
    const uint8_t rx_idx_signature = 0;                                   // 1st buf in rx payload (begins at byte 0)
//    const uint8_t rx_idx_dum       = rx_idx_signature + size_signature;   // 2nd buf in rx payload
#endif


/***************************************************************************
 * BLE Data Buffers
 **************************************************************************/
//#if   defined(TARGET_MAX32630FTHR) // Master Device: BLE-to-LoRa Gateway
//    static char curr_ble_data_to_slave[size_of_ble];
//#elif defined(TARGET_MAX32620FTHR) // Slave Device: LoRa Controlled Robot
//    static char curr_ble_data_from_master[size_of_ble];
//    static char prev_ble_data_from_master[size_of_ble];
//#endif

/***************************************************************************
 * Dummy Data Buffers
 **************************************************************************/
#if MASTER == 1  // Master Device
//    static char curr_dum_data_to_slave[size_of_dum];
#elif SLAVE == 1 // Slave Device
//    static char curr_dum_data_from_master[size_of_dum];
//    static char prev_dum_data_from_master[size_of_dum];
#endif


/***************************************************************************
 * Grid Eye Sensor Data Buffers
 **************************************************************************/
//#if   defined(TARGET_MAX32630FTHR) // Master Device: BLE-to-LoRa Gateway
//    static char curr_raw_frame_data_from_slave[size_of_grid_eye];
//    static char prev_raw_frame_data_from_slave[size_of_grid_eye];
//    static int16_t conv_frame_data_from_slave[64];
//#elif defined(TARGET_MAX32620FTHR) // Client Device: LoRa Controlled Robot
//    static char curr_raw_frame_data_to_master[size_of_grid_eye];
//    static char prev_raw_frame_data_to_master[size_of_grid_eye];
//    static int16_t conv_frame_data_to_master[64];
//#endif


/***************************************************************************
 * MAX30208 Data Buffers
 **************************************************************************/
#if   MASTER == 1 // Master Device
    static char curr_raw_temp_data_from_slave[size_of_MAX30208];
    static char prev_raw_temp_data_from_slave[size_of_MAX30208];
//    static int16_t conv_frame_data_from_slave[64];
#elif SLAVE == 1 // Slave Device
    static char curr_raw_temp_data_to_master[size_of_MAX30208];
//    static char prev_raw_temp_data_to_master[size_of_MAX30208];
//    static int16_t conv_frame_data_to_master[64];
#endif

/***************************************************************************
 * MAX44009 Data Buffers
 **************************************************************************/
#if   MASTER == 1 // Master Device
    static char curr_raw_light_data_from_slave[size_of_MAX44009];
    static char prev_raw_light_data_from_slave[size_of_MAX44009];
//    static int16_t conv_frame_data_from_slave[64];
#elif SLAVE == 1 // Slave Device
    static char curr_raw_light_data_to_master[size_of_MAX44009];
//    static char prev_raw_light_data_to_master[size_of_MAX44009];
//    static int16_t conv_frame_data_to_master[64];
#endif

/***************************************************************************
 * MAX20361 Data Buffers
 **************************************************************************/
#if   MASTER == 1 // Master Device
    static char curr_raw_AO32_data_from_slave[size_of_MAX20361];
    static char prev_raw_AO32_data_from_slave[size_of_MAX20361];
//    static int16_t conv_frame_data_from_slave[64];
#elif SLAVE == 1 // Slave Device
    static char curr_raw_AO32_data_to_master[size_of_MAX20361];
//    static char prev_raw_AO32_data_to_master[size_of_MAX20361];
//    static int16_t conv_frame_data_to_master[64];
#endif

/***************************************************************************
 * Other Data Buffers
 **************************************************************************/
#if   MASTER == 1 // Master Device
//    static char curr_raw_other_data_from_slave[size_of_other];
//    static char prev_raw_other_data_from_slave[size_of_other];
//    static int16_t conv_frame_data_from_slave[64];
#elif SLAVE == 1 // Slave Device
//    static char curr_raw_other_data_to_master[size_of_other];
//    static char prev_raw_other_data_to_master[size_of_other];
//    static int16_t conv_frame_data_to_master[64];
#endif

/***************************************************************************
 * GPS Data Buffers
 **************************************************************************/
//#if   defined(TARGET_MAX32630FTHR) // Master Device: BLE-to-LoRa Gateway
//    static char curr_gps_data_from_slave[size_of_gps];
//    static char prev_gps_data_from_slave[size_of_gps];
//#elif defined(TARGET_MAX32620FTHR) // Slave Device: LoRa Controlled Robot
//    static char curr_gps_data_to_master[size_of_gps];
//#endif

/***************************************************************************
 * MAX17055 Data Buffers
 **************************************************************************/
//#if   defined(TARGET_MAX32630FTHR) // Master Device: BLE-to-LoRa Gateway
//    static char curr_MAX17055_from_slave[size_of_MAX17055];
//    static char prev_MAX17055_from_slave[size_of_MAX17055];
//#elif defined(TARGET_MAX32620FTHR) // Slave Device: LoRa Controlled Robot
//    static char curr_MAX17055_to_master[size_of_MAX17055];
//#endif

/***************************************************************************
 * MAX77650 Data Buffers
 **************************************************************************/
//#if   defined(TARGET_MAX32630FTHR) // Master Device: BLE-to-LoRa Gateway
//    static char curr_MAX77650_from_slave[size_of_MAX77650];
//    static char prev_MAX77650_from_slave[size_of_MAX77650];
//#elif defined(TARGET_MAX32620FTHR) // Slave Device: LoRa Controlled Robot
//    static char curr_MAX77650_to_master[size_of_MAX77650];
//#endif

/** 
 * @brief This function constructs a payload buffer that is used in transmitting data in a LoRa Message
 */
void fillPayloadWithGlobalBufs(uint8_t * payload_buffer_tx); 

/** 
 * @brief This function deconstructs a payload buffer that contains data from a received LoRa Message
 */
void fillGlobalBufsWithPayload(uint8_t * payload_buffer_rx); 

#endif // __GLOBAL_BUFFERS_H__