#include "mbed.h"

//AO32 Registers
#define AO32_DEVICE_ID      0x00    // AO32 Chip ID
#define AO32_STATUS         0x01    // AO32 Status Register
#define AO32_INT            0x02    // AO32 Interrupt Register
#define AO32_INT_MSK        0x03    // AO32 Interrupt Mask Register
#define AO32_SYS_CFG        0x04    // AO32 SYS configuration
#define AO32_WK_CFG         0x05    // AO32 Wake configuration
#define AO32_MPPT_CFG       0x06    // AO32 Maxim Power Point Tracking configuration
#define AO32_MEAS_CFG       0x07    // AO32 Measurement configuration
#define AO32_DEV_CTRL       0x08    // AO32 Device Control
#define AO32_VOC            0x09    // AO32 VOC Measurement
#define AO32_HARV_H         0x0A    // AO32 Harvesting count High byte
#define AO32_HARV_L         0x0B    // AO32 Harvesting count Low byte
#define AO32_SLP            0x0C    // AO32 Sleep threshold
   

#define DEVICE_ACK      0
#define DEVICE_NACK     1
#define DEVICE_BAD_RESP 2

#define MAX_DEVICES 64      // Maximum number of rom devices allowed
#define ID_LENGTH   6       // Rom ID length in bytes

struct AO32_struct {
    char rom_id[ID_LENGTH];     // device ROM ID
    char I2C_address;           // I2C addess, based on GPIO0 and GPIO1 at power up
                                // Why char?
}; 

struct VOCMeas {
    double VOCvalue;  // Open-Circuit Voltage Measurement in Volt
    int status;       // Status of OCV read. 0 for success, 1 for error
};

struct HarvCnt {
    double harvCount;   // Harvester Count "LX Pulses" in decimal
    int status;       // Status of harvesting count read. 0 for success, 1 for error
};

// *****************************************************************************
//   AO32_write_register(char, char, char)  writes single byte to AO32
//                       char   I2C address
//                       char   AO32 register address
//                       char   data byte to be writen
//   returns                    0 on success ACK, 1 on NACK 
// *****************************************************************************
int AO32_write_register(I2C *i2c, char I2C_add, char reg_add, char byte);

/// ****************************************************************************
//   AO32_write_register(char, char, char *, int)  writes multiple bytes to AO32
//                       char   I2C address
//                       char   AO32 register address
//                       char * data vector of bytes to be written
//                       int    number of bytes to write
//   returns                    0 on success ACK, 1 on NACK 
// *****************************************************************************
int AO32_write_register(I2C *i2c, char I2C_add, char reg_add, char *bytes, int n);

// *****************************************************************************
//   AO32_read_register(char, char, char *)  reads single byte from AO32
//                       char   I2C address
//                       char   AO32 register address
//                       char * data vector for read bytes to be stored in 
//   returns                    0 on success, 1 on fail 
// *****************************************************************************
int AO32_read_register(I2C *i2c, char I2C_add, char reg_add, char *bytes);

// *****************************************************************************
//   AO32_read_register(char, char, char *, int)  reads byte(s) from AO32
//                       char   I2C address
//                       char   OT07 register address
//                       char * data vector for read bytes to be stored in 
//                       int    number of bytes to read
//   returns                    0 on success, 1 on fail 
// *****************************************************************************
int AO32_read_register(I2C *i2c, char I2C_add, char reg_add, char *bytes, int n);

//******************************************************************************
// get_luxvalue(char)       read lux value from AO32 device register
//                 char     I2C address
// returns                  LuxResponse luxValue = light intensity in lux 
//                          status = register read result
//******************************************************************************
//LuxResponse get_luxvalue(I2C *i2c, char I2C_add);

//******************************************************************************
// get_OCV(char)       read Open Circuit Voltage from AO32 device register
//                 char     I2C address
// returns                  VOCMeas VOCvalue = Open-Circuit Voltage Measurement in Volt
//                          status = Status of OCV read. 0 for success, 1 for error
//******************************************************************************
VOCMeas get_OCV(I2C *i2c, char I2C_add);

//******************************************************************************
// get_Harvest(char)       read harvesting counts from AO32 device register
//                 char     I2C address
// returns                  HarvCnt harvCount = Harvester Count "LX Pulses" in decimal
//                          status = Status of harvesting count read. 0 for success, 1 for error
//******************************************************************************
HarvCnt get_Harvest(I2C *i2c, char I2C_add);

double calc_OCV(char *data);

int calc_Harvest(char *data);