
#include "mbed.h"
#include "AO32_lib.h"

// *****************************************************************************
//   AO32_write_register(char, char, char)  writes single byte to AO32
//                       char   I2C address
//                       char   AO32 register address
//                       char   data byte to be writen
//   returns                    0 on success ACK, 1 on NACK 
// *****************************************************************************

int AO32_write_register(I2C *i2c, char I2C_add, char reg_add, char byte){
    char data[2];               // char type ranges from 0 to 255 (8 bytes)
    int error;
    data[0] = reg_add;
    data[1] = byte;
    error = i2c->write(I2C_add,data,2);  
    //if(DEBUG)db.printf("wr[%02X %02X %d]\r\n", data[0], data[1], error);
    return error; 
    
}

/// ****************************************************************************
//   AO32_write_register(char, char, char *, int)  writes multiple bytes to AO32
//                       char   I2C address
//                       char   AO32 register address
//                       char * data vector of bytes to be written
//                       int    number of bytes to write
//   returns                    0 on success ACK, 1 on NACK 
// *****************************************************************************

int AO32_write_register(I2C *i2c, char I2C_add, char reg_add, char *bytes, int n){
    int i;   
    //set start address
    char data[16];
    int error;                          
    data[0] = reg_add; 
    for(i=1;i<=n;i++){                   
       data[i] = bytes[i-1];
    }
    error = i2c->write(I2C_add,data,n+1);  // send n bytes of data
  
    return error;      
}

// *****************************************************************************
//   AO32_read_register(char, char, char *)  reads single byte from AO32
//                       char   I2C address
//                       char   AO32 register address
//                       char * data vector for read bytes to be stored in 
//   returns                    0 on success, 1 on fail 
// *****************************************************************************

int AO32_read_register(I2C *i2c, char I2C_add, char reg_add, char *bytes){
    int error;
    error = i2c->write(I2C_add,&reg_add,1,1); 
    if(error)return error;   
    error = i2c->read(I2C_add,bytes,1);  
    //if(DEBUG)db.printf("rr e[%d]\r\n",error);
    return error; 
}

// *****************************************************************************
//   AO32_read_register(char, char, char *, int)  reads byte(s) from AO32
//                       char   I2C address
//                       char   OT07 register address
//                       char * data vector for read bytes to be stored in 
//                       int    number of bytes to read
//   returns                    0 on success, 1 on fail 
// *****************************************************************************

int AO32_read_register(I2C *i2c, char I2C_add, char reg_add, char *bytes, int n){
    int error;
    error = i2c->write(I2C_add,&reg_add,1,1); 
    if(error)return error;   
    error = i2c->read(I2C_add,bytes,n);  
    //if(DEBUG)db.printf("rr e[%d]\r\n",error);
    return error; 
}

//******************************************************************************
// get_OCV(char)       read Open Circuit Voltage from AO32 device register
//                 char     I2C address
// returns                  VOCMeas VOCvalue = Open-Circuit Voltage Measurement in Volt
//                          status = Status of OCV read. 0 for success, 1 for error
//******************************************************************************
VOCMeas get_OCV(I2C *i2c, char I2C_add) {
    
    char data[2];           // only needs 1 byte
    double voltage;
    
    // Read lux value, 2 bytes 
    int error = AO32_read_register(i2c, I2C_add, AO32_VOC, data);
    
    // Calculate Open Circuit Voltage from data
    voltage = int(data[0]) / 100;
    
    VOCMeas resp;
    resp.VOCvalue = voltage;
    resp.status = error; // 1 for nack/error. 0 for ack/success
    return resp;  
}

//******************************************************************************
// get_Harvest(char)       read harvesting counts from AO32 device register
//                 char     I2C address
// returns                  HarvCnt harvCount = Harvester Count "LX Pulses" in decimal
//                          status = Status of harvesting count read. 0 for success, 1 for error
//******************************************************************************
HarvCnt get_Harvest(I2C *i2c, char I2C_add) {
    
    char data[2];
    int countHi;
    int countLo;
    int counts;
    
    // Read harvesting counts, 2 bytes 
    int error = AO32_read_register(i2c, I2C_add, AO32_HARV_H, data, 2);     // burst read AO32_HARV_H and AO32_HARV_L
    
    // Calculate harvesting counts from data
//    countHi = int(data[0] << 8);        // might cause trouble, * 256 instead?
    countHi = int(data[0]) * 256;
    countLo = int(data[1]);
    counts = countHi + countLo;
    
    HarvCnt resp;
    resp.harvCount = counts;
    resp.status = error; // 1 for nack/error. 0 for ack/success
    return resp;  
}

double calc_OCV(char *data) {
    
    // data is an array of size 2, only needs the first byte
    
    double voltage = (double)(data[0]) / 100;
    
    return voltage;
    
}

int calc_Harvest(char *data) {
    
    // data is an array of size 2
    
    int countHi = int(data[0]) * 256;
    int countLo = int(data[1]);
    int harvest_counts = countHi + countLo;
    
    return harvest_counts;
    
}