
#include "mbed.h"

#include "lp.h"

// Board pins related
#include "PinMap.h"

// LORA related
#include "global_buffers.h"
#include "GenericPingPong2.h"
#define FEATURE_LORA
#include "sx1276-mbed-hal.h"

#include "main.h"

// OT07 related
#include "OT07_lib.h"      
#define CONVERT_T_DELAY 30 

// MAX44009 related
#include "MAX44009_lib.h"    

// AO32 related
#include "AO32_lib.h" 

// AO19 related
#include "AO19_lib.h"

// Virtual COM related
#include "USBSerial.h"      // for virtual COM
#define BS          8       // ASCII Back Space
#define CR          13      // ASCII Carriage Return

// Virtual serial port over USB TODO NEW VID PID NEEDED!!
USBSerial pc(0x0B6A, 0x0042, 0x0001, false);

/***************************************************************************
 * LEDs Instantiation
 **************************************************************************/
DigitalOut myled(LED2);          // LED = LED2 green
DigitalOut rLED(LED1);          // red LED
DigitalOut bLED(LED3);          // blue LED
//DigitalOut *led3;

/***************************************************************************
 * I2C Bus Instantiation
 **************************************************************************/
I2C i2cBus0(P1_6, P1_7);            // I2C of MAX32625PICO


bool get_data_flag = false;         // used for data tramission frequency on the SENSOR side
bool print_data_flag = false;       // used for data display on the GATEWAY side

//Timer setup
Ticker timer_1;                     // timer for data tramission frequency on the SENSOR side
Ticker timer_M;                     // timer for data print out on the GATEWAY side

void onTimerInterrupt(){
    get_data_flag = true;
}

void onGatewayInterrupt(){
    print_data_flag = true;
}

int main() {

/***************************************************************************

 
 
    /* Setup begins here: */
    #if   MASTER == 1 // Master Device
        rLED = LED_ON;      // red LED on
        myled = LED_OFF;
        bLED = LED_OFF;
    #elif SLAVE == 1 // Slave Device
        rLED = LED_OFF;
        myled = LED_OFF;
        bLED = LED_ON;      // blue LED on
        
        sleep();            // put slave in sleep mode 
        
    #endif
    
     /***************************************************************************
     * Combined Payload Buffers for LoRa Communications
     **************************************************************************/
    uint8_t BufferTx[BufferSizeTx];             // more info in global_buffers.h
    uint8_t BufferRx[BufferSizeRx];             // more info in global_buffers.h
        
    /***************************************************************************
     * MAX30208 Data Buffers
     **************************************************************************/
    #if   MASTER == 1 // Master Device
        char curr_raw_temp_from_slave[size_of_MAX30208];   // to match data type
        char prev_raw_temp_from_slave[size_of_MAX30208];  
    #elif SLAVE == 1 // Slave Device
        uint8_t curr_raw_temp_to_master[size_of_MAX30208];
    #endif
    
    /***************************************************************************
     * MAX44009 Data Buffers
     **************************************************************************/
    #if   MASTER == 1 // Master Device
        char curr_raw_light_from_slave[size_of_MAX44009];    // to match data type
        char prev_raw_light_from_slave[size_of_MAX44009];
    #elif SLAVE == 1 // Slave Device
        uint8_t curr_raw_light_to_master[size_of_MAX44009];
    #endif
    
    /***************************************************************************
     * MAX20361 Data Buffers
     **************************************************************************/
    #if   MASTER == 1 // Master Device
        char curr_raw_AO32_from_slave[size_of_MAX20361];            // to match data type
        char prev_raw_AO32_from_slave[size_of_MAX20361];
    #elif SLAVE == 1 // Slave Device
        uint8_t curr_raw_AO32_to_master[size_of_MAX20361];
    #endif
    
    /***************************************************************************
     * Finish Setting up LoRa Radios: This passes in pointers to Buffers to send
     **************************************************************************/
    SX1276PingPongSetup(BufferTx, BufferRx, &pc);
    
//    Pointer Tutorials
//    https://www.tutorialspoint.com/cprogramming/c_pointers.htm
    
    /***************************************************************************
     * Loop Counter
     **************************************************************************/
    int loopCnt = 0;
    
    #if MASTER == 1
     //************* init ticker timer callbacks  ****************
    timer_M.attach(&onGatewayInterrupt, 3);        //Gateway data print out frequency
    
    #endif
    
    #if SLAVE == 1
    //************* init ticker timer callbacks  ****************
    timer_1.attach(&onTimerInterrupt, 3);        //Sensor data transmission frequency
    
    //************* OT07 Variables  **************** 
    char rawtempdata[2];          
    char OT07_i2c_add = 0xA0;
    
    //************* MAX44009 Variables  ****************
    char rawluxdata[2];          
    char MAX44009_i2c_add = 0x94;      // 0b1001 010x
    
    //************* AO32 Variables  ****************
    char rawOCVdata[1];                 // only needs 1
    char rawCntdata[2];
    char AO32_i2c_add = 0x2A;      // 0b0010 101x
    
     //************* AO19 Settings  ****************
    // Enable AO19
    char AO19_i2c_add = 0xD0;      // 
    char AO19_BB_CFG[2];              // store result of AO19's register 0x01 BBstCfg0
    int AO19_read = AO19_read_register(&i2cBus0, AO19_i2c_add, AO19_BB_CFG0, AO19_BB_CFG);
    
//    if (AO19_read) {
//        pc.printf("AO19 Initial Read fail!\r\n");
//    }
//    else {
//        pc.printf("AO19 Initial Read success!\r\n");
//    } 
    
    int AO19_en = AO19_write_register(&i2cBus0, AO19_i2c_add, AO19_BB_CFG0, AO19_BB_CFG[0]|0x80);
    
   // if (AO19_en) {
//        pc.printf("AO19 Enabled fail!\r\n");
//    }
//    else {
//        pc.printf("AO19 Enabledsuccess!\r\n");
//    } 
    
    
    #endif
    
    while (1) {
    
    #if SLAVE == 1          // only send sensor data
    
    // Slave sleeps before interrupt calls
    sleep();
    
    if(get_data_flag) {
        
        //reset the flag
        get_data_flag = false;
//        pc.printf("Timer interval reached!\r\n");
        
        /***************************************************************************
        * Temperature Sensor Data Measurement
        **************************************************************************/
        // obtain register hex values
        convert_temperature(&i2cBus0, OT07_i2c_add);  //send OW convert selected device
        wait_ms(CONVERT_T_DELAY);  //wait 20 ms for convert temperature to complete
        int temp_error = OT07_read_register(&i2cBus0, OT07_i2c_add, OT07_FIFO_DATA, rawtempdata, 2); 
        double tempFinal = calc_temperature(rawtempdata);
        
        //fill raw temp data into the array
        curr_raw_temp_to_master[0] = rawtempdata[0];
        curr_raw_temp_to_master[1] = rawtempdata[1];      
        
        /***************************************************************************
        * Light Intensity Sensor Data Measurement
        **************************************************************************/
        // obtain register hex values
        int lux_error = MAX44009_read_lux_register(&i2cBus0, MAX44009_i2c_add, MAX44009_LUX_HI, rawluxdata);
        int luxFinal = (int) (calc_lux(rawluxdata));
        
        //fill raw lux data into the array
        curr_raw_light_to_master[0] = rawluxdata[0];
        curr_raw_light_to_master[1] = rawluxdata[1];
        
        /***************************************************************************
        * Solar Harvester Data Measurement
        **************************************************************************/
        
        int ocv_error = AO32_read_register(&i2cBus0, AO32_i2c_add, AO32_VOC, rawOCVdata);
        int cnt_error = AO32_read_register(&i2cBus0, AO32_i2c_add, AO32_HARV_H, rawCntdata, 2); // burst read 2 bytes
        
        //calculate open circuit voltage from data
//        double voltage = (double)(rawOCVdata[0]) / 100;
        double OCVFinal = calc_OCV(rawOCVdata);
        
        //calculate harvesting counts from data
        int countFinal = calc_Harvest(rawCntdata);
        
        //fill raw AO32 data into the array
        curr_raw_AO32_to_master[0] = rawOCVdata[0];         // Fill OCV hex first
        curr_raw_AO32_to_master[1] = rawCntdata[0];         // Fill Harvesting count high byte
        curr_raw_AO32_to_master[2] = rawCntdata[1];         // Fill Harvesting count low byte
        
        // print out sensor data
         pc.printf("SENSOR: [%.3f] [%d] [%.2f] [%d] [0]\r\n", tempFinal, luxFinal, OCVFinal, countFinal);
        
        /***************************************************************************
        * Fill Payload Buffer With Data From Main Program Buffers for next LoRa Transmition
        **************************************************************************/
        memcpy(&BufferTx[tx_idx_MAX30208],  curr_raw_temp_to_master, size_of_MAX30208);
        memcpy(&BufferTx[tx_idx_MAX44009],  curr_raw_light_to_master, size_of_MAX44009);
        memcpy(&BufferTx[tx_idx_MAX20361],  curr_raw_AO32_to_master,  size_of_MAX20361);

        /***************************************************************************
        * LoRa Communication: Send Sensor Data
        **************************************************************************/
//        SX1276PingPong(); 
        SX1276SlaveSendData();
        loopCnt = loopCnt + 1;
    } // end of transmission frequency for slave
    #endif    
    
    
    #if MASTER == 1         // only Receive Sensor Data
    
     /***************************************************************************
     * LoRa Communication: Gateway Receive Sensor Data
     **************************************************************************/
//    SX1276PingPong();
    if(SX1276MasterCheckForNewData())
    {
    
    /***************************************************************************
     * Fill Main Program Buffers With Data From Received Payload Buffer
     **************************************************************************/
    memcpy(curr_raw_temp_from_slave, &BufferRx[rx_idx_MAX30208],  size_of_MAX30208);
    memcpy(curr_raw_light_from_slave, &BufferRx[rx_idx_MAX44009], size_of_MAX44009);
    memcpy(curr_raw_AO32_from_slave,  &BufferRx[rx_idx_MAX20361],  size_of_MAX20361);

    /***************************************************************************
    * Master Device: Print out Slave Data
    **************************************************************************/
    
    double tempResult = calc_temperature(curr_raw_temp_from_slave);
    int luxResult = (int) calc_lux(curr_raw_light_from_slave);
    
    char OCVrawHex[1];
    OCVrawHex[0] = curr_raw_AO32_from_slave[0];
    char CntrawHex[2];
    CntrawHex[0] = curr_raw_AO32_from_slave[1];
    CntrawHex[1] = curr_raw_AO32_from_slave[2];
    
    double OCVResult = calc_OCV(OCVrawHex);
    int CntResult = calc_Harvest(CntrawHex);
        
    //reset the flag
    print_data_flag = false;        // no need
    pc.printf("MSG: [%.3f] [%d] [%.2f] [%d] [0]\r\n", tempResult, luxResult, OCVResult, CntResult);
    
    }   // end of SX1276MasterCheckForNewData

    loopCnt = loopCnt + 1;
    
    #endif
        
    sleep();        // going back to sleep mode    
        
    } // end of while(1) loop
        
}  // end of main()