/** Description:
 *  This File is responsible for defining the buffers that will be used in
 *  various areas of the program. The buffers that are defined here will be 
 *  refferenced in the main.cpp and will be filled with data in one of two ways:
 *  
 *  1.  The buffers will be filled with data upon receiving a successful payload
 *      transmission over LoRa. In GenericPingPong.cpp, the function OnRxDone() 
 *      will call the function fillGlobalBufsWithPayload() that is responsible for 
 *      filling the appropriate buffers with the payload data that was received
 *      by either the master or slave device by the other device that sent it.
 *      
 *  2.  The buffers will be used to fill up a payload buffer that will be sent
 *      over LoRa by either the master or slave device to the receiving device.
 *      The function fillPayloadWithGlobalBufs() will use the global buffers that
 *      are filled with data for various sensors in the main.cpp to construct a
 *      payload to deliver to the other device when SX1276PingPong() is called 
 *      in file GenericPingPong.cpp. 
 *     
 */

#ifndef __GLOBAL_BUFFERS_H__
#define __GLOBAL_BUFFERS_H__\
 
#include "mbed.h"
//#include "GenericPingPong2.h"

/***************************************************************************
 * MASTER Device
 **************************************************************************/\
//#define MASTER 1

/***************************************************************************
 * SLAVE Device
 **************************************************************************/\
#define SLAVE 1


/***************************************************************************
 * Indexes for which byte specific data begins at in the payload buffer
 **************************************************************************/
/* size of ID data that defines what the signature of the device is */
//const uint8_t size_signature = 1;

const uint8_t size_of_MAX30208  = 2;            // temperature data
const uint8_t size_of_MAX44009  = 2;            // light data
const uint8_t size_of_MAX20361  = 3;            // AO32 data


/* These are the sizes of each payload in bytes. Since there is different amount 
 * of data being sent in each direction, we need to declare the total size of 
 * the payload so we can instatiate the correct buffer sizes to store data that 
 * is to be delivered and for data that is received. 
 */

//const uint16_t PAYLOAD_BUFFER_SIZE_MASTER_TO_SLAVE = 0;
const uint16_t PAYLOAD_BUFFER_SIZE_MASTER_TO_SLAVE = 1;             // can't be zero, otherwise internal error
const uint16_t PAYLOAD_BUFFER_SIZE_SLAVE_TO_MASTER = size_of_MAX30208 + size_of_MAX44009 + size_of_MAX20361;

/* determine the appropriate buffer sizes */
#if MASTER == 1     // Master Device 
    const uint16_t BufferSizeTx = PAYLOAD_BUFFER_SIZE_MASTER_TO_SLAVE;
    const uint16_t BufferSizeRx = PAYLOAD_BUFFER_SIZE_SLAVE_TO_MASTER;
#elif SLAVE == 1    // Slave Device
    const uint16_t BufferSizeTx = PAYLOAD_BUFFER_SIZE_SLAVE_TO_MASTER;
    const uint16_t BufferSizeRx = PAYLOAD_BUFFER_SIZE_MASTER_TO_SLAVE;
#endif    

#if   MASTER == 1 // Master Device: Gateway Device, only receiving data from Slave
    
    /* These are indexs used to deconstruct received payload buffer sent by the Slave */
    const uint8_t rx_idx_MAX30208 = 0;                                      // 1st buf in rx payload (begins at byte 0), temp data
    const uint8_t rx_idx_MAX44009 = rx_idx_MAX30208 + size_of_MAX30208;     // 2nd buf in rx payload, lux data
    const uint8_t rx_idx_MAX20361 = rx_idx_MAX44009 + size_of_MAX44009;     // 3rd buf in rx payload, AO32 data
        
#elif SLAVE == 1 // Slave Device: Sensor Node, only sending data to Master
    /* These are indexs used to create the payload buffer to send to Master */
    const uint8_t tx_idx_MAX30208  = 0;                                     // 1st buf in tx payload (begins at byte 0), temp data
    const uint8_t tx_idx_MAX44009  = tx_idx_MAX30208 + size_of_MAX30208;    // 2nd buf in tx payload, lux data
    const uint8_t tx_idx_MAX20361  = tx_idx_MAX44009 + size_of_MAX44009;    // 3rd buf in tx payload, AO32 data
#endif

/***************************************************************************
 * MAX30208 Data Buffers
 **************************************************************************/
#if   MASTER == 1 // Master Device
    static char curr_raw_temp_data_from_slave[size_of_MAX30208];
    static char prev_raw_temp_data_from_slave[size_of_MAX30208];
#elif SLAVE == 1 // Slave Device
    static char curr_raw_temp_data_to_master[size_of_MAX30208];
#endif

/***************************************************************************
 * MAX44009 Data Buffers
 **************************************************************************/
#if   MASTER == 1 // Master Device
    static char curr_raw_light_data_from_slave[size_of_MAX44009];
    static char prev_raw_light_data_from_slave[size_of_MAX44009];
#elif SLAVE == 1 // Slave Device
    static char curr_raw_light_data_to_master[size_of_MAX44009];
#endif

/***************************************************************************
 * MAX20361 Data Buffers
 **************************************************************************/
#if   MASTER == 1 // Master Device
    static char curr_raw_AO32_data_from_slave[size_of_MAX20361];
    static char prev_raw_AO32_data_from_slave[size_of_MAX20361];
#elif SLAVE == 1 // Slave Device
    static char curr_raw_AO32_data_to_master[size_of_MAX20361];
#endif

/** 
 * @brief This function constructs a payload buffer that is used in transmitting data in a LoRa Message
 */
void fillPayloadWithGlobalBufs(uint8_t * payload_buffer_tx); 

/** 
 * @brief This function deconstructs a payload buffer that contains data from a received LoRa Message
 */
void fillGlobalBufsWithPayload(uint8_t * payload_buffer_rx); 

#endif // __GLOBAL_BUFFERS_H__