#include "mbed.h"

#define max7219_reg_noop         0x00
#define max7219_reg_digit0       0x01
#define max7219_reg_digit1       0x02
#define max7219_reg_digit2       0x03
#define max7219_reg_digit3       0x04
#define max7219_reg_digit4       0x05
#define max7219_reg_digit5       0x06
#define max7219_reg_digit6       0x07
#define max7219_reg_digit7       0x08
#define max7219_reg_decodeMode   0x09
#define max7219_reg_intensity    0x0a
#define max7219_reg_scanLimit    0x0b
#define max7219_reg_shutdown     0x0c
#define max7219_reg_displayTest  0x0f

#define LOW 0
#define HIGH 1

SPI max72_spi(PTD2, NC, PTD1);
DigitalOut load(PTD0); //will provide the load signal

int const NUM_COLS = 8;
short pattern[NUM_COLS] = {0x0000, 0x0018, 0x003c, 0x007e, 0x00ff, 0x00e7, 0x0042, 0x0000};
short unresponsive_pattern[NUM_COLS] = {0x1000, 0x0018, 0x103c, 0x107e, 0x18ff, 0x04e7, 0x2442, 0x3c00};

long long symbols[10] = {
    0x0205050505050200, //0
    0x0702020202060200, //1
    0x0704040201050200, //2
    0x0205010301050200, //3
    0x0101070505030100, //4
    0x0705010107040700, //5
    0x0205050704020100, //6
    0x0404040402010700, //7
    0x0205050205050200, //8
    0x0205010107050700  //9
};
char split_pattern[NUM_COLS];

void write_to_max( int reg, int col1, int col2)
{
    load = LOW;            // begin
    max72_spi.write(reg);  // specify register
    max72_spi.write(col2);  // put data

    max72_spi.write(reg);  // specify register
    max72_spi.write(col1);  // put data
    load = HIGH;           // make sure data is loaded (on rising edge of LOAD/CS)
}

void setup_dot_matrix ()
{
    // initiation of the max 7219
    // SPI setup: 8 bits, mode 0
    max72_spi.format(8, 0);



    max72_spi.frequency(100000); //down to 100khx easier to scope ;-)


    write_to_max(max7219_reg_scanLimit, 0x07, 0x07);
    write_to_max(max7219_reg_decodeMode, 0x00, 0x00);  // using an led matrix (not digits)
    write_to_max(max7219_reg_shutdown, 0x01, 0x01);    // not in shutdown mode
    write_to_max(max7219_reg_displayTest, 0x00, 0x00); // no display test
    for (int e=1; e<=8; e++) {    // empty registers, turn all LEDs off
        write_to_max(e,0,0);
    }
    // maxAll(max7219_reg_intensity, 0x0f & 0x0f);    // the first 0x0f is the value you can set
    write_to_max(max7219_reg_intensity,  0x08,  0x08);

}

void clear()
{
    for (int e=1; e<=8; e++) {    // empty registers, turn all LEDs off
        write_to_max(e,0,0);
    }
}

//writes 8 bytes to the display
void pattern_to_display(short *testdata)
{
    int cdata1;
    int cdata2;
    for(int idx = 0; idx < 8; idx++) {
        cdata1 = testdata[idx]& 0xff;
        cdata2 = testdata[idx] >> 8;
        write_to_max(idx+1,cdata1,cdata2);
    }
}

int float_to_int(float flt)
{
    return (int) (flt + 0.5);
}

void write_to_display(int data, int prev_data)
{
    for (int j = 0; j < NUM_COLS; j++) {
        pattern[j] = pattern[j]<<1;
    }


    //Create line based on various conditions
    if (data < prev_data) {
        for (int i = data; i < prev_data; i++) {
            pattern[i] = pattern[i] + 0x0001;
        }
    }

    else if ( data > prev_data) {
        for (int i = prev_data+1; i <= data; i++) {
            pattern[i] = pattern[i] + 0x0001;
        }
    }

    else {
        pattern[data] = pattern[data] + 0x0001;
    }

    pattern_to_display(pattern);
}

//Number code

void split_to_chars (long long sym)
{
    split_pattern[0] = sym>>56;
    split_pattern[1] = (sym>>48)& 0x00FF;
    split_pattern[2] = (sym>>40)& 0x0000FF;
    split_pattern[3] = (sym>>32)& 0x000000FF;
    split_pattern[4] = (sym>>24)& 0x00000000FF;
    split_pattern[5] = (sym>>16)& 0x0000000000FF;
    split_pattern[6] = (sym>>8) & 0x000000000000FF;
    split_pattern[7] = sym & 0x00000000000000FF;
}

void get_num_pattern(int data)
{
    short units[NUM_COLS];
    short tens[NUM_COLS];
    short hundreds[NUM_COLS];

    short pattern[NUM_COLS];

    int unit = data%10;
    data = data - unit;

    int ten = data%100;
    data = data-ten;
    ten = ten/10;

    int hundred = data%1000;
    hundred = hundred/100;

    split_to_chars(symbols[unit]);

    for (int i = 0; i<NUM_COLS; i++) {
        units[i] = split_pattern[i];
    }

    split_to_chars(symbols[ten]);
    for (int i=0; i<NUM_COLS; i++) {
        tens[i] = split_pattern[i];
    }

    split_to_chars(symbols[hundred]);
    for (int i=0; i<NUM_COLS; i++) {
        hundreds[i] = split_pattern[i];
    }

    for(int i = 0; i<NUM_COLS; i++) {
        tens[i] = tens[i]<<4;
        hundreds[i] = hundreds[i]<<8;
        pattern[i] = ((hundreds[i])|(units[i]+tens[i]));
    }

    pattern_to_display(pattern);

}

//-------------------------------------------------------------Display setup end-----------------------------------------------------------------------------------------------------------------------------------------//

DigitalOut pulse(LED1);

AnalogIn Ain(PTB1);
AnalogOut Aout(PTE30);

bool dataReady = false;   //Data ready to be processed flag
bool intialised = false;  //Setup during first two pulses
bool triggered = false ;    //Represntative if rising edge has been detected

//Constants
int const AVG_LEN = 160;   //Lenght of Avg length buffer
int const NORM_LEN = 80;   //Lenght of Avg length buffer
int const BUFF_LEN = 20;   //Length of sample buffer
float const ALPHA = 0.2;     //Aplha value used for filtering
float const OFFSET = 0.2;  //Used to ensure normalised values don't be come negative

//Buffers
float sample_buffer[BUFF_LEN] = {};  //Circular array storing values from sample interupt
float avg_buffer[AVG_LEN] = {};      //circular array containing most recent 160 values
float norm_buffer[NORM_LEN] = {};      //circular array used for calculating max and min of nofmalise data

//Averageing values
float avg_sum = 0;           //Sum of most recent 160 values

//Sample Buffer pointers
int read = 0;
int write = 0;
//int counter = 0;  //Keeps track of number of data values read in

//Avg buffer pointer
int avg_write = 0;

//Norm buffer pointer
int norm_write = 0;

Ticker sampler;
Ticker but;

//BPM
DigitalIn button (PTB0);
Ticker bpm;
int bpmRate;
int bpmCount;
int Time = 15;
bool Display = true;
bool pressed = false;

void bpmSample()
{
    bpmRate = bpmCount*4;
    bpmCount = 0;
}

void buttonCheck (){
    if (!pressed) {
        if (button == 0) {
            Display = !Display;
            pressed = true;
        }
}
}

bool unresponsive = false;

void sampling ()    //Sample Signal
{
    float sample = Ain.read();
    sample_buffer[write++] = sample;
    write = write%(BUFF_LEN);  //Ensure buffer pointer rolls over i.e circular buffer

    dataReady = true;
    }


float Min()   //Get min value of two pulses data
{
    float min = norm_buffer[0];
    for (int i = 0; i < NORM_LEN; i++) {
        if (norm_buffer[i] < min) {
            min = norm_buffer[i];
        }
    }
    return min;
}

float Max()   //Get max value of two pulse data
{
    float max = norm_buffer[0];
    for (int i = 0; i < NORM_LEN; i++) {
        if (norm_buffer[i] > max) {
            max = norm_buffer[i];
        }
    }
    return max;
}

float scale(float data)    //Centralise data
{
    float max = Max();
    float min = Min();
    return ((data-min)*((NUM_COLS-1)/(max-min)));

}

float filter(float data, float normalisedData)    //Digital low plasss filter
{
    return data*ALPHA + (1-ALPHA)*normalisedData;   // X = alpha*x + (1-alpha)prevX
}
float average (float data)                      //Calcualte new average and remove value from signal
{
    if (!intialised) {                                // Gathering data for first two intial pulses
        avg_buffer[avg_write] = data;
        avg_write = ++avg_write% AVG_LEN; //Ensure avg_write pointer wraps around
        avg_sum += data;                   //Update running sum

        if (avg_write == 0) {                 //First 160 values have been added to buffer
            intialised= true;
        }

        return 0.0;
    } else {                                    //Main runtime of program
        float old_data = avg_buffer[avg_write];   // oldest value
        avg_buffer[avg_write] = data;           // over write oldest value with new data
        avg_sum = avg_sum - old_data + data;    //Update running sum to contain most recent 160 values

        avg_write = ++avg_write%AVG_LEN;          //Ensure avg_write wraps around

        return data + OFFSET - (avg_sum/AVG_LEN) ;
    }
}

float processData(float data, float normalisedData)     //Normalise and filter signal
{
    float processing, processed;
    processing = average(data);   //Remove runnning average
    processed = filter(processing, normalisedData); //Low pass filter signal
    norm_buffer[norm_write] = processed;           // over write oldest value with new data
    norm_write = ++norm_write%NORM_LEN;          //Ensure avg_write wraps around

    return processed ;
}

void detectPulse(float normalisedData)   //Turn on LED during high part of pulse
{
    if(!unresponsive) {
        if(triggered) {  //Previously above high trigger
            if (normalisedData < 3) {  //If it drops below ,ow trigger point
                triggered = false;
                pulse = 1;
            } else { //Still above low trigger
                pulse = 0;
            }
        } else { //Previously below low trigger
            if (normalisedData > 5) {//If it goes above high trigger point
                triggered = true;
                pulse = 0;
                bpmCount++;
                //bpmCount ++;
            } else { //Still below high trigger
                pulse = 1;
            }
        }
    }
}

int main()
{
    /* setup display */
    setup_dot_matrix ();
    pattern_to_display(pattern);
    wait(1);

    sampler.attach(&sampling, 0.0125); //Sample at 80Hz
    bpm.attach(&bpmSample, Time);      //BPM sample
    but.attach(&buttonCheck, 0.15);      //BPM sample
         //BPM sample
    float normalisedData = 0;          //Data point which has been normalised
    float output;
    int processed_data;
    int prev_data = 0;
    int count = 0;
    float data ;


    while(1) {
        if(dataReady) {  //If more data has been sampled
            dataReady = false;

            while(read != write) {  //While there is data left in sample buffer
                data = sample_buffer[read] ;
                read = ++read%BUFF_LEN;  //Ensure read pointer rolls around in sample buffer
                normalisedData = processData(data, normalisedData);
                count++;
                output = scale(normalisedData); //Normalise as per formula

                if(intialised and !unresponsive) { //Passed first two pulse
                    detectPulse(output);
                }


                if(count == 7) { // Write at approx. 8Hz
                    if (Display) {
                        if ((Max()-Min()) > 0.002)  {
                            unresponsive = false;
                            count = 0;
                            processed_data = float_to_int(output);
                            write_to_display(processed_data, prev_data);
                            prev_data = processed_data;
                        } else {
                            unresponsive = true;
                            count = -60;
                            pulse = 1;
                            pattern_to_display(unresponsive_pattern);
                        }
                        pressed = false;
                    } else {
                        get_num_pattern(bpmRate);
                        count = 0;
                        pressed = false;

                    }

                }
            }

        }
    }
}