/*
 * @file DS1721.cpp
 * @author Cameron Haegle
 *
 * @section LICENSE
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * Library for the DS1721, 2-Wire Digital Thermometer and Thermostat, from Maxim Semiconductor
 *  (www.maiximintegrated.com). 
 *
 * @section LIMITATIONS
 * 
 * This library was not written with for use with the mbed-RTOS, as the I2C read/write calls are 
 *  not thread safe.
 *
 * ChangeLog
 * 07/30/2013 
 *   - 
 */



#include "DS1721.h"


DS1721::DS1721(I2C &i2c, int addr) : m_i2c(i2c) , m_addr(addr)
{    
    m_i2c.frequency(400000);
    _resolution    = RES_9_BIT;
    _polarity      = POLARITY_ACTIVE_HIGH;
    _mode          = CONV_FOREVER;
    _num_read = 1;
     
    //setConfig(_resolution | _polarity | _mode);
}

DS1721::DS1721(I2C &i2c, int resolution, int polarity, int mode, int addr) 
        : m_i2c(i2c) , _resolution(resolution) , _polarity(polarity) , _mode(mode) , m_addr(addr)
{ 
    m_i2c.frequency(400000);   
    _num_read = 1;     
    //setConfig(_resolution | _polarity | _mode);
}

DS1721::~DS1721(void) 
{ 
}

int DS1721::setConfig(int config)
{
    char cmd[2];
    cmd[0] = CMD_ACCESS_CFG;
    cmd[1] = (config & 0xFF);
        
    if(writeData(cmd, 2) != 0)
    {        
        return 1;
    }    
    return 0;
}

int DS1721::getConfig(int* config)
{
    char cmd[2];
    
    cmd[0] = CMD_ACCESS_CFG;      // 0xAC
  
    if(writeData(cmd, 1) != 0)
    {       
        return 1;
    }

    if(readData(cmd, 1) != 0)
    {      
        return 1;
    }      
    *config = cmd[0];
    return 0;
}
 

float DS1721::getTemp(void)
{
    char cmd[3];
   
    cmd[0] = CMD_READ_TEMP;
    
    // if this is the first read, since power up, perform the read twice
    // this eliminate  an erroneous initial read.
    do
    {        
        if(writeData(cmd, 1) != 0)
        {       
            return 1;
        }    
        
        // read 1 byte, ignoring fractional portion of temperature
        if(readData(cmd, 2) != 0)
        {
            return 1;
        }
        _num_read--;
    }while(_num_read);
    
    _num_read = 1;
    
    return ((float)(cmd[1] + (cmd[0] << 8))/256);
}

int DS1721::setPolarity(int polarity)
{
    _polarity = polarity;
    return setConfig(_resolution | _polarity | _mode);
}

int DS1721::getPolarity(void)
{
    int config = 0;    
    getConfig(&config);
    return ((config & (1<<1)) ? 1: 0);
}

int DS1721::startConversion(void)
{
   char cmd[1];
    
    cmd[0] = CMD_START_CONVT; // 0x51
    
    if(writeData(cmd, 1) != 0)
    {      
        return 0;
    }  
    return 1;
}

int DS1721::stopConversion(void)
{
    char cmd[1];
    
    cmd[0] = CMD_STOP_CONVT; // 0x51
    
    if(writeData(cmd, 1) != 0)
    {       
        return 0;
    }  
    return 1;
}

float DS1721::getLowSp(void)
{
    char cmd[4];
        
    cmd[0] = CMD_ACCESS_TL;
    
    if(writeData(cmd, 1) != 0)
    {     
        //return FALSE;
    }
    
    // read back TL msb & lsb bytes
    if(readData(cmd, 2) != 0)
    {       
        return 0;
    }    
       
    _tl = ((cmd[0] << 8) + cmd[1]);  
    
    return ((float)_tl/256);
}

int DS1721::setLowSp(float newSp)
{
    char cmd[3];
    
    _tl = (newSp*256);
    
    cmd[0] = CMD_ACCESS_TL;
    cmd[1] = (char)(_tl >> 8);         // temp MSB
    cmd[2] = (char)(_tl & 0x00FF);     // temp LSB (+/-0.5)
    
    if(writeData(cmd, 3) != 0)
    {       
        return 0;
    }
    return 1;
}

float DS1721::getHighSp(void)
{
    char cmd[3];
        
    cmd[0] = CMD_ACCESS_TH;

    if(writeData(cmd, 1) != 0)
    {
        //return 0;
    }
    // read back TL msb & lsb bytes
    if(readData(cmd, 2) != 0)
    {
        //return 0;
    }    
    
    _th = ((cmd[0] << 8) + cmd[1]);  
    
    return ((float)_th/256);
}

int DS1721::setHighSp(float newSp)
{
    char cmd[3];
    
    _th = (newSp * 256);

    cmd[0] = CMD_ACCESS_TH;
    cmd[1] = (char)(_th >> 8);         // setpoint MSB
    cmd[2] = (char)(_th & 0x00FF);     // setpoint LSB (+/-0.5)

    if(writeData(cmd, 3) != 0)
    {      
        return 0;
    }
    return 1;
}

float DS1721::temp_CtoF(float tempC)
{
    return ((tempC * 1.8) + 32);
}

float DS1721::temp_FtoC(float tempF)
{
    return ((tempF - 32)* 0.55);
}

int DS1721::writeData(char* data, int length)
{
    return (m_i2c.write(m_addr, data, length));
}

int DS1721::readData(char* data, int length)
{
    return (m_i2c.read(m_addr, data, length));
}