#include "mbed.h"
#include "L6206.h"
#include "BDCMotor.h"
#include <math.h>

/****************************/
/*      PIN DEFINITION      */
/****************************/

//  CAN
#define CAN_RX  PB_8
#define CAN_TX  PB_9

//  ENCODER
#define CH_A    PA_8
#define CH_B    PA_9

/********************************/
/*             CAN              */
/********************************/

typedef enum
{
    JOINT_SET_SPEED = 20,
    JOINT_SET_POSITION,
    JOINT_CURRENT_POSITION,
    JOINT_CURRENT_SPEED,
    JOINT_STATUS,
    JOINT_ERROR,
    JOINT_TORQUE,
    JOINT_MAXTORQUE,
    JOINT_ZERO,
}   CAN_COMMANDS;

typedef enum
{
    BASE = 1,
    SHOULDER,
    ELBOW,
    WRIST1,
    WRIST2,
    WRIST3,
    END_EFFECTOR,
    CAMERA1,
    CAMERA2,
}   JOINT;

//
DigitalOut myled(LED1);

static volatile uint16_t gLastError;
static volatile uint8_t gStep = 0;

L6206_init_t init =
{
    L6206_CONF_PARAM_PARALLE_BRIDGES,                                                    
    {L6206_CONF_PARAM_FREQ_PWM1A, L6206_CONF_PARAM_FREQ_PWM2A, L6206_CONF_PARAM_FREQ_PWM1B, L6206_CONF_PARAM_FREQ_PWM2B},
    {100,100,100,100},
    {FORWARD,BACKWARD,FORWARD,BACKWARD},
    {INACTIVE,INACTIVE,INACTIVE,INACTIVE},
    {FALSE,FALSE}
};

//  Motor definition
L6206 *LinAct;
L6206 *EndEff;

int speed_elbow = 0;
int speed_ee = 0;

/*********************************/
/*      Interrupt Handlers       */
/*********************************/

//  Error Handler (called by the library when it reports an error)
void my_error_handler(uint16_t error)
{
  /* Backup error number */
  gLastError = error;
  
  /* Enter your own code here */
}

//  Flag Handler (overcurrent and thermal alarms reporting)
void my_flag_irq_handler(void)
{
    /* Get the state of bridge A */
    uint16_t bridgeState  = EndEff->get_bridge_status(0);

    if (bridgeState == 0) 
    {
        if ((EndEff->get_device_state(0) != INACTIVE)||
            (EndEff->get_device_state(1) != INACTIVE)) 
        {
            /* Bridge A was disabling due to overcurrent or over temperature */
            /* When at least on of its  motor was running */
            my_error_handler(0XBAD0);
        }
    }
  
    /* Get the state of bridge B */
    bridgeState  = LinAct->get_bridge_status(1);
  
    if (bridgeState == 0)  
    {
        if ((LinAct->get_device_state(2) != INACTIVE)||
            (LinAct->get_device_state(3) != INACTIVE)) 
        {
            /* Bridge A was disabling due to overcurrent or over temperature */
            /* When at least on of its  motor was running */
            my_error_handler(0XBAD1);
        }
    }  
}

/****************************/
/*      CAN Interface       */
/****************************/

CAN can1(PB_8, PB_9);     // RX, TX

Thread t_canrx, t_cantx;

uint32_t gen_can_id(CAN_COMMANDS message_id, JOINT can_id)
{
    uint32_t id = (uint32_t)can_id;     // LSB byte is the controller id.
    id |= (uint32_t)message_id << 8;  // Next lowest byte is the packet id.
    id |= 0x80000000;              // Send in Extended Frame Format.
    return id;
}

bool can_rx()
{
    CANMessage messageIn;
    messageIn.format = CANExtended;
    bool status = can1.read(messageIn);
    printf ("CAN ID : %d Message received : %d\n\r", messageIn.id, status);
    
    if(can1.read(messageIn))
    {
        myled = 1;
        wait(0.5);
        myled = 0;
        wait(0.5);
        if(messageIn.id == gen_can_id(JOINT_SET_SPEED, ELBOW))
        {
            speed_elbow = messageIn.data[3] + (messageIn.data[2] << 8) + (messageIn.data[1] << 16) + (messageIn.data[0] << 24);
        }
    }
    
    /*
    if(can1.read(messageIn))
    {
        myled = 1;
        wait(0.5);
        myled = 0;
        wait(0.5);
        if(messageIn.id == gen_can_id(JOINT_SET_SPEED, END_EFFECTOR))
        {
            speed_ee = messageIn.data[3] + (messageIn.data[2] << 8) + (messageIn.data[1] << 16) + (messageIn.data[0] << 24);
        }
    }
    */
    
    if(can1.read(messageIn) && messageIn.id == gen_can_id(JOINT_ZERO,ELBOW))
    {
        if((messageIn.data[0] + (messageIn.data[1] << 8) + (messageIn.data[2] << 16) + (messageIn.data[3] << 24)) == 1)
        {
            LinAct->run(1, BDCMotor::BWD);
        }
    }
    
    return status;
}

void can_rx_isr()
{
    while(1)
    {
        can_rx();
        osDelay(10);
    }
}

/*****************************/
/*            MAIN           */
/*****************************/

int main()
{
    can1.frequency(125000);

//  Motor Initialization   
#ifdef TARGET_STM32F429
    LinAct = new L6206(PB_14, PB_15, PA_8, PA_9, PC_6, PC_7);     // EN_A, EN_B, IN1_A, IN2_A, IN1_B, IN2_B
    EndEff = new L6206(PB_14, PB_15, PA_8, PA_9, PC_6, PC_7);     // EN_A, EN_B, IN1_A, IN2_A, IN1_B, IN2_B
#else
    //LinAct = new L6206(PB_14, PB_15, PA_8, PA_9, PC_6, PC_7);
    LinAct = new L6206(D2, A4, D5, D4, A0, A1);
    EndEff = new L6206(D2, A4, D5, D4, A0, A1);
#endif

    if (LinAct->init(&init) != COMPONENT_OK) 
    {
        printf("ERROR: vvMotor Init\n\r");
        exit(EXIT_FAILURE);
    }
    if (EndEff->init(&init) != COMPONENT_OK) 
    {
        printf("ERROR: vvMotor Init\n\r");
        exit(EXIT_FAILURE);
    }
    
    LinAct->attach_flag_interrupt(my_flag_irq_handler);
    LinAct->attach_error_handler(my_error_handler);
    EndEff->attach_flag_interrupt(my_flag_irq_handler);
    EndEff->attach_error_handler(my_error_handler);
    printf("DONE: Motor Init\n\r");
    
    LinAct->set_dual_full_bridge_config(PARALLELING_NONE___1_BIDIR_MOTOR_BRIDGE_A__1_BIDIR_MOTOR_BRIDGE_B);
    EndEff->set_dual_full_bridge_config(PARALLELING_NONE___1_BIDIR_MOTOR_BRIDGE_A__1_BIDIR_MOTOR_BRIDGE_B);

    // CAN Initialization
    t_canrx.start(can_rx_isr);
    printf("DONE: CAN Init\n\r");

    while(true)
    {            
        EndEff->set_speed(0, speed_ee);
        if(speed_ee < 0)
            EndEff->run(0, BDCMotor::BWD);
        else if(speed_ee > 0)
            EndEff->run(0, BDCMotor::FWD);
        else if(speed_ee == 0)
            EndEff->hard_stop(0);
            
        LinAct->set_speed(1, speed_elbow);
        if(speed_elbow < 0)
            LinAct->run(1, BDCMotor::BWD);
        else if(speed_elbow > 0)
            LinAct->run(1, BDCMotor::FWD);
        else if(speed_elbow == 0)
            LinAct->hard_stop(1);
          
        osDelay(100);
    }
}