#include <iterator>
#include "monitoringExecutor.h"
#include "circleMonitoringStrategy.h"
#include "squareMonitoringStrategy.h"
#include "monitoringContext.h"


MonitoringExecutor::MonitoringExecutor(Configuration* config, Serial* pc, SRF05* ranger, PwmOut* rangerServo, PwmOut* fan, PwmOut* fanServo, DigitalOut* greenLed, DigitalOut* redLed):
        m_config(config), m_pc(pc), m_ranger(ranger), m_rangerServo(rangerServo), m_fan(fan), m_fanServo(fanServo), m_greenLed(greenLed), m_redLed(redLed)
{
}

MonitoringExecutor::~MonitoringExecutor()
{
}

void MonitoringExecutor::startMonitoring()
{
    MonitoringContext* monitoringContext;
    MonitoringStrategyIf* strategy;
    ObjectPosition* objPos = new ObjectPosition(0,0);
    
    m_moitoringDurationTimer.start();
    m_greenLed->write(1);
    m_redLed->write(0);
    
    if(m_config->getMode() == 0) { //circle
            strategy = new CircleMonitoringStrategy(m_config);
        } else if(m_config->getMode() == 1) { //squere
            strategy = new SquereMonitoringStrategy(m_config);
        }
    monitoringContext = new MonitoringContext(strategy);

    while(checkMonitoringDurationTimer()) {
        m_distancesMap.clear();
        m_pc->printf("Monitoring start\n\r");
        // m_pc->printf("----Map size: %d\n\r", m_distancesMap.size()); 
        findObjects();
        //m_pc->printf("----TIMER: %f\n\r", m_moitoringDurationTimer.read());
        if(monitoringContext->executePresenceCheck(m_distancesMap, objPos)) {
            //m_pc->printf("INFO: Object founded into monitoring area, on position -> distance:%f, angle:%f\n\r", objPos->getDistance(), objPos->getAngle());
            m_greenLed->write(0);
            m_redLed->write(1);
            activateFan(objPos);
            delete[] objPos;
        }
        else {
            m_greenLed->write(1);
            m_redLed->write(0);
            }
    }
    m_moitoringDurationTimer.stop();
    m_pc->printf("Monitoring timer exceeded. \n\r");
    delete[] monitoringContext;
    delete[] strategy;
    m_greenLed->write(0);
    m_redLed->write(0);
}

void MonitoringExecutor::findObjects()
{
    m_pc->printf("Find objects. \n\r");
    for (int i = 1500; i <= 2500; i+=50) {
        m_rangerServo->pulsewidth_us(i);
        wait(0.2);
        float angle = toDeeg(i-1500);
        float dis = m_ranger->read();//-
        m_distancesMap.insert(std::pair<float, float>(angle, dis));//for each object found set the corresponding angle and distance
        m_pc->printf("Angle = %1.f ; Distance = %.1f; xxx: %d\n\r", toDeeg(i-1500), m_ranger->read(), i);
    }
    m_rangerServo->pulsewidth_us(500);
}

void MonitoringExecutor::activateFan(ObjectPosition* objPos)
{
    float angle;
    float dist;
    Timer fanDuration;
    ObjectPosition fanPos(0, m_config->getFanYCoordinate());
    dist = fanPos.getDistanceFromOtherObj(objPos);
    wait(2);
    m_pc->printf("Activate Fan.\n\r");
    
    if(objPos->getYCoordinate() > fanPos.getYCoordinate()) {
        angle = 90 + (acos(objPos->getXCoordinate() / dist) *180/PI);
    } else {
        angle = 90 - (acos(objPos->getXCoordinate() / dist)*180/PI);
    }
    
    m_fanServo->pulsewidth_us(500 + (2000 * (angle / 180)));//-
    //m_pc->printf("---angle_pwm: %f,  angle: %f, dist_from_fan: %f \n\r", 500 + (2000 * (angle / 180)), angle, dist);//-
    
    wait(2);
    m_fanWorkingDurationTimer.start();
    m_fan->pulsewidth_us((dist/m_config->getDistance()) * 20000);//fan rotation speed depend on distance between fan and object//-
    //m_pc->printf("---fan = %f\n\r", (dist/m_config->getDistance()) * 10000);
    while(checkFanWorkingDurationTimer()) {
        }
    m_fan->pulsewidth_us(0);
    m_fanServo->pulsewidth_us(500);
    m_fanWorkingDurationTimer.stop();
    m_fanWorkingDurationTimer.reset();
}

float MonitoringExecutor::toDeeg(int angle)
{
    return angle/50*4.5;//4.5 should be seted by config value
}

bool MonitoringExecutor::checkMonitoringDurationTimer(){
    return m_config->getMonitoringDuration() > m_moitoringDurationTimer.read();
}

bool MonitoringExecutor::checkFanWorkingDurationTimer(){
    return m_config->getFanWorkingDuration() > m_fanWorkingDurationTimer.read();
}