#ifndef DRV8301_H
#define DRV8301_H

#include "mbed.h"

/** DRV8301 klasa izrađena za potrebe završnog rada.
  * Koristi se za pokretanje i podešavanje postavki predupravljača DRV8301 pisanjem pomoću SPI sučelja izravno u kontrolne registre predupravljača
  * te čitajući podatke iz statusnih registara kako bi se detektirale pogreške. Ova biblioteka služi kao jednostavno sučelje između korisnika i sustava
  * koji obuhvaća predupravljač DRV8301 i mbed mikroupravljač  
*/
class DRV8301{
    public:
        DRV8301();
        /** Jasno je da je to konstruktor. Tu navedite opis parametara konstruktora         */
        DRV8301(PinName cs, PinName mosi, PinName miso, PinName sck, PinName en_gate);
        
        /* TB: potpuno nepotrebne varijable  
        uint8_t inputCL;
        uint8_t inputGain;
        uint8_t inputPWMNumber;
        uint8_t inputOCPMode;
        uint8_t inputOCTWMode;
        uint8_t inputGateCurrent;
        */

        /* TB: maknuti//korištene funkcije      */
    
        /** Postavlja ograničenje struje u željenu vrijednost definirane prema tablici 13 u datasheetu DRV8301
            @param inputCL Broj od 0 do 31 koji unosi korisnik kako bi odabrao koje ograničenje struje želi postaviti 
            definiran prema tablici 13 u datasheetu DRV8301 uzimajući drain-to-source napon i računajući iznos struje
        */
        void setCurrentLimit(uint8_t inputCL);
        
        /** Postavlja pojačanje integriranih pojačala na 10, 20, 40 ili 80 V/V
            @param inputGain Broj kojega korisnik unosi kako bi odabrao željeni 
            iznos pojačanja prema tablici 12 u datasheetu DRV8301
        */
        void setGain(uint8_t inputGain);
        
        /** Postavlja željeni broj PWM signala (6 ili 3)
            @param inputPWMNumber Broj koji korisnik unosi na osnovu kojega se 
            prema tablici 11 u datasheetu DRV8301 odabire broj PWM signala
        */
        void setPWMNumber(uint8_t inputPWMNumber);
        
        /** Postavlja prekostrujnu zaštitu predupravljača na jedan od četiri moguća moda: 
            Current limit, OC latch shutdown, Report only, OC disabled prema tablici 11 u datasheetu DRV8301
            @param inputOCPMode Broj kojega korisnik unosi koji određuje koji mod prekostrujne zaštite 
            aktivirati prema tablici 11 u datasheetu DRV8301
        */
        void setOCPMode(uint8_t inputOCPMode);
        
        /** Postavlja što će OCTW pin javljati u slučaju detektirane pogreške: 
        prekomjernu struju i temperaturu, prekomjernu struju ili prekomjernu temperaturu,
            @param inputOCTWMode Broj kojega korisnik unosi koji određuje koji od 
            četiri moda dojave na OCTW pinu će biti aktivan. Četvrti mod je rezerviran za kasnije nadogradnje i zasad nije upotrebljiv
        */
        void setOCTW(uint8_t inputOCTWMode);
        
        /** Dohvaća vrijednost spremljenu u kontrolni registar 1
            @return vraća kroz SPI 16-bitnu riječ koja se sprema u varijablu controlRegister1 koja se može dalje koristiti 
        */
        uint16_t getControlRegister1();
        
        /** Dohvaća vrijednost spremljenu u kontrolni registar 2
            @return vraća kroz SPI 16-bitnu riječ koja se sprema u varijablu controlRegister2 koja se može dalje koristiti 
        */
        uint16_t getControlRegister2();
        
        /** Dohvaća vrijednost spremljenu u statusni registar 1
            @return vraća kroz SPI 16-bitnu riječ koja se sprema u varijablu statusRegister1 koja se može dalje koristiti 
        */
        uint16_t getStatusRegister1();
        
        /** Dohvaća vrijednost spremljenu u statusni registar 2
            @return vraća kroz SPI 16-bitnu riječ koja se sprema u varijablu statusRegister2 koja se može dalje koristiti 
        */
        uint16_t getStatusRegister2();
        
        /** Postavlja ograničenje struje gatea
            @param inputGateCurrent Broj kojega korisnik unosi koji odrađuje iznos ograničenja
            struje gatea prema tablici 11 DRV8301 datasheeta. Četvrti mod je rezerviran za kasnije nadogradnje i zasad nije upotrebljiv
        */
        void setGateCurrent(uint8_t inputGateCurrent);
        
    private:
        SPI spi;
        
//korištene varijable

        DigitalOut cs;  //"chip select" digital out line which selects the SPI slave
        /* TB: nepotrebno jer je to dio spi-a
        DigitalOut mosi;    //"master output slave input" digital out line through which SPI master sends data to the slave
        DigitalIn miso; //"master input slave output" digital in line through which SPI master receives data from SPI slave
        DigitalOut sck; //"SPI clock" digital out line, used for synchronizing SPI communication */
        DigitalOut en_gate; //"enable gate" digital out line used to reset the DRV8301 if necessary
        
        uint16_t controlRegister1;  //control register 1 data is saved here
        uint16_t controlRegister2;  //control register 2 data is saved here
        uint16_t statusRegister1;   //status register 1 data is saved here
        uint16_t statusRegister2;   //status register 2 data is saved here
        
        /** reboots the entire DRV8301, shutting down then starting up the pre-driver unlatching faults which will remain unlatched if no fault is still present
        */
        void startup(); 
};
#endif