#ifndef PI_H
#define PI_H

#include "mbed.h"

/** Data type for choosing the operation mode of the controller ("automatic" or "manual"). */
enum Mode {automatic, manual};

/** Transfer function of a PI controller with output limitation, anti-windup and bumpless automatic and manual mode.
 * G_C(s) = U_C(s) / E(s) = K_C ( 1 + 1 / (T_I s) ) = K_C (1 + T_I s) / (T_I s)
 *
 * K_C is the controller gain, and T_I is the controller integral time constant in seconds.
 *
 * Author(s): TVZ Mechatronics Team
 *
 */
class PI {
    public:
        /** Default constructor.
         * K_C = 1, T_I = 1 s, sampleTime = 0.1 s.
         */
        PI();
        /** PI controller gain, integral time constant in seconds and sample time in seconds. */
        PI(double K_C, double T_I, double sampleTime);
        /** Update PI controller gain, integral time constant and sample time. */
        void setParameters(double K_C, double T_I, double sampleTime);
        /** Set the controller lower and upper output limit. */
        void setOutputLimits(double lowerOutputLimit, double upperOutputLimit);
        /** Set the operation mode to "automatic" or "manual". */
        void setOperationMode (Mode operationMode);
        /** Set the controller output manually. */
        void setOutputManually(double u_C);
        /** Calculate the controller output u_C. */
        double out();
        /** Set the PI controller input e = SP - PV (setpoint - process value). */
        void in(double e);
    private:
        double K_R, T_I, T_d, y, y_p1, u, u_p1, y_max, y_min, y_manual;
        Mode operationMode;
};

#endif // PI_H