#include "mbed.h"

DigitalOut LED_3(LED3);

/******
PIN + SPI CONFIG
******/

SPI SPI_AFE_ADC_AND_DDS(p5, p6, p7); //SPI0: MOSI, MISO, SCLK
DigitalOut ADC_nCS(p8);  //ADC_!CS

//  These pins control the Phase register and the Frequency register, IF THE DDS IS CONFIGURED TO BE CONTROLLED BY THE PINS,
//BY DEFAULT, AND HOW I HAVE IT CONFIGURED IS TO NOT USE THESE PINS, REPEAT, it does NOT USE THE PINS, instead uses the SPI bus.
//  By toggling them after config, you can create PSK, FSK, BPSK...
DigitalOut CURRENT(p14); //if=1, output of CSM to input selector / switch 2
DigitalOut DDS_nCS(p15); //nCS line used for SPI on 
DigitalOut DDS_PSEL(p16);    
DigitalOut DDS_FSEL(p17);  

DigitalOut SQUARE(p18); //if=1 closes switch to SQUARE path that is not populated
DigitalOut DDS(p19); //if=1, DDS output connected to TX_PORT
DigitalOut AFE(p20); //if=1, Electrode routed to Amplification Path

#define SSP_SR_TFE (1<<0)
#define SSP_SR_TNF (1<<1)
#define SSP_SR_RFE (1<<2)
#define SSP_SR_RFF (1<<3)
#define SSP_SR_BSY (1<<4)

void writeReg(uint16_t val){
    __disable_irq();

    DDS_nCS = 0;
    //DDS_nCS0;
    
    while ( !(LPC_SSP0->SR & SSP_SR_TNF) ); //while ssp is not writable
    LPC_SSP0->DR = val;                //write this out 
    while ( (LPC_SSP0->SR & SSP_SR_BSY) ); //wait until transmission is over to pull ncs up
    
    DDS_nCS = 1;
    //DDS_nCS1;
    
    __enable_irq();
}

/******************
DDS Library ported from https://github.com/arachnidlabs/ad983x/blob/master/ad983x.cpp
******************/
#define REG_OPBITEN 0x0020
#define REG_SIGNPIB 0x0010
#define REG_DIV2    0x0008
#define REG_MODE    0x0002
#define SIGN_OUTPUT_MASK (REG_OPBITEN | REG_SIGNPIB | REG_DIV2 | REG_MODE)

uint16_t m_reg;

enum SignOutput {
  SIGN_OUTPUT_NONE        = 0x0000,
  SIGN_OUTPUT_MSB         = 0x0028,
  SIGN_OUTPUT_MSB_2       = 0x0020,
  SIGN_OUTPUT_COMPARATOR  = 0x0038,
};

void setSignOutput(SignOutput out) {
  m_reg = (m_reg & ~SIGN_OUTPUT_MASK) | out;
  writeReg(m_reg);
}

enum OutputMode {
  OUTPUT_MODE_SINE        = 0x0000,
  OUTPUT_MODE_TRIANGLE    = 0x0002,
};

void setOutputMode(OutputMode out) {
  if(out == OUTPUT_MODE_TRIANGLE) {
    m_reg = (m_reg & ~SIGN_OUTPUT_MASK) | out;
  } else {
    m_reg &= ~REG_MODE;
  }
  writeReg(m_reg);
}

#define REG_FREQ1   0x8000
#define REG_FREQ0   0x4000
void setFrequencyWord(bool reg, uint32_t frequency) {
  writeReg(0x2000);                                                             // sets the PIN/SW D9 pin to high >> FSELECT pin can be used to use either FREQ0 or FREQ1
  writeReg((reg?REG_FREQ1:REG_FREQ0) | (frequency & 0x3FFF));
  writeReg((reg?REG_FREQ1:REG_FREQ0) | ((frequency >> 14) & 0x3FFF));
}

#define REG_PHASE0  0xC000
#define REG_PHASE1  0xE000
void setPhaseWord(bool reg, uint32_t phase) {
  writeReg((reg?REG_PHASE1:REG_PHASE0) | (phase & 0x0FFF));
}

#define REG_FSEL    0x0800
void selectFrequency(bool reg) {
  if(reg) {
    m_reg |= REG_FSEL;
  } else {
    m_reg &= ~REG_FSEL;
  }
  writeReg(m_reg);
}

#define REG_PSEL    0x0400
void selectPhase(bool reg) {
  if(reg) {
    m_reg |= REG_PSEL;
  } else {
    m_reg &= ~REG_PSEL;
  }
  writeReg(m_reg);
}

#define REG_RESET   0x0100
void reset(bool in_reset) {
  if(in_reset) {
    m_reg |= REG_RESET; // the reset pin goes up
  } else {
    m_reg &= ~REG_RESET; // the reset pin goes down
  }
  writeReg(m_reg); //writes 16 bits over SPI
}

void begin(){
    reset(true);
    writeReg(m_reg);
    // Initialize frequency and phase registers to 0
    setFrequencyWord(0, 0);
    setFrequencyWord(1, 0);
    setPhaseWord(0, 0);
    setPhaseWord(1, 0);
    reset(false);
}

uint8_t Crc8(uint8_t *vptr, int len)
{
    uint8_t *data = vptr;
    unsigned crc = 0;
    int i, j;
    for (j = len; j; j--, data++) {
        crc ^= (*data << 8);
        for(i = 8; i; i--) {
            if (crc & 0x8000)
                crc ^= (0x1070 << 3);
            crc <<= 1;
        }
    }
    return (uint8_t)(crc >> 8);
}

/*******************
MAIN
*******************/
int main() {
    //init pins
    DDS_PSEL = 0;
    DDS_FSEL = 0;
    SQUARE = 0;
    DDS = 0;
    AFE = 0;
    LED_3 = 1;
    
    //nCS lines
    DDS_nCS = 1;
    ADC_nCS = 1;
    CURRENT = 0;
    
    //init spi
    SPI_AFE_ADC_AND_DDS.format(16,2);
    SPI_AFE_ADC_AND_DDS.frequency(30000000);
    
    //make sure SPI is setup to mode 2, MSB first
    //init DDS
    begin();
    //output calculation
    //fOut = fMCLK / 2^28 * FREQREG = 50e6 / 2^28 * FREQREG
    //FREQREG = fOut * 2^28 / 50e6 = fOut * 5.36870912
    //100K: 536870, 1 MHz: 5368709, 2 MHz: 10737418, 3 MHz: 16106127, 4 MHz: 21474836, 8 MHz: 42949673 MHz, 20 MHz: 107374182, Max: 0x3FFF
    setFrequencyWord(0, 536871); // freq 100kHz - period 1 us
    setFrequencyWord(1, 0x3FFF);
    selectFrequency(0);
              
    
    setPhaseWord(0, 0); // 0 degrees
    setPhaseWord(1, 2048); // 180 degrees
    selectPhase(0);
              
   
    
    setSignOutput(SIGN_OUTPUT_MSB);
    SQUARE = 1;
    
    
    
    
    // BPSK test
    
    while(1){
        
        selectPhase(0);
        wait_us(100); // send a symbol during 10 T; T = 1/fc; T = 10us = 1/100kHz
        
        selectPhase(1);
        wait_us(100);
               
        
        }
      
    
}
