    /// @file Watchdog.cpp provides the interface to the Watchdog module
///
/// This provides basic Watchdog service for the mbed. You can configure
/// various timeout intervals that meet your system needs. Additionally,
/// it is possible to identify if the Watchdog was the cause of any 
/// system restart.
/// 
/// Adapted from Simon's Watchdog code from http://mbed.org/forum/mbed/topic/508/
///
/// @note Copyright &copy; 2011 by Smartware Computing, all rights reserved.
///     This software may be used to derive new software, as long as
///     this copyright statement remains in the source file.
/// @author David Smart
///
/// \li v2.10 - 20160914: Changed TARGET_STM by mutech, t.kuroki

#include "Watchdog.h"

#if defined( TARGET_LPC1768 )
/// Watchdog gets instantiated at the module level
Watchdog::Watchdog() {
    wdreset = (LPC_WDT->WDMOD >> 2) & 1;    // capture the cause of the previous reset
}

/// Load timeout value in watchdog timer and enable
void Watchdog::Configure(float s) {
    LPC_WDT->WDCLKSEL = 0x1;                // Set CLK src to PCLK
    uint32_t clk = SystemCoreClock / 16;    // WD has a fixed /4 prescaler, PCLK default is /4
    LPC_WDT->WDTC = (uint32_t)(s * (float)clk);
    LPC_WDT->WDMOD = 0x3;                   // Enabled and Reset
    Service();
}

void Watchdog::Configure(int ms) {
    LPC_WDT->WDCLKSEL = 0x1;                // Set CLK src to PCLK
    uint32_t clk = SystemCoreClock / 1000;  // 
    LPC_WDT->WDTC = (ms * clk) / 16;        // WD has a fixed /4 prescaler, PCLK default is /4
    LPC_WDT->WDMOD = 0x3;                   // Enabled and Reset
    Service();
}

/// "Service", "kick" or "feed" the dog - reset the watchdog timer
/// by writing this required bit pattern
void Watchdog::Service() {
    LPC_WDT->WDFEED = 0xAA;
    LPC_WDT->WDFEED = 0x55;
}

/// get the flag to indicate if the watchdog causes the reset
bool Watchdog::WatchdogCausedReset() {
    return wdreset;
}
#elif defined( TARGET_LPC4088 )
// from Gesotec Gesotec
/// Watchdog gets instantiated at the module level
Watchdog::Watchdog() {
    wdreset = (LPC_WDT->MOD >> 2) & 1;    // capture the cause of the previous reset
}
 
/// Load timeout value in watchdog timer and enable
void Watchdog::Configure(float s) {
    //LPC_WDT->CLKSEL = 0x1;                // Set CLK src to PCLK
    uint32_t clk = 500000 / 4;    // WD has a fixed /4 prescaler, and a 500khz oscillator
    LPC_WDT->TC = (uint32_t)(s * (float)clk);
    LPC_WDT->MOD = 0x3;                   // Enabled and Reset
    Service();
}

void Watchdog::Configure(int ms) {
    //LPC_WDT->CLKSEL = 0x1;                // Set CLK src to PCLK
    uint32_t clk = 500000 / 4;    // WD has a fixed /4 prescaler, and a 500khz oscillator
    LPC_WDT->TC = (ms * clk) / 1000;
    LPC_WDT->MOD = 0x3;                   // Enabled and Reset
    Service();
}

/// "Service", "kick" or "feed" the dog - reset the watchdog timer
/// by writing this required bit pattern
void Watchdog::Service() {
    LPC_WDT->FEED = 0xAA;
    LPC_WDT->FEED = 0x55;
}
 
/// get the flag to indicate if the watchdog causes the reset
bool Watchdog::WatchdogCausedReset() {
    return wdreset;
}
#elif defined(TARGET_STM)
Watchdog::Watchdog()
{
    wdreset = (RCC->CSR & (1<<29)) ? true : false;  // read the IWDGRSTF (Independent WD, not the windows WD)
}

// 整数Xを含む最小のべき乗指数
int Watchdog::calcExponent16bit(uint16_t v)
{
//  return (v == 0) ? 0 : MSB16bit(v - 1) + 1;
    if (!v)
        return 0;
    --v;
// 最大有効ビット数（MSB：Most Significant Bit）
    v |= (v >> 1);
    v |= (v >> 2);
    v |= (v >> 4);
    v |= (v >> 8);
//  return count16bit(v) - 1;
// 立っているビットの数を数える
    v = (v & 0x5555) + ((v >> 1) & 0x5555);
    v = (v & 0x3333) + ((v >> 2) & 0x3333);
    v = (v & 0x0f0f) + ((v >> 4) & 0x0f0f);
    return  (v & 0x00ff) + ((v >> 8) & 0x00ff);
}

#if defined(TARGET_STM32F0)
    #define WDT_CLOCK     40000U      // 40 kHz    
#else
    #define WDT_CLOCK     32768U      // 32.768 kHz
#endif

/// Load timeout value in watchdog timer and enable
void Watchdog::Configure(float s)
{
    // http://www.st.com/web/en/resource/technical/document/reference_manual/CD00171190.pdf

    // Newer Nucleo boards have 32.768 kHz crystal. Without it, the internal 
                                // RC clock would have an average frequency of 40 kHz (variable between 30 and 60 kHz)
    uint32_t tick = (uint32_t)(s * WDT_CLOCK + 0.5f);
    // The RLR register is 12 bits and beyond that a prescaler should be used
    int scale = calcExponent16bit((tick + 4095) >> 12);
    if (scale < 2)
        scale = 2;
    else if (scale > 8)         // STM32 allows a maximum time of around 26.2 seconds for the Watchdog timer
        scale = 8;

    int residual = tick / (1 << scale);   // The value for the RLR register
    if (residual < 1)
        residual = 1;
    else if (residual > 4096)
        residual = 4096;

    IWDG->KR  = 0x5555;         // enable write to PR, RLR
    IWDG->PR  = scale - 2;      // Prescaler has values of multiples of 4 (i.e. 2 ^2), page 486 Reference Manual
    IWDG->RLR = residual - 1;   // Init RLR
    IWDG->KR  = 0xAAAA;         // Reload the watchdog
    IWDG->KR  = 0xCCCC;         // Starts the WD
}

void Watchdog::Configure(int ms)
{
    // http://www.st.com/web/en/resource/technical/document/reference_manual/CD00171190.pdf

    // Newer Nucleo boards have 32.768 kHz crystal. Without it, the internal 
    // RC clock would have an average frequency of 40 kHz (variable between 30 and 60 kHz)
//  tick = (ms / (1/WDT_CLOCK))/1000;
    uint32_t tick = ((uint32_t)ms * WDT_CLOCK + 500U) / 1000U;
    // The RLR register is 12 bits and beyond that a prescaler should be used
    int scale = calcExponent16bit((tick + 4095) >> 12);
    if (scale < 2)
        scale = 2;
    else if (scale > 8)         // STM32 allows a maximum time of around 26.2 seconds for the Watchdog timer
        scale = 8;

    int residual = tick / (1 << scale);   // The value for the RLR register
    if (residual < 1)
        residual = 1;
    else if (residual > 4096)
        residual = 4096;

    IWDG->KR  = 0x5555;         // enable write to PR, RLR
    IWDG->PR  = scale - 2;      // Prescaler has values of multiples of 4 (i.e. 2 ^2), page 486 Reference Manual
    IWDG->RLR = residual - 1;   // Init RLR
    IWDG->KR  = 0xAAAA;         // Reload the watchdog
    IWDG->KR  = 0xCCCC;         // Starts the WD
}

/// "Service", "kick" or "feed" the dog - reset the watchdog timer
void Watchdog::Service()
{
    IWDG->KR  = 0xAAAA;
}

/// get the flag to indicate if the watchdog causes the reset
bool Watchdog::WatchdogCausedReset()
{
    if (wdreset)
    {
        RCC->CSR |= (1<<24); // clear reset flag
    }
    return wdreset;
}
#endif
