/* WS281XU.cpp (for LPC82X/STM32F0x/STM32F746xx)
 * mbed Microcontroller Library
 * Copyright (c) 2016 muetch, t.kuroki
 * Allrights reserved.
 *
 * Rev 0.97 2016-09-07
 * Rev 0.98 2016-09-08
 */

#include "WS281XU.h"
#if defined(TARGET_STM)
#include "pinmap.h"
#endif

// TARGET_STM32F7
// TARGET_DISCO_F746NG
// TARGET_NUCLEO_F746ZG
// TARGET_NUCLEO_F446RE
// TARGET_STM32F0
// TARGET_NUCLEO_F030R8
// TARGET_NUCLEO_F070RB
// TARGET_NUCLEO_F072RB
// TARGET_NUCLEO_F091RC
// TARGET_LPC82X
// TARGET_LPC824

//----------------------------------------------------------------------------
/* USART は下位ビットから送信
 * 800k*3 = 2.4Mbps, N, 7, 1
 * start_bit + 7bit_data + stop_bit = 9bit
 * 出力BITは反転させる
 * WS2811(RGB)/WS2812'GRB)は最上位ビットから送信
 */

#define _0  0b110
#define _1  0b100

#define _B(b2,b1,b0)    ((((b2)|((b1)<<3)|((b0)<<6)) >> 1) & 0x7f)

static const uint8_t wstab[8] =
{
    _B(_0, _0, _0),
    _B(_0, _0, _1),
    _B(_0, _1, _0),
    _B(_0, _1, _1),
    _B(_1, _0, _0),
    _B(_1, _0, _1),
    _B(_1, _1, _0),
    _B(_1, _1, _1)
};

//----------------------------------------------------------------------------
WS281XU::WS281XU(PinName txPin, PinMode pinMode, int maxPixels, RGBOrder order)
    : RGBPixels(maxPixels), _serial(), _irq_enabled(false)
{
    pin_init(txPin, pinMode);
    rgbOrder(order);
    show(CL_BLACK);
}

WS281XU::WS281XU(PinName txPin, PinMode pinMode,
        RGBColor *buffer, int maxPixels, RGBOrder order)
    : RGBPixels(buffer, maxPixels), _serial(), _irq_enabled(false)
{
    pin_init(txPin, pinMode);
    rgbOrder(order);
    show(CL_BLACK);
}

WS281XU::~WS281XU()
{
    serial_free(&_serial);
}

void WS281XU::pin_init(PinName txPin, PinMode pinMode)
{
    _txPin = txPin;
    serial_init(&_serial, _txPin, NC);
    serial_baud(&_serial, 800000*3);
    serial_format(&_serial, 7, ParityNone, 1);
#if defined(TARGET_STM)
    pin_mode_ex(txPin, pinMode);
#endif
}

#if defined(TARGET_STM)
/**
 * Configure pin pull-up/pull-down/OpenDrain
 * typedef enum {
 *     PullNone  = 0,
 *     PullUp    = 1,
 *     PullDown  = 2,
 *     OpenDrain = 3,
 *     PullDefault = PullNone
 * } PinMode;
 */
void WS281XU::pin_mode_ex(PinName pin, PinMode mode)
{
    int port_index = STM_PORT(pin);
    int pin_index = STM_PIN(pin);
    int offset = pin_index << 1;
    GPIO_TypeDef * port_reg = ((GPIO_TypeDef *) (GPIOA_BASE + (port_index << 10)));

    // Configure pull-up/pull-down resistors
    uint32_t pupd = (uint32_t)mode & 3;
    if (pupd > 2)
        pupd = 0; // Open-drain = No pull-up/No pull-down

    if (mode == OpenDrain)
    {
        port_reg->PUPDR &= ~(0x3 << offset);    // Open-drain = No pull-up/No pull-down
        port_reg->OTYPER |= 1 << pin_index;
    }
    else
    {
        port_reg->OTYPER &= ~(1 << pin_index);
//      pin_mode(pin, mode);
        port_reg->PUPDR &= ~(0x3 << offset);
        port_reg->PUPDR |= (mode & 0x03) << offset;
    }
}
#endif

WS281XU::RGBOrder WS281XU::rgbOrder(RGBOrder order)
{
    switch(_rgb_order = order)
    {
        case RGB: _1st = 0; _2nd = 1; _3rd = 2; break;  // WS2811
        case RBG: _1st = 0; _2nd = 2; _3rd = 1; break;
        case GRB: _1st = 1; _2nd = 0; _3rd = 2; break;  // WS2812
        case GBR: _1st = 2; _2nd = 0; _3rd = 1; break;
        case BRG: _1st = 1; _2nd = 2; _3rd = 0; break;
        case BGR: _1st = 2; _2nd = 1; _3rd = 0; break;
        default:
            _1st = 0; _2nd = 1; _3rd = 2;
            _rgb_order = GRB;    // WS2812
            break;
    }
    return _rgb_order;
}

void WS281XU::show()
{
    if (!_pixels)
        return;

    uint8_t *pix = (uint8_t *)_pixels;
    uint8_t *end = pix + (_num_pixels * sizeof(_pixels[0]));

    if (!_irq_enabled)
        __disable_irq();   // Disable interrupts temporarily because we don't want our pulse timing to be messed up.

    uint32_t value;
    do
    {
        value  = pix[_1st];
        putByte(wstab[(value >> 5) & 7]);
        putByte(wstab[(value >> 2) & 7]);

        value = (value << 8) | pix[_2nd];
        putByte(wstab[(value >> 7) & 7]);
        putByte(wstab[(value >> 4) & 7]);
        putByte(wstab[(value >> 1) & 7]);

        value = (value << 8) | pix[_3rd];
        putByte(wstab[(value >> 6) & 7]);
        putByte(wstab[(value >> 3) & 7]);
        putByte(wstab[value & 7]);

        pix += sizeof(_pixels[0]);
    } while (pix < end);

    if (!_irq_enabled)
        __enable_irq();   // Re-enable interrupts now that we are done.

//@-    wait_us(50);
}

// 指定色でバッファを埋めた後表示
void WS281XU::show(const RGBColor color)
{
    fill(color);
    show();
}

//----------------------------------------------------------------------------
