/* ColorLib.h
 * mbed Microcontroller Library
 * Copyright (c) 2016 muetch, t.kuroki, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#pragma once

#ifndef COLORLIB_H
#define COLORLIB_H

#include <stdint.h>

//----------------------------------------------------------------------------
#define RGB_MAX_VAL         255
#define HSV_MAX_HUE         3600
#define HSV_MAX_SAT         255
#define HSV_MAX_VAL         255

#define GetRValue(rgb)      ((uint8_t)(rgb))
#define GetGValue(rgb)      ((uint8_t)((rgb) >> 8))
#define GetBValue(rgb)      ((uint8_t)((rgb) >> 16))
#define COLORREF(r, g ,b)   ((uint32_t)((uint8_t)(r) | ((uint8_t)(g) << 8) | ((uint8_t)(b) << 16)))

enum SystemColor
{
    CL_BLACK             = COLORREF(0x00, 0x00, 0x00),
    CL_RED               = COLORREF(0xFF, 0x00, 0x00),
    CL_LIME              = COLORREF(0x00, 0xFF, 0x00),
    CL_YELLOW            = COLORREF(0xFF, 0xFF, 0x00),
    CL_BLUE              = COLORREF(0x00, 0x00, 0xFF),
    CL_FUCHSIA           = COLORREF(0xFF, 0x00, 0xFF),
    CL_MAGENTA           = COLORREF(0xFF, 0x00, 0xFF),
    CL_AQUA              = COLORREF(0x00, 0xFF, 0xFF),
    CL_CYAN              = COLORREF(0x00, 0xFF, 0xFF),
    CL_WHITE             = COLORREF(0xFF, 0xFF, 0xFF),

    CL_MAROON            = COLORREF(0x80, 0x00, 0x00),
    CL_GREEN             = COLORREF(0x00, 0x80, 0x00),
    CL_OLIVE             = COLORREF(0x80, 0x80, 0x00),
    CL_NAVY              = COLORREF(0x00, 0x00, 0x80),
    CL_PURPLE            = COLORREF(0x80, 0x00, 0x80),
    CL_TEAL              = COLORREF(0x00, 0x80, 0x80),
    CL_GRAY              = COLORREF(0x80, 0x80, 0x80),

    CL_GOLD              = COLORREF(0xFF, 0xD7, 0x00),
    CL_SILVER            = COLORREF(0xC0, 0xC0, 0xC0),

    CL_IVORY             = COLORREF(0xFF, 0xFF, 0xF0),
    CL_LIGHTBLUE         = COLORREF(0xAD, 0xD8, 0xE6),
    CL_LIGHTCYAN         = COLORREF(0xE0, 0xFF, 0xFF),
    CL_LIGHTGREEN        = COLORREF(0x90, 0xEE, 0x90),
    CL_LIGHTGRAY         = COLORREF(0xD3, 0xD3, 0xD3),
    CL_LIGHTPINK         = COLORREF(0xFF, 0xB6, 0xC1),
    CL_LIGHTYELLOW       = COLORREF(0xFF, 0xFF, 0xE0),
    CL_PINK              = COLORREF(0xFF, 0xC0, 0xCB),
    CL_SKYBLUE           = COLORREF(0x87, 0xCE, 0xEB),
    CL_VIOLET            = COLORREF(0xEE, 0x82, 0xEE),
    CL_YELLOWGREEN       = COLORREF(0x9A, 0xCD, 0x32),
    CL_GREENYELLOW       = COLORREF(0xAD, 0xFF, 0x2F),
    CL_CANDLE            = COLORREF(255, 147,  41),  /*  1900 K */
    CL_TUNGSTEN          = COLORREF(255, 197, 143),  /*  2600 K */
    CL_HALOGEN           = COLORREF(255, 241, 224),  /*  3200 K */
};

//----------------------------------------------------------------------------
struct RGBColor;
struct HSVColor;

void rgb2hsv(RGBColor const& rgb, HSVColor& hsv);
void hsv2rgb(HSVColor const& hsv, RGBColor& rgb);

//----------------------------------------------------------------------------
/**
    RGB Color
*/
struct RGBColor
{
    union
    {
        struct
        {
            union {
                uint8_t r;
                uint8_t red;
            };
            union {
                uint8_t g;
                uint8_t green;
            };
            union {
                uint8_t b;
                uint8_t blue;
            };
        };
        uint8_t raw[3];
    };

    /**
        Constructor with rgb initializing

        @param r - the red byte
        @param g - the green byte
        @param b - the blue byte
    */
    RGBColor(uint8_t r, uint8_t g, uint8_t b)
    {
        red   = r;
        green = g;
        blue  = b;
    }

    RGBColor(int r, int g, int b)
    {
        red   = (r < 0) ? 0 : ((RGB_MAX_VAL < r) ? RGB_MAX_VAL : r);
        green = (g < 0) ? 0 : ((RGB_MAX_VAL < g) ? RGB_MAX_VAL : g);
        blue  = (b < 0) ? 0 : ((RGB_MAX_VAL < b) ? RGB_MAX_VAL : b);
    }

    RGBColor(const int rgb)
    {
        red   = GetRValue(rgb);
        green = GetGValue(rgb);
        blue  = GetBValue(rgb);
    }

    RGBColor(const HSVColor& hsv)
    {
        hsv2rgb(hsv, *this);
    }

    RGBColor(HSVColor& hsv)
    {
        hsv2rgb(hsv, *this);
    }

    /**
        Default constructor
    */
    RGBColor() {}

    // allow copy construction
    RGBColor(const RGBColor& rhs)
    {
        red   = rhs.red;
        green = rhs.green;
        blue  = rhs.blue;
    }

    // allow assignment from one RGB struct to another
    RGBColor& operator= (const RGBColor& rhs)
    {
        red   = rhs.red;
        green = rhs.green;
        blue  = rhs.blue;
        return *this;
    }

    RGBColor& operator= (const HSVColor& rhs)
    {
        hsv2rgb(rhs, *this);
        return *this;
    }

    RGBColor& operator= (const int rgb)
    {
        red   = GetRValue(rgb);
        green = GetGValue(rgb);
        blue  = GetBValue(rgb);
        return *this;
    }

    operator int()
    {
        return COLORREF(red, green, blue);
    }

};

//----------------------------------------------------------------------------
/**
    HSV Color
*/
struct HSVColor
{
    union
    {
        struct
        {
            union {
                int16_t hue;
                int16_t h;
            };
            union {
                uint8_t saturation;
                uint8_t sat;
                uint8_t s;
            };
            union {
                uint8_t value;
                uint8_t val;
                uint8_t v;
            };
        };
        uint8_t  raw[4];
        uint32_t code;
    };

    int nomalize_hue(int h)
    {
        if (h < 0)
            h = HSV_MAX_HUE - (-h % HSV_MAX_HUE);
        return h % HSV_MAX_HUE;
    }

    /**
        Constructor with hsv initializing

        @param h - the hue byte
        @param s - the sat byte
        @param v - the val byte
    */
    HSVColor(int h, int s, int v)
    {
//        hue = nomalize_hue(h);
        hue = h;
        sat = (s < 0) ? 0 : ((HSV_MAX_SAT < s) ? HSV_MAX_SAT : s);
        val = (v < 0) ? 0 : ((HSV_MAX_VAL < v) ? HSV_MAX_VAL : v);
    }

    HSVColor(int16_t h, uint8_t s, uint8_t v)
    {
//        hue = nomalize_hue(h);
        hue = h;
        sat = s;
        val = v;
    }

    HSVColor(const RGBColor& rgb)
    {
        rgb2hsv(rgb, *this);
    }

    HSVColor(RGBColor& rgb)
    {
        rgb2hsv(rgb, *this);
    }

    HSVColor(const int rgbcolor)
    {
        RGBColor rgb(rgbcolor);
        rgb2hsv(rgb, *this);
    }

    /**
        Default constructor
    */
    HSVColor() {}

    // allow copy construction
    HSVColor(const HSVColor& rhs)
    {
        hue = rhs.hue;
        sat = rhs.sat;
        val = rhs.val;
    }

    // allow assignment from one hsv struct to another
    HSVColor& operator= (const HSVColor& rhs)
    {
        hue = rhs.hue;
        sat = rhs.sat;
        val = rhs.val;
        return *this;
    }

    HSVColor& operator= (const RGBColor& rhs)
    {
        rgb2hsv(rhs, *this);
        return *this;
    }

    HSVColor& operator= (const int& rhs)
    {
        RGBColor rgb(rhs);
        rgb2hsv(rhs, *this);
        return *this;
    }

    operator int()
    {
        RGBColor rgb(*this);
        return (int)rgb;
    }

};

//----------------------------------------------------------------------------
extern const uint8_t gamma20_table[256];

//uint8_t GetGammaTable();
#define GetGammaTable() (gamma20_table)
RGBColor GammaColor(RGBColor color);
RGBColor GammaColor(RGBColor color, int brightness);
RGBColor* GammaCorrection(RGBColor *color, int size);
HSVColor* GammaCorrection(HSVColor *color, int size);

//----------------------------------------------------------------------------
#endif      // end of COLORLIB_H
