/* ColorLib.cpp
 * mbed Microcontroller Library
 * Copyright (c) 2016 muetch, t.kuroki, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <stdlib.h>
#include "ColorLib.h"

//----------------------------------------------------------------------------
/*
#define RGB_MAX_VAL     255
#define HSV_MAX_HUE     3600
#define HSV_MAX_SAT     255
#define HSV_MAX_VAL     255
*/

#define MAX_COLOR(r, g, b)  (((r) > (g)) ? (((r) > (b)) ? (r) : (b)) : (((g) > (b)) ? (g) : (b)))
#define MIN_COLOR(r, g, b)  (((r) < (g)) ? (((r) < (b)) ? (r) : (b)) : (((g) < (b)) ? (g) : (b)))

inline __attribute__((always_inline))
static int MulDiv(int nNumber, int nNumerator, int nDenominator)
{
    return (nNumber * nNumerator) / nDenominator;
}

//#define constrain(amt, low, high) ((amt) < (low) ? (low) : ((amt) > (high) ? (high) : (amt)))
template<class T>
__attribute__((always_inline))
inline const T& constrain(const T& x, const T& a, const T& b)
{
    return (x < a) ? a : ((b < x) ? b : x);
}

//----------------------------------------------------------------------------
// Integer version

void rgb2hsv(RGBColor const& rgb, HSVColor& hsv)
{
    int r, g, b;        // rgb : 0 - 255
    int h, s, v;        // h: 0 - 3600, sv: 0 - 255
    int min, max, delta;

    r = rgb.r;
    g = rgb.g;
    b = rgb.b;

    max = MAX_COLOR(r, g, b);
    min = MIN_COLOR(r, g, b);
    delta = max - min;

    v = MulDiv(max, HSV_MAX_VAL, RGB_MAX_VAL);
    s = (max == 0) ? 0 : MulDiv(delta, HSV_MAX_SAT, max);

    if (s == 0)
        h = 0;
    else
    {
#define MAXHUE_DIV_6    (int)(HSV_MAX_HUE/6.00f+0.5f)
#define MAXHUE_1of3     (int)(HSV_MAX_HUE/3.00f+0.5f)
#define MAXHUE_2of3     (int)(2.0f*HSV_MAX_HUE/3.00f+0.5f)
        if (r >= max)                       // > is bogus, just keeps compilor happy
            h = MulDiv(g - b, MAXHUE_DIV_6, delta);         // between yellow & magenta
        else if (g >= max)
            h = MAXHUE_1of3 + MulDiv(b - r, MAXHUE_DIV_6, delta);   // between cyan & yellow
        else
            h = MAXHUE_2of3 + MulDiv(r - g, MAXHUE_DIV_6, delta);   // between magenta & cyan

        if (h < 0)
            h += HSV_MAX_HUE;
    }
    hsv.h = h;
    hsv.s = s;
    hsv.v = v;
}

void hsv2rgb(HSVColor const& hsv, RGBColor& rgb)
{
    const int DIVISOR = 255 * (HSV_MAX_HUE/6);
    int h, s, v;
    int p, q, t;

    h = hsv.h;
    s = hsv.s;
    v = hsv.v;

    //  H = 0 to 3600 (corresponding to 0..3600 degrees around hexcone)
    //      0 (undefined) for S = 0
    //  S = 0 (shade of gray) to 255 (pure color)
    //  V = 0 (black) to 255 (white)

    if (s <= 0 || v <= 0)       // < is bogus, just shuts up warnings
    {
        if (v < 0)
            v = 0;
        rgb.r = rgb.g = rgb.b = v;
        return;
    }

    if (h < 0)
        h = HSV_MAX_HUE - (-h % HSV_MAX_HUE);
    h %= HSV_MAX_HUE;

    div_t hh = div(h, (HSV_MAX_HUE/6)); // quot:0..6, rem:0..599
    int vs = v * s;
    p = v - vs / 255;                                       // p = v * (1 - s)
    q = v - MulDiv(vs, hh.rem, DIVISOR);                    // q = v * (1 - s*f)
    t = v - MulDiv(vs, (HSV_MAX_HUE/6) - hh.rem, DIVISOR);  // t = v * (1 - s * (1 - f))

    switch (hh.quot)
    {
        case  0: rgb.r = v; rgb.g = t; rgb.b = p; break;
        case  1: rgb.r = q; rgb.g = v; rgb.b = p; break;
        case  2: rgb.r = p; rgb.g = v; rgb.b = t; break;
        case  3: rgb.r = p; rgb.g = q; rgb.b = v; break;
        case  4: rgb.r = t; rgb.g = p; rgb.b = v; break;
        default: rgb.r = v; rgb.g = p; rgb.b = q; break;
    }
}

//----------------------------------------------------------------------------
//----------------------------------------------------------------------------
// Gamma 2.0
const uint8_t gamma20_table[256] =
{
      0,   1,   1,   1,   1,   1,   1,   1,  1,   1,   1,   1,   1,   2,   2,   2,
      2,   2,   2,   2,   2,   3,   3,   3,  3,   3,   3,   4,   4,   4,   4,   5,
      5,   5,   5,   6,   6,   6,   6,   7,  7,   7,   8,   8,   8,   9,   9,   9,
     10,  10,  10,  11,  11,  12,  12,  12, 13,  13,  14,  14,  15,  15,  16,  16,
     17,  17,  18,  18,  19,  19,  20,  20, 21,  21,  22,  23,  23,  24,  24,  25,
     26,  26,  27,  27,  28,  29,  29,  30, 31,  31,  32,  33,  34,  34,  35,  36,
     37,  37,  38,  39,  40,  40,  41,  42, 43,  44,  44,  45,  46,  47,  48,  49,
     50,  50,  51,  52,  53,  54,  55,  56, 57,  58,  59,  60,  61,  62,  63,  64,
     65,  66,  67,  68,  69,  70,  71,  72, 73,  74,  75,  76,  77,  78,  79,  80,
     82,  83,  84,  85,  86,  87,  88,  90, 91,  92,  93,  94,  96,  97,  98,  99,
    101, 102, 103, 104, 106, 107, 108, 109,111, 112, 113, 115, 116, 117, 119, 120,
    122, 123, 124, 126, 127, 129, 130, 131,133, 134, 136, 137, 139, 140, 142, 143,
    145, 146, 148, 149, 151, 152, 154, 155,157, 158, 160, 162, 163, 165, 166, 168,
    170, 171, 173, 175, 176, 178, 180, 181,183, 185, 186, 188, 190, 192, 193, 195,
    197, 199, 200, 202, 204, 206, 207, 209,211, 213, 215, 217, 218, 220, 222, 224,
    226, 228, 230, 232, 233, 235, 237, 239,241, 243, 245, 247, 249, 251, 253, 255
};

//----------------------------------------------------------------------------
#define GAMMA(c)    gamma20_table[c]
//#define GAMMA(c)    ((((c)+1)*((c)+1)-1)>>8)

RGBColor GammaColor(RGBColor color)
{
    RGBColor ret;

    ret.red   = GAMMA(color.red);
    ret.green = GAMMA(color.green);
    ret.blue  = GAMMA(color.blue);
    return ret;
}

RGBColor GammaColor(RGBColor color, int brightness)
{
    HSVColor hsv(color);
    hsv.val = (brightness < 0) ? 0 : (brightness > 255) ? 255 : brightness;
    return GammaColor(hsv);
}

RGBColor* GammaCorrection(RGBColor *color, int size)
{
    if (color && size > 0)
    {
        RGBColor *p = color;
        for (int i = 0; i < size; ++i)
        {
            p->red   = GAMMA(p->red);
            p->green = GAMMA(p->green);
            p->blue  = GAMMA(p->blue);
            ++p;
        }
    }
    return color;
}

HSVColor* GammaCorrection(HSVColor *color, int size)
{
    if (color && size > 0)
    {
        HSVColor *p = color;
        for (int i = 0; i < size; ++i)
        {
            p->val = GAMMA(p->val);
            ++p;
        }
    }
    return color;
}

//----------------------------------------------------------------------------
