/* PixelBuffer.h
 * mbed Microcontroller Library
 * Copyright (c) 2016 muetch, t.kuroki
 * Allrights reserved.
 *
 * Rev 0.97 2016-09-07
 * Rev 0.98 2016-09-08
 */

#pragma once

#ifndef PIXELBUFFER_H
#define PIXELBUFFER_H

#ifdef _WIN32
#include <stdint.h>
#include <string.h>
#else
#include "mbed.h"
#endif
#include "ColorLib.h"

//----------------------------------------------------------------------------
#ifndef MAX_PIXELS
#define MAX_PIXELS      170
#endif

#ifndef nullptr
#define nullptr         (0)
#endif

//----------------------------------------------------------------------------
/**
 * RGBPixels
 */
class RGBPixels
{
public:
    /**
    *   Initializes the addressable led bus
    *
    *   @param Buffer - The Pixel array buffer address.
    *   @param maxPixels - Number of the addressable leds
    */
    RGBPixels(RGBColor *buffer = nullptr, int maxPixels = MAX_PIXELS);
    RGBPixels(int maxPixels = MAX_PIXELS);
    virtual ~RGBPixels();

    void setPixelBuffer(RGBColor *buffer, int maxPixels);
    int  maxPixels() { return _max_pixels; }
    int  numPixels(int value = -1);

    void setPixels(int index, RGBColor *color, int len);
    void setPixels(int index, HSVColor *color, int len);
    void setPixels(RGBColor *color, int len) { setPixels(0, color, len); }
    void setPixels(HSVColor *color, int len) { setPixels(0, color, len); }

    void setGammaPixels(int index, RGBColor *color, int len);
    void setGammaPixels(int index, HSVColor *color, int len);
    void setGammaPixels(RGBColor *color, int len) { setGammaPixels(0, color, len); }
    void setGammaPixels(HSVColor *color, int len) { setGammaPixels(0, color, len); }

    void fillPixels(int index, const RGBColor color, int len);
    void fillPixels(int index, const HSVColor color, int len);
    void fillPixels(int index, const int color, int len)        { fillPixels(index, (RGBColor)color, len); }
    void fillPixels(const RGBColor color, int len = MAX_PIXELS) { fillPixels(0, color, len); }
    void fillPixels(const HSVColor color, int len = MAX_PIXELS) { fillPixels(0, color, len); }
    void fillPixels(const int color, int len = MAX_PIXELS)      { fillPixels(0, (RGBColor)color, len); }

    void fill(int index, const RGBColor color, int len)   { fillPixels(index, color, len); }
    void fill(int index, const HSVColor color, int len)   { fillPixels(index, color, len); }
    void fill(int index, const int color, int len)        { fillPixels(index, (RGBColor)color, len); }
    void fill(const RGBColor color, int len = MAX_PIXELS) { fillPixels(0, color, len); }
    void fill(const HSVColor color, int len = MAX_PIXELS) { fillPixels(0, color, len); }
    void fill(const int color, int len = MAX_PIXELS)      { fillPixels(0, (RGBColor)color, len); }

    void clear(const RGBColor color) { fillPixels(color); }
    void clear(const HSVColor color) { fillPixels(color); }
    void clear(const int color = 0)  { fillPixels((RGBColor)color); }

    void repeatPixels(int block_size);
    void repeatPixels(RGBColor *source, int size);
    void repeatPixels(HSVColor *source, int size);

    void makeGradation(int index, RGBColor from, RGBColor to, int len);
    void makeGradation(RGBColor from, RGBColor to, int len = MAX_PIXELS) { makeGradation(0, from, to, len); }

    void makeRainbow(int index, HSVColor color, int len, int direction);
    void makeRainbow(HSVColor color, int len = MAX_PIXELS, int direction = 1) { makeRainbow(0, color, len, direction); }

    RGBColor operator[](int index) const
    {
        if (_pixels && (uint16_t)index < _num_pixels)
            return _pixels[index];
        return _dummy_pixel;
    }

    RGBColor& operator[](int index)
    {
        if (_pixels && (uint16_t)index < _num_pixels)
            return _pixels[index];
        return _dummy_pixel;
    }

    RGBPixels& operator=(const RGBPixels& rhs);

    operator RGBColor*() const { return _pixels; }

protected:
    uint16_t _max_pixels;
    uint16_t _num_pixels;
    RGBColor *_pixels;
    RGBColor _dummy_pixel;

private:
    bool     _owned_buffer;

};

//----------------------------------------------------------------------------
/**
 * HSVPixels
 */
class HSVPixels
{
public:
    /**
    *   Initializes the addressable led bus
    *
    *   @param Buffer - The Pixel array buffer address.
    *   @param maxPixels - Number of the addressable leds
    */
    HSVPixels(HSVColor *buffer = nullptr, int maxPixels = MAX_PIXELS);
    HSVPixels(int maxPixels = MAX_PIXELS);
    virtual ~HSVPixels();

    void setPixelBuffer(HSVColor *buffer, int maxPixels);
    int  maxPixels() { return _max_pixels; }
    int  numPixels(int value = -1);

    void setPixels(int index, HSVColor *color, int len);
    void setPixels(int index, RGBColor *color, int len);
    void setPixels(HSVColor *color, int len) { setPixels(0, color, len); }
    void setPixels(RGBColor *color, int len) { setPixels(0, color, len); }

    void fillPixels(int index, const HSVColor color, int len);
    void fillPixels(int index, const RGBColor color, int len);
    void fillPixels(int index, const int color, int len)        { fillPixels(index, (RGBColor)color, len); }
    void fillPixels(const HSVColor color, int len = MAX_PIXELS) { fillPixels(0, color, len); }
    void fillPixels(const RGBColor color, int len = MAX_PIXELS) { fillPixels(0, color, len); }
    void fillPixels(const int color, int len = MAX_PIXELS)      { fillPixels(0, (RGBColor)color, len); }

    void fill(int index, const HSVColor color, int len)   { fillPixels(index, color, len); }
    void fill(int index, const RGBColor color, int len)   { fillPixels(index, color, len); }
    void fill(int index, const int color, int len)        { fillPixels(index, (RGBColor)color, len); }
    void fill(const HSVColor color, int len = MAX_PIXELS) { fillPixels(0, color, len); }
    void fill(const RGBColor color, int len = MAX_PIXELS) { fillPixels(0, color, len); }
    void fill(const int color, int len = MAX_PIXELS)      { fillPixels(0, (RGBColor)color, len); }

    void clear(const HSVColor color) { fillPixels(color); }
    void clear(const RGBColor color) { fillPixels(color); }
    void clear(const int color = 0)  { fillPixels((RGBColor)color); }

    void repeatPixels(int block_size);
    void repeatPixels(HSVColor *source, int size);
    void repeatPixels(RGBColor *source, int size);

    void makeGradation(int index, HSVColor from, HSVColor to, int len);
    void makeGradation(HSVColor from, HSVColor to, int len = MAX_PIXELS) { makeGradation(0, from, to, len); }

    void makeRainbow(int index, HSVColor color, int len, int direction);
    void makeRainbow(HSVColor color, int len = MAX_PIXELS, int direction = 1) { makeRainbow(0, color, len, direction); }

    HSVColor operator[](int index) const
    {
        if (_pixels && (uint16_t)index < _num_pixels)
            return _pixels[index];
        return _dummy_pixel;
    }

    HSVColor& operator[](int index)
    {
        if (_pixels && (uint16_t)index < _num_pixels)
            return _pixels[index];
        return _dummy_pixel;
    }

    HSVPixels& operator=(const HSVPixels& rhs);

    operator HSVColor*() const { return _pixels; }

protected:
    uint16_t _max_pixels;
    uint16_t _num_pixels;
    HSVColor *_pixels;
    HSVColor _dummy_pixel;

private:
    bool     _owned_buffer;

};

//----------------------------------------------------------------------------
#endif      // end of PIXELBUFFER_H