//Includes
#include "mbed.h"
#include "7366_lib.h"
#include "TLE5206_lib.h"

// Caractéristiques de structure
#define RAYON_ROUE 0.025 // [m]
#define DIST1 0.15 // [m]
#define DIST2 0.15 // [m]
#define DIST3 0.075 // [m]
#define PI 3.14159265359
#define RESOLUTION_ENCO 14336 // = 14*1024 = rapport_reduction * nbr de top par tour
#define ETALONNAGE_LACET 1.083209
#define ETALONNAGE_XY 1.06040

// Liaison SPI avec les compteurs 
#define SPI_SCLK PA_5 //A4
#define SPI_MISO PA_6 //A5
#define SPI_MOSI PA_7 //A6
#define SPI_CS3 PA_0//A0
#define SPI_CS2 PA_1
#define SPI_CS1 PA_3

// Vers le pont en H
#define H1_IN1 PA_9  //D1 - PWM1/2
#define H1_IN2 PA_10 //D0 - PWM1/3
#define H2_IN1 PA_8  //D9 - PWM1/1
#define H2_IN2 PB_7  //D4 - PWM17/1
#define H3_IN1 PA_4  //A3 - PWM3/2
#define H3_IN2 PB_6  //D5 - PWM16/1

#define DECOUP_HACH 50 //us - 20 kHz pour les oreilles
#define PERIODE_AFF 1000 //ms
#define PERIODE_POSITION 1000 //us
#define PERIODE_ASSERV 10 //ms
#define PERIODE_TRAPEZE 10 //ms

// Constantes moteur
#define MOT_SAT_HIGH 0.99
#define MOT_SAT_LOW 0.4
#define MOT_NO_SAT_LOW 0.2

// Constantes Asservissement
#define GAIN_POS 10
#define GAIN_ANG 1
#define GAIN_POS_INT 0.003 // 1/Ti
#define GAIN_ANG_INT 0.003 // 1/Ti
#define ERREUR_POS 0.05 // transition P-PI
#define ERREUR_ANG PI/15

// Constantes Trapeze
#define VITESSE 0.6 // m/s
#define ACCELERATION 0.2 // m/s^2

Serial pc(USBTX,USBRX);
Timer timer;
DigitalOut myled(LED3);

SPI_7366 compt1(SPI_MOSI, SPI_MISO, SPI_SCLK,SPI_CS1);
SPI_7366 compt2(SPI_MOSI, SPI_MISO, SPI_SCLK,SPI_CS2);
SPI_7366 compt3(SPI_MOSI, SPI_MISO, SPI_SCLK,SPI_CS3);


TLE5206 moteur1(H1_IN1,H1_IN2);
TLE5206 moteur2(H2_IN1,H2_IN2);
TLE5206 moteur3(H3_IN1,H3_IN2);

struct Vect3{
    double x;
    double y;
    double z;
};

struct Vect3int{
    int x;
    int y;
    int z;
};

struct Vect2{
    float x;
    float y;
};

Vect3 initVect3(){
    Vect3 result;
    result.x = 0;
    result.y = 0;
    result.z = 0;
    return result;
}

int sgn(float val) {
    return (0.0 < val) - (val < 0.0);
}

Vect3 calculatePosition(){
    static Vect3 theta = initVect3();
    static Vect3 position = initVect3();
    Vect3 dTheta;
    double dPsi = 0;
    
    dTheta.x = 2*PI/RESOLUTION_ENCO*compt1.read_value() - theta.x;
    dTheta.y = 2*PI/RESOLUTION_ENCO*compt2.read_value() - theta.y;
    dTheta.z = 2*PI/RESOLUTION_ENCO*compt3.read_value() - theta.z;
    theta.x += dTheta.x;
    theta.y += dTheta.y;
    theta.z += dTheta.z;
    
    dPsi = ETALONNAGE_LACET*RAYON_ROUE*(dTheta.x+dTheta.y+dTheta.z)/(DIST1+DIST2+DIST3);

    position.z += dPsi;// Psi actuel
    
    position.x += -ETALONNAGE_XY*(
                     (RAYON_ROUE*dTheta.x - DIST1*dPsi)*sin(position.z+4*PI/3)
                    +(RAYON_ROUE*dTheta.y - DIST2*dPsi)*sin(position.z+2*PI/3)
                    +(RAYON_ROUE*dTheta.z - DIST3*dPsi)*sin(position.z+0*PI/3)
                    )*2/3;
    position.y += +ETALONNAGE_XY*(
                     (RAYON_ROUE*dTheta.x - DIST1*dPsi)*cos(position.z+4*PI/3)
                    +(RAYON_ROUE*dTheta.y - DIST2*dPsi)*cos(position.z+2*PI/3) 
                    +(RAYON_ROUE*dTheta.z - DIST3*dPsi)*cos(position.z+0*PI/3)
                    )*2/3;
    
    return position;
}

Vect3 calcErreur(Vect3 consigne, Vect3 position){
    Vect3 erreur;
    erreur.x = position.x - consigne.x;
    erreur.y = position.y - consigne.y;
    erreur.z = position.z - consigne.z;
    return erreur;
}

Vect3 calcCommandeXYZ(Vect3 erreur){
    
    static Vect3 integrateur = initVect3();
    Vect3 commande;
    if (abs(erreur.x) > ERREUR_POS){
        integrateur.x = 0;
    } else {
        integrateur.x = integrateur.x + PERIODE_ASSERV*erreur.x;
    }
    commande.x = GAIN_POS*(erreur.x + GAIN_POS_INT*integrateur.x);
    
    if (abs(erreur.y) > ERREUR_POS){
        integrateur.y = 0;
    } else {
        integrateur.y = integrateur.y + PERIODE_ASSERV*erreur.y;
    }
    commande.y = GAIN_POS*(erreur.y + GAIN_POS_INT*integrateur.y);
    
    if (abs(erreur.z) > ERREUR_ANG){
        integrateur.z = 0;
    } else {
        integrateur.z = integrateur.z + PERIODE_ASSERV*erreur.z;
    }
    commande.z = GAIN_ANG*(erreur.z + GAIN_ANG_INT*integrateur.z);
    
    return commande;
    
}

Vect3 calcCommande123(Vect3 commandeXYZ, Vect3 position){
    Vect3 commande123;
    commande123.x = -commandeXYZ.x*sin(position.z+4*PI/3)
                    +commandeXYZ.y*cos(position.z+4*PI/3)
                    +commandeXYZ.z;
    commande123.y = -commandeXYZ.x*sin(position.z+2*PI/3)
                    +commandeXYZ.y*cos(position.z+2*PI/3)
                    +commandeXYZ.z;
    commande123.z = -commandeXYZ.x*sin(position.z+0*PI/3)
                    +commandeXYZ.y*cos(position.z+0*PI/3)
                    +commandeXYZ.z;
    return commande123;
}

float calcCommande1M(float duty_cycle){
    float cmdMotor;
    if (duty_cycle < -MOT_SAT_HIGH){
        cmdMotor = -MOT_SAT_HIGH;
    }else if (duty_cycle < -MOT_SAT_LOW){
        cmdMotor = duty_cycle;
    }else if (duty_cycle < -MOT_NO_SAT_LOW){
        cmdMotor = -MOT_SAT_LOW;
    }else if (duty_cycle < MOT_NO_SAT_LOW){
        cmdMotor = MOT_SAT_LOW/MOT_NO_SAT_LOW*duty_cycle;
    }else if (duty_cycle < MOT_SAT_LOW){
        cmdMotor = MOT_SAT_LOW;
    }else if (duty_cycle < MOT_SAT_HIGH){
        cmdMotor = duty_cycle;
    }else {
        cmdMotor = MOT_SAT_HIGH;
    }
    return cmdMotor;
}

Vect3 calcCommande3M(Vect3 commande123){
    Vect3 cmd_motor;
    cmd_motor.x = calcCommande1M(commande123.x);    
    cmd_motor.y = calcCommande1M(commande123.y);
    cmd_motor.z = calcCommande1M(commande123.z);
    return cmd_motor;
}

void moveBot(Vect3 commande123){
    moteur1.write(commande123.x); 
    moteur2.write(commande123.y);
    moteur3.write(commande123.z);  
}

int trapezeMAE(int etat, float position, float positionVoulue, float vitesse, float vitesseMax){
    float distanceFreinage = 0.5*vitesse*vitesse/ACCELERATION;
    int sensX = sgn(positionVoulue-position);
    int sensV = sgn(vitesse);
    float vitesseAbs = abs(vitesse);
    float distance = abs(position-positionVoulue);
    switch (etat){
        case 0 : // Arret
            if (position != positionVoulue)
                etat = 1;
            break;
        case 1 : // Accélération
            if (sensX == sensV){
                if (distance > distanceFreinage && vitesseAbs < vitesseMax){
                    if (vitesseAbs == vitesseMax){
                        etat = 2;
                    }
                }else{
                    etat = 3;    
                } 
            }
            break;
        case 2 : // Vitesse Constante
            if (sensX == sensV){
                if (distance > distanceFreinage){
                    if (vitesseAbs < vitesseMax){
                        etat = 1;
                    }
                } else {
                    etat = 3;
                }    
            }else{
                etat = 1;
            }
            break;
        case 3 : // Freinage
            if (position != positionVoulue && vitesseAbs != 0){
                if (distance > distanceFreinage){
                    etat = 1;
                }
            }else{
                etat = 0;
            }
            break;
        default :
            etat = 0;
            break;
    }
    return etat;
} 

float trapezeCalcVit(int etat, float position, float positionVoulue, float vitesse, float vitesseMax){
    int sensX = sgn(positionVoulue-position);
    switch (etat) {
        case 0: // Arret
            vitesse = 0;
            break;
        case 1: // Acceleration
            vitesse += sensX*ACCELERATION*PERIODE_TRAPEZE/1000;
            if (abs(vitesse) > vitesseMax){
                vitesse = sensX*vitesseMax;
            }
            break;
        case 2: // Vitesse constante
            break;
        case 3: // Freinage
            vitesse -= sensX*ACCELERATION*PERIODE_TRAPEZE/1000;
            if (sgn(vitesse) !=  sensX){
                vitesse = 0;
            }
            break;
        default:
            vitesse = 0;
            break;
    }
    return vitesse;
}

Vect3 trapeze(Vect3 consigneGlobale, float vitesseMax){
    static Vect3int etats;
    static Vect3 consigneAsservPos = initVect3();
    static Vect3 vitesse = initVect3();
    
    etats.x = trapezeMAE(etats.x,consigneAsservPos.x,consigneGlobale.x,vitesse.x,vitesseMax);
    etats.y = trapezeMAE(etats.y,consigneAsservPos.y,consigneGlobale.y,vitesse.y,vitesseMax);
    etats.z = trapezeMAE(etats.z,consigneAsservPos.z,consigneGlobale.z,vitesse.z,vitesseMax);
    
    vitesse.x = trapezeCalcVit(etats.x,consigneAsservPos.x,consigneGlobale.x,vitesse.x,vitesseMax);
    vitesse.y = trapezeCalcVit(etats.y,consigneAsservPos.y,consigneGlobale.y,vitesse.y,vitesseMax);
    vitesse.z = trapezeCalcVit(etats.z,consigneAsservPos.z,consigneGlobale.z,vitesse.z,vitesseMax);
    
    consigneAsservPos.x += vitesse.x*PERIODE_TRAPEZE/1000;
    consigneAsservPos.y += vitesse.y*PERIODE_TRAPEZE/1000;
    consigneAsservPos.z += vitesse.z*PERIODE_TRAPEZE/1000;
    
    return consigneAsservPos;
}

int isRobotReady(Vect3 consigneGlobale, Vect3 position){
    return (abs(consigneGlobale.x-position.x) < ERREUR_POS) && (abs(consigneGlobale.y-position.y) < ERREUR_POS) && (abs(consigneGlobale.z-position.z) < ERREUR_ANG);
}

Vect3 trajectoireMAE(){
    Vect3 consigneGlobale = initVect3();
    static int etat = 0;
    switch (etat%4){
         case 0:
            consigneGlobale.x = 0;
            consigneGlobale.y = 0;
            consigneGlobale.z = 0;
            break;
         case 1:
            consigneGlobale.x = -1;
            consigneGlobale.y = 0;
            consigneGlobale.z = 0;
            break;
         case 2:
            consigneGlobale.x = -1;
            consigneGlobale.y = 1;
            consigneGlobale.z = 0;
            break;
         case 3:
            consigneGlobale.x = 0;
            consigneGlobale.y = 1;
            consigneGlobale.z = 0;
            break;
    }   
    etat++;
    return consigneGlobale;
}

int main(){
    
    //setup
    compt1.setup();
    compt2.setup();
    compt3.setup();
    
    compt3.set_as_inverted();
    
    moteur1.setup(DECOUP_HACH);
    moteur2.setup(DECOUP_HACH);
    moteur3.setup(DECOUP_HACH);
    
    timer.start();
    pc.printf("SETUP effectue\n\r");
    
    //variables
    Vect3 position    = initVect3();
    Vect3 erreur      = initVect3();
    Vect3 commandeXYZ = initVect3();
    Vect3 commande123 = initVect3();
    Vect3 cmdMotor123 = initVect3();
    Vect3 consigneGlobale = initVect3();
    Vect3 consigneAsserv  = initVect3();
    
    consigneGlobale.x = 0.0;
    consigneGlobale.y = 0.0;
    consigneGlobale.z = 0.0;

    uint32_t seuilAffichage = PERIODE_AFF;
    uint32_t seuilAsserv    = PERIODE_ASSERV;
    uint32_t seuilPosition  = PERIODE_POSITION;
    uint32_t seuilTrapeze   = PERIODE_TRAPEZE;
    
    // Loop
    while(1) {
        
        // Liaison Série
        if (timer.read_ms() > seuilAffichage){
            seuilAffichage += PERIODE_AFF;
            /*pc.printf("position\tpsi : %.5f\tx : %.5f\ty : %.5f\n\r",180/PI*position.z, position.x, position.y);
            pc.printf("erreur\t\tpsi : %.5f\tx : %.5f\ty : %.5f\n\r",180/PI*erreur.z, erreur.x, erreur.y);
            //pc.printf("commande\t1 : %.5f\t2 : %.5f\t3 : %.5f\n\r",commande123.x, commande123.y, commande123.z);
            pc.printf("cmdMotor\t1 : %.5f\t2 : %.5f\t3 : %.5f\n\r",cmdMotor123.x, cmdMotor123.y, cmdMotor123.z);
            //pc.printf("compt1 : %d\tcompt2 : %d\tcompt3: %d\n\r",compt1.read_value(),compt2.read_value(),compt3.read_value());
            pc.printf("Trapeze\t\tpsi : %.5f\tx : %.5f\ty : %.5f\n\r",180/PI*consigneAsserv.z, consigneAsserv.x, consigneAsserv.y);
            pc.printf("Consigne\tpsi : %.5f\tx : %.5f\ty : %.5f\n\r",180/PI*consigneGlobale.z, consigneGlobale.x, consigneGlobale.y);
            pc.printf("\n");
            myled = !myled;
            */
            if (isRobotReady(consigneGlobale,position)){
                consigneGlobale = trajectoireMAE();
            }
        }
        // Calcul de la position
        if (timer.read_us() > seuilPosition){
            seuilPosition += PERIODE_POSITION;
            position    = calculatePosition();
        }
        // Trapeze
        /*if (timer.read_ms() > seuilTrapeze){
            seuilTrapeze += PERIODE_TRAPEZE;
            consigneAsserv = trapeze(consigneGlobale, VITESSE);    
        }*/
        // Asservissement
        if (timer.read_ms() > seuilAsserv){
            seuilAsserv += PERIODE_ASSERV;
            consigneAsserv = trapeze(consigneGlobale, VITESSE);
            erreur      = calcErreur(consigneAsserv, position);
            commandeXYZ = calcCommandeXYZ(erreur);
            commande123 = calcCommande123(commandeXYZ, position);
            cmdMotor123 = calcCommande3M(commande123);
            moveBot(cmdMotor123);
        }
        
    }
}