// MBED SCRIPT FOR CONTROLLING THE TEMPERATURE CONTROLLED TEST FIXTURE
// AUTHOR: JUSTIN RODENBURG
// DATE: SEPTEMBER 2017

//PLAN TO HAVE 18 CHANNELS PER MBED
//HARDWARE: 
//2 X VALVE DRIVER BOARD
//1 X MBED
//1 X ADC MUX (16 CHN)

//FORMAT FOR CELL DATA: CHANNEL.TEMPERATURE.STATUS.E 1.23.1.E, 1.23.1.E

#include "mbed.h"
#include "MCP23017.h"
#include "MODSERIAL.h"
#include <string>

#ifdef DEBUG
#define printf(fmt, ...) (0)
#endif


DigitalOut rLed(LED1);
DigitalOut gLed(LED2);

//DIGITAL CONTROL PINS
DigitalOut valve(PTB8);
DigitalOut heater(PTB9);

//Init. I2C for CE Board
MCP23017 mcp = MCP23017(PTC9, PTC8, 0x40);

//GLOBAL
// pullUp resistance
int rUp = 10000; 
//LUT size 
int sizeLUT = 34;

//LOOK UP TABLE
const struct LUT {
   int adc;
   int temp;
};

LUT thermLUT[] = {
    {62349,-40},
    {61393,-35},
    {60223,-30},
    {58818,-25},
    {57159,-20},
    {55240,-15},
    {53054,-10},
    {50605,-5},
    {47928,0},
    {45069,5},
    {42068,10},
    {38975,15},
    {35856,20},
    {32768,25},
    {29753,30},
    {26867,35},
    {24147,40},
    {21602,45},
    {19257,50},
    {17116,55},
    {15178,60},
    {13465,65},
    {11941,70},
    {10579,75},
    {9374,80},
    {8309,85},
    {7375,90},
    {6548,95},
    {5817,100},
    {5179,105},
    {4618,110},
    {4127,115},
    {3686,120},
    {3304,125}
};

//ANALOG PINS
AnalogIn ain0(A0);
AnalogIn ain1(A1);
AnalogIn ain2(A2);
AnalogIn ain3(A3);
AnalogIn ain4(A4);
AnalogIn ain5(A5);

MODSERIAL pc(USBTX, USBRX);

volatile int bufA_index;
const int numChnls = 15;
float chTemps[numChnls];
float chGoalTemps[numChnls];  
int chStatus[numChnls];
int numSamples = 5;
volatile bool dataRecieved = false;
int chnlSel;
char rxBuf[50];

void rxInterrupt(MODSERIAL_IRQ_INFO *info){
    gLed = 0;
    wait(2);
    gLed = 1;
    dataRecieved = true;
}

void parseRXData(){
    int pCount = 0; //count data collected
    int i = 0;
    int chnl;
    
    string data = "";

    //pc.printf("buff1 = <%s> \r\n", rxBuf);
    
    while(pCount < 3){
        if(rxBuf[i] != '.'){
            data += rxBuf[i];
        }
        else{
            pCount++; 
            if(pCount == 1){ //get channel
                if((atoi(data.c_str()) < 0) || (atoi(data.c_str()) > 15)){
                    //check if channel is out of array index
                    rLed = 0; 
                }
                else{
                    chnl = atoi(data.c_str());
                    chnl = chnl-1;  //fix for array indexing
                    chnlSel = chnl;
                    //pc.printf("CHANNEL: %i \r\n", chnl);
                }
            }
            else if(pCount == 2){ //get channel temperature
                chGoalTemps[chnl] = atoi(data.c_str());
                //pc.printf("TEMPERATURE: %i \r\n", atoi(data.c_str()));
            }
            else if(pCount == 3){ //get channel status
                chStatus[chnl] = atoi(data.c_str());
                //pc.printf("STATUS: %i \r\n", atoi(data.c_str())); 
            } 
            data = "";  
        }
        i++;
    }
    //wait(30);
}

float getTemp(float ADC_val){ 
    int i = 0;  
    while((i < sizeLUT) && (thermLUT[i].adc > ADC_val)){
        i++;    
    }  
    
    float a = float(thermLUT[i-1].temp - thermLUT[i].temp);
    float b = float(thermLUT[i-1].adc - thermLUT[i].adc);
    
    float m = a/b;
    float y = (m*(ADC_val - thermLUT[i-1].adc))+thermLUT[i-1].temp;
    
    return y;
}

//16-bit ADC --> 2^16 = 65536
void readTemps(){
    float a_0 = 0.0;
    
    for(int i=0; i<numSamples; i++){
        a_0 += ain0.read_u16();
    }
    
    a_0 = a_0/numSamples;
    
    //printf("ADC VAL: %f \r\n", a_0);

    chTemps[0] = getTemp(a_0); 
    
    //send current temperature of selected channel to GUI for display
    pc.printf("%f \r\n", chTemps[chnlSel]);
    
    wait(1.5);
    
    //printf("TEMPERATURE (C): %f \r\n", chTemps[0]);
    
    //printf("TEMPERATURE (C): 15 \r\n");
    
    //printf("TEMPERATURE (C): 15 \r\n");
}


    

int main() {
    //printf("Main Entered.\n");
    //turn off LEDs
    rLed = 1;
    gLed = 1;
    
    //configure MCP23017
    mcp.config(0,0,0);
    
    //
    pc.baud(9600);
    pc.autoDetectChar('E');
    pc.attach(&rxInterrupt, MODSERIAL::RxAutoDetect);
        
    while(1) {
        //read temperatures
        readTemps();
        
        //buffer for noise
        double hyst = 0.2;
        
        //check if we recieved data/need to update TCTF data
        if(dataRecieved){
            dataRecieved = false;
            pc.move(rxBuf, 50);
            pc.rxBufferFlush();
            parseRXData();
        }
                
        //Control loop:
        //Loop through all channels and control
        //chiller = 18C
        for(int i=0; i<=numChnls; i++){
            //control chiller
            if(chStatus[i] == 1){
                if(chTemps[i] > ((chGoalTemps[i])+hyst)){
                    //Turn on chiller
                    //printf("CHILLER ON \r\n");
                    mcp.config(0,0,0);
                    mcp.write_bit(1, 8); 
                    //printf("HEATER OFF \r\n");
                    mcp.write_bit(0, 9);
                }
                else if (chTemps[i] < ((chGoalTemps[i])-hyst)){
                    //turn off chiller
                    //printf("CHILLER OFF \r\n");
                    mcp.config(0,0,0);
                    mcp.write_bit(0, 8); 
                    //printf("HEATER ON  \r\n");
                    mcp.write_bit(1, 9);
                }
                else{
                    //turn off chiller
                    //printf("CHILLER OFF \r\n");
                    mcp.config(0,0,0);
                    mcp.write_bit(0, 8); 
                    //printf("HEATER OFF \r\n");
                    mcp.write_bit(0, 9);
                }
                
                
            }
            wait(0);
        }
    }
}
