#include "mbed.h"
Serial pc(SERIAL_TX, SERIAL_RX);

#define CURVE_SECTION_MM 0.5
float X_STEPS_PER_MM = 40;
int X_MOTOR_STEPS   = 100;
float Y_STEPS_PER_MM  = 40;
int Y_MOTOR_STEPS   = 100;
long FAST_XY_FEEDRATE = 2500;

PinName X_STEP_PIN = PB_13;
PinName X_DIR_PIN = PB_14;
PinName Y_STEP_PIN = PA_12;
PinName Y_DIR_PIN = PC_5;
PinName X_MIN_PIN = PC_0;
PinName X_MAX_PIN = PC_1;
PinName Y_MIN_PIN = PB_0;
PinName Y_MAX_PIN = PA_4;

struct FloatPoint {
  float x;
  float y;
};

FloatPoint current_units;
FloatPoint target_units;
FloatPoint delta_units;

FloatPoint current_steps;
FloatPoint target_steps;
FloatPoint delta_steps;

int8_t x_direction = 1;
int8_t y_direction = 1;
long feedrate_micros = 0;
long max_delta;
long x_counter;
long y_counter;
int milli_delay;
bool x_can_step;
bool y_can_step;



long calculate_feedrate_delay(float feedrate);
void dda_move(long micro_delay);
void do_step(PinName pinA, PinName pinB, int8_t dir);
bool can_step(PinName min_pin, PinName max_pin, long current, long target, int8_t direction);
void calculate_deltas();
void set_target(float x, float y);


int main()
{
  pc.baud(115200);
  pc.printf("Hello World\n");

  current_units.x = 0.0;
  current_units.y = 0.0;

  target_units.x = 0.0;
  target_units.y = 0.0;

  calculate_deltas();

  set_target(100.0, 200.0);

  feedrate_micros = calculate_feedrate_delay(FAST_XY_FEEDRATE);
  dda_move(feedrate_micros);
}

long calculate_feedrate_delay(float feedrate)
{
  //how long is our line length?
  float distance = sqrt(delta_units.x * delta_units.x + delta_units.y * delta_units.y);
  long master_steps = 0;

  //find the dominant axis.
  if (delta_steps.x > delta_steps.y)
  {
    master_steps = delta_steps.x;
  }
  else
  {
    master_steps = delta_steps.y;
  }
  return ((distance * 60000000.0) / feedrate) / master_steps;
}


void dda_move(long micro_delay)
{
  //figure out our deltas
  max_delta = delta_steps.x>delta_steps.y? delta_steps.x:delta_steps.y;

  //init stuff.
  long x_counter = -max_delta / 2;
  long y_counter = -max_delta / 2;

  //our step flags
  bool x_can_step = 0;
  bool y_can_step = 0;

  if (micro_delay >= 16383)
    milli_delay = micro_delay / 1000;
  else
    milli_delay = 0;

  //do our DDA line!
  do
  {
    x_can_step = can_step(X_MIN_PIN, X_MAX_PIN, current_steps.x, target_steps.x, x_direction);
    y_can_step = can_step(Y_MIN_PIN, Y_MAX_PIN, current_steps.y, target_steps.y, y_direction);
   
    if (x_can_step)
    {
      x_counter += delta_steps.x;

      if (x_counter > 0)
      {
        do_step(X_STEP_PIN, X_DIR_PIN, x_direction);
        x_counter -= max_delta;

        if (x_direction)
          current_steps.x++;
        else
          current_steps.x--;
      }
    }

    if (y_can_step)
    {
      y_counter += delta_steps.y;

      if (y_counter > 0)
      {
        do_step(Y_STEP_PIN, Y_DIR_PIN, y_direction);
        y_counter -= max_delta;

        if (y_direction)
          current_steps.y++;
        else
          current_steps.y--;
      }
    }


    //wait for next step.
    if (milli_delay > 0)
      wait(milli_delay);
    else
      wait_us(micro_delay);
  }
  while (x_can_step || y_can_step);

  //set our points to be the same
  current_units.x = target_units.x;
  current_units.y = target_units.y;
  calculate_deltas();
}
bool can_step(PinName min_pin, PinName max_pin, long current, long target, int8_t direction)
{
  DigitalIn Max(max_pin,PullUp);
  DigitalIn Min(min_pin,PullUp);
  //stop us if we're on target
  if (target == current)
    return false;
  //stop us if we're at home and still going
  else if (!Min.read() && !direction)
    return false;
  //stop us if we're at max and still going
  else if (!Max.read() && direction)
    return false;

  //default to being able to step
  return true;
}

void calculate_deltas()
{
  //figure our deltas.
  delta_units.x = abs(target_units.x - current_units.x);
  delta_units.y = abs(target_units.y - current_units.y);

  //set our steps current, target, and delta
  current_steps.x = X_STEPS_PER_MM * current_units.x;
  current_steps.y = Y_STEPS_PER_MM * current_units.y;

  target_steps.x = X_STEPS_PER_MM * target_units.x;
  target_steps.y = Y_STEPS_PER_MM * target_units.y;

  delta_steps.x = abs(target_steps.x - current_steps.x);
  delta_steps.y = abs(target_steps.y - current_steps.y);

  //what is our direction
  x_direction = (target_units.x >= current_units.x);
  y_direction = (target_units.y >= current_units.y);

  //set our direction pins as well
  DigitalOut X_DIR(X_DIR_PIN,x_direction);
  DigitalOut Y_DIR(Y_DIR_PIN,y_direction);
}


void set_target(float x, float y)
{
  target_units.x = x;
  target_units.y = y;

  calculate_deltas();
}

void do_step(PinName pinA, PinName pinB, int8_t dir)
{
  DigitalOut A(pinA);
  DigitalOut B(pinB);
  switch (dir << 2 | A.read() << 1 | B.read()) {
    case 0: /* 0 00 -> 10 */
    case 5: /* 1 01 -> 11 */
      A.write(1);
      break;
    case 1: /* 0 01 -> 00 */
    case 7: /* 1 11 -> 10 */
      B.write(0);
      break;
    case 2: /* 0 10 -> 11 */
    case 4: /* 1 00 -> 01 */
      B.write(1);
      break;
    case 3: /* 0 11 -> 01 */
    case 6: /* 1 10 -> 00 */
      A.write(0);
      break;
  }
  wait_us(5);
}