// Library for the STMicroelectronics LSM6DS0 3D accelerometer and 3D gyroscope

#include "lsm6ds0.h"
#include "functions.h"

static Serial pc(SERIAL_TX, SERIAL_RX);

void LSM6DS0::setFIFO(LSM6DS0_FIFO_mode_t work_mode)
{
    char data_write[1];
    
    data_write[0] = work_mode;                          // set FIFO working mode
    i2c_write(LSM6DS0_ADDRESS, LSM6DS0_SUB_FIFO_CTRL, data_write, 1);
    data_write[0] = 0x50;                               // set FIFO-full and overrun interrupts
    i2c_write(LSM6DS0_ADDRESS, LSM6DS0_SUB_INT_CTRL, data_write, 1);
    data_write[0] = 0x02;                               // enable FIFO. To enable FIFO_threshold stop set value 3
    i2c_write(LSM6DS0_ADDRESS, LSM6DS0_SUB_CTRL_REG9, data_write, 1);
    
}

void LSM6DS0::resetFIFO(void)
{
    char data_write[1];
    LSM6DS0_FIFO_mode_t work_mode;
    work_mode = LSM6DS0_FIFO_mode_BYPASS;
    data_write[0] = work_mode;                          // set FIFO working mode
    i2c_write(LSM6DS0_ADDRESS, LSM6DS0_SUB_FIFO_CTRL, data_write, 1);
    
}

void LSM6DS0::getFIFO(void)
{
    /* This function reads the content of the entire FIFO */
    char data_read[192]; //3 axes * 2 read for each ax * 32 FIFO depth = 192
    
    i2c_read(LSM6DS0_ADDRESS, LSM6DS0_SUB_OUT_X_L_XL, data_read, 192);
    
    for(int i=0; i<32; i++)
    {
        printf("x = %d, y = %d, z = %d\n\r", (int16_t)((data_read[1+i*6] << 8)| data_read[0+i*6]), (int16_t)((data_read[3+i*6] << 8)| data_read[2+i*6]), (int16_t)((data_read[5+i*6] << 8)| data_read[4+i*6]));
    }
}

void LSM6DS0::storeFIFO(int *offset, xl_output *data_FIFO)
{
    int i;
    
    /* This function reads the content of the entire FIFO */
    char data_read[192]; //3 axes * 2 read for each ax * 32 FIFO depth = 192
    
    i2c_read(LSM6DS0_ADDRESS, LSM6DS0_SUB_OUT_X_L_XL, data_read, 192);
    
    if(*offset == FIFO_length_by_five)
    {
        *offset = 0;
    }
    
    for(i=0; i<32; i++)
    {
        
        data_FIFO[i+*offset].x = (int16_t)((data_read[1+i*6] << 8)| data_read[0+i*6]);
        data_FIFO[i+*offset].y = (int16_t)((data_read[3+i*6] << 8)| data_read[2+i*6]);
        data_FIFO[i+*offset].z = (int16_t)((data_read[5+i*6] << 8)| data_read[4+i*6]);
        
    }
    *offset = *offset + i;
}

void LSM6DS0::computeValue(xl_output *data_FIFO)
 {
     int i;
     float sensValue = getSensitivity();
     
     for(i=0; i<FIFO_length_by_five; i++)
     {
     pc.printf("X : %f ", (float)(data_FIFO[i].x*sensValue));
     pc.printf("Y : %f ", (float)(data_FIFO[i].y*sensValue));
     pc.printf("Z : %f \n\r", (float)(data_FIFO[i].z*sensValue));
     }
 }
 
float LSM6DS0::getSensitivity(void)
{
    char data_read[1];
    float sensValue;
    
    i2c_read(LSM6DS0_ADDRESS, LSM6DS0_SUB_CTRL_REG6_XL, data_read, 1);
    
    switch(data_read[0] & 0x18) {
        case 0x00:
            sensValue = LSM6DS0_ACC_SENS_2G;
            break;
        case 0x08:
            sensValue = LSM6DS0_ACC_SENS_16G;
            break;
        case 0x10:
            sensValue = LSM6DS0_ACC_SENS_4G;
            break;
        case 0x18:
            sensValue = LSM6DS0_ACC_SENS_8G;
            break;
        default:
            sensValue = 1;
            pc.printf("\n\rSensitivity error. No value\n\r");
            break;
    }
    
    return sensValue;
}

void LSM6DS0::Init(LSM6DS0_ACC_ODR_t odr, LSM6DS0_ACC_HR_t high_res, LSM6DS0_FIFO_mode_t work_mode)
{
    char data_write[1];
    data_write[0] = odr;            // Set frequency
    i2c_write(LSM6DS0_ADDRESS, LSM6DS0_SUB_CTRL_REG6_XL, data_write, 1);
    data_write[0] = high_res;       // Set High-Resolution
    i2c_write(LSM6DS0_ADDRESS, LSM6DS0_SUB_CTRL_REG7_XL, data_write, 1);
    
    setFIFO(work_mode);
}

void LSM6DS0::LSM6DS0_reset(void)
{
    char data_write[1];
    data_write[0] = 0x01;            // Software reset
    i2c_write(LSM6DS0_ADDRESS, LSM6DS0_SUB_CTRL_REG8, data_write, 1);
}

void LSM6DS0::printValue(xl_output *data_FIFO)
{
    int i;
    for(i=0; i<FIFO_length_by_five; i++)
    {
        pc.printf("\r\n");
        pc.printf("%d: x = %d, y = %d, z = %d\n\r", i, data_FIFO[i].x, data_FIFO[i].y, data_FIFO[i].z);
        
    }
}

void LSM6DS0::printReg(void)
{
    char data_read[1];
    
    i2c_read(LSM6DS0_ADDRESS, LSM6DS0_SUB_CTRL_REG6_XL, data_read, 1);
    pc.printf("CTRL_REG6_XL = %#4X\r\n", data_read[0]);
    i2c_read(LSM6DS0_ADDRESS, LSM6DS0_SUB_CTRL_REG7_XL, data_read, 1);
    pc.printf("CTRL_REG7_XL = %#4X\r\n", data_read[0]);
    i2c_read(LSM6DS0_ADDRESS, LSM6DS0_SUB_FIFO_CTRL, data_read, 1);
    pc.printf("SUB_FIFO_CTRL = %#4X\r\n", data_read[0]);
    i2c_read(LSM6DS0_ADDRESS, LSM6DS0_SUB_INT_CTRL, data_read, 1);
    pc.printf("INT_CTRL = %#4X\r\n", data_read[0]);
    i2c_read(LSM6DS0_ADDRESS, LSM6DS0_SUB_CTRL_REG9, data_read, 1);
    pc.printf("CTRL_REG9 = %#4X\r\n", data_read[0]);
}