// Library for the STMicroelectronics LSM6DS0 3D accelerometer and 3D gyroscope

// Define to prevent recursive inclusion -------------------------------------
#ifndef __LSM6DS0_H
#define __LSM6DS0_H

#include "mbed.h"


/******************************************************************************/
/*********** 3D ACCELEROMETER AND 3D GYROSCOPE REGISTER MAPPING  **************/
/******************************************************************************/

#define  LSM6DS0_ACC_SENS_2G                     ((float)0.061)         // Accelerometer sensitivity with 2 G full scale [mg / LSB]
#define  LSM6DS0_ACC_SENS_4G                     ((float)0.122)         // Accelerometer sensitivity with 4 G full scale [mg / LSB]
#define  LSM6DS0_ACC_SENS_8G                     ((float)0.244)         // Accelerometer sensitivity with 8 G full scale [mg / LSB]
#define  LSM6DS0_ACC_SENS_16G                    ((float)0.732)          // Accelerometer sensitivity with 16 G full scale [mg / LSB]

#define  LSM6DS0_GYRO_SENS_245DPS                ((float)8.75f)         //
#define  LSM6DS0_GYRO_SENS_500DPS                ((float)17.50f)        // Gyroscope sensitivity with 500 dps full scale [mdps / LSB]
#define  LSM6DS0_GYRO_SENS_2000DPS               ((float)70f)           // Gyroscope sensitivity with 2000 dps full scale [mdps / LSB]

#define  LSM6DS0_GYRO_SENS_245DPS                ((float)8.75f)         //Gyroscope sensitivity with 245 dps full scale [mdps / LSB]
#define  LSM6DS0_GYRO_SENS_500DPS                ((float)17.50f)        // Gyroscope sensitivity with 500 dps full scale [mdps / LSB]
#define  LSM6DS0_GYRO_SENS_2000DPS               ((float)70f)           // Gyroscope sensitivity with 2000 dps full scale [mdps / LSB]

// LSM6DS0 register names

/******************************************************************************/
/*                                                                            */
/*                        LSM6DS0 on board MEMS                               */
/*                                                                            */
/******************************************************************************/
/*****************  Bit definition for I2C/SPI communication  *****************/
#define  LSM6DS0_ADDRESS                         0xD6
#define  LSM6DS0_SUB                             ((uint8_t)0x7F)            // SUB[6:0] Sub-registers address Mask
#define  LSM6DS0_SUB_0                           ((uint8_t)0x01)            // bit 0
#define  LSM6DS0_SUB_1                           ((uint8_t)0x02)            // bit 1
#define  LSM6DS0_SUB_2                           ((uint8_t)0x08)            // bit 3
#define  LSM6DS0_SUB_4                           ((uint8_t)0x10)            // bit 4
#define  LSM6DS0_SUB_5                           ((uint8_t)0x20)            // bit 5
#define  LSM6DS0_SUB_6                           ((uint8_t)0x40)            // bit 6

#define  LSM6DS0_SUB_MSB                         ((uint8_t)0x80)            // Multiple data read\write bit

/*****************  Bit definition for Registers Addresses  *******************/
#define  LSM6DS0_SUB_ACT_THS                     ((uint8_t)0x04)            // Activity threshold register
#define  LSM6DS0_SUB_ACT_DUR                     ((uint8_t)0x05)            // Inactivity duration register
#define  LSM6DS0_SUB_INT_GEN_CFG_XL              ((uint8_t)0x06)            // Accelerometer interrupt generator configuration register
#define  LSM6DS0_SUB_INT_GEN_THS_X_XL            ((uint8_t)0x07)            // Accelerometer X-axis interrupt threshold register
#define  LSM6DS0_SUB_INT_GEN_THS_Y_XL            ((uint8_t)0x08)            // Accelerometer Y-axis interrupt threshold register
#define  LSM6DS0_SUB_INT_GEN_THS_Z_XL            ((uint8_t)0x09)            // Accelerometer Z-axis interrupt threshold register
#define  LSM6DS0_SUB_INT_GEN_DUR_XL              ((uint8_t)0x0A)            // Accelerometer interrupt duration register
#define  LSM6DS0_SUB_REFERENCE_G                 ((uint8_t)0x0B)            // Gyroscope reference value register for digital high-pass filter
#define  LSM6DS0_SUB_INT_CTRL                    ((uint8_t)0x0C)            // INT pin control register
#define  LSM6DS0_SUB_WHO_AM_I                    ((uint8_t)0x0F)            // Who_AM_I register
#define  LSM6DS0_SUB_CTRL_REG1_G                 ((uint8_t)0x10)            // Gyroscope control register 1
#define  LSM6DS0_SUB_CTRL_REG2_G                 ((uint8_t)0x11)            // Gyroscope control register 2
#define  LSM6DS0_SUB_CTRL_REG3_G                 ((uint8_t)0x12)            // Gyroscope control register 3
#define  LSM6DS0_SUB_ORIENT_CFG_G                ((uint8_t)0x13)            // Gyroscope sign and orientation register
#define  LSM6DS0_SUB_INT_GEN_SRC_G               ((uint8_t)0x14)            // Gyroscope interrupt source register
#define  LSM6DS0_SUB_OUT_TEMP_L                  ((uint8_t)0x15)            // Temperature data output low register
#define  LSM6DS0_SUB_OUT_TEMP_H                  ((uint8_t)0x16)            // Temperature data output high register
#define  LSM6DS0_SUB_STATUS_REG1                 ((uint8_t)0x17)            // Status register 1
#define  LSM6DS0_SUB_OUT_X_L_G                   ((uint8_t)0x18)            // Gyroscope X-axis low output register
#define  LSM6DS0_SUB_OUT_X_H_G                   ((uint8_t)0x19)            // Gyroscope X-axis high output register
#define  LSM6DS0_SUB_OUT_Y_L_G                   ((uint8_t)0x1A)            // Gyroscope Y-axis low output register
#define  LSM6DS0_SUB_OUT_Y_H_G                   ((uint8_t)0x1B)            // Gyroscope Y-axis high output register
#define  LSM6DS0_SUB_OUT_Z_L_G                   ((uint8_t)0x1C)            // Gyroscope Z-axis low output register
#define  LSM6DS0_SUB_OUT_Z_H_G                   ((uint8_t)0x1D)            // Gyroscope Z-axis high output register
#define  LSM6DS0_SUB_CTRL_REG4                   ((uint8_t)0x1E)            // Control register 4
#define  LSM6DS0_SUB_CTRL_REG5_XL                ((uint8_t)0x1F)            // Accelerometer Control Register 5
#define  LSM6DS0_SUB_CTRL_REG6_XL                ((uint8_t)0x20)            // Accelerometer Control Register 6
#define  LSM6DS0_SUB_CTRL_REG7_XL                ((uint8_t)0x21)            // Accelerometer Control Register 7
#define  LSM6DS0_SUB_CTRL_REG8                   ((uint8_t)0x22)            // Control register 8
#define  LSM6DS0_SUB_CTRL_REG9                   ((uint8_t)0x23)            // Control register 9
#define  LSM6DS0_SUB_CTRL_REG10                  ((uint8_t)0x24)            // Control register 10
#define  LSM6DS0_SUB_INT_GEN_SRC_XL              ((uint8_t)0x26)            // Accelerometer interrupt source register
#define  LSM6DS0_SUB_STATUS_REG2                 ((uint8_t)0x27)            // Status register
#define  LSM6DS0_SUB_OUT_X_L_XL                  ((uint8_t)0x28)            // Accelerometer X-axis low output register
#define  LSM6DS0_SUB_OUT_X_H_XL                  ((uint8_t)0x29)            // Accelerometer X-axis high output register
#define  LSM6DS0_SUB_OUT_Y_L_XL                  ((uint8_t)0x2A)            // Accelerometer Y-axis low output register
#define  LSM6DS0_SUB_OUT_Y_H_XL                  ((uint8_t)0x2B)            // Accelerometer Y-axis high output register
#define  LSM6DS0_SUB_OUT_Z_L_XL                  ((uint8_t)0x2C)            // Accelerometer Z-axis low output register
#define  LSM6DS0_SUB_OUT_Z_H_XL                  ((uint8_t)0x2D)            // Accelerometer Z-axis high output register
#define  LSM6DS0_SUB_FIFO_CTRL                   ((uint8_t)0x2E)            // FIFO control register
#define  LSM6DS0_SUB_FIFO_SRC                    ((uint8_t)0x2F)            // FIFO status control register
#define  LSM6DS0_SUB_INT_GEN_CFG_G               ((uint8_t)0x30)            // Gyroscope interrupt generator configuration register
#define  LSM6DS0_SUB_INT_GEN_THS_XH_G            ((uint8_t)0x31)            // Gyroscope X-axis low interrupt generator threshold registers
#define  LSM6DS0_SUB_INT_GEN_THS_XL_G            ((uint8_t)0x32)            // Gyroscope X-axis high interrupt generator threshold registers
#define  LSM6DS0_SUB_INT_GEN_THS_YH_G            ((uint8_t)0x33)            // Gyroscope Y-axis low interrupt generator threshold registers
#define  LSM6DS0_SUB_INT_GEN_THS_YL_G            ((uint8_t)0x34)            // Gyroscope Y-axis high interrupt generator threshold registers
#define  LSM6DS0_SUB_INT_GEN_THS_ZH_G            ((uint8_t)0x35)            // Gyroscope Z-axis low interrupt generator threshold registers
#define  LSM6DS0_SUB_INT_GEN_THS_ZL_G            ((uint8_t)0x36)            // Gyroscope Z-axis high interrupt generator threshold registers
#define  LSM6DS0_SUB_INT_GEN_DUR_G               ((uint8_t)0x37)            // Gyroscope interrupt generator duration register

#define BIT_0                                    0
#define BIT_1                                    1
#define BIT_2                                    2
#define BIT_3                                    3
#define BIT_4                                    4
#define BIT_5                                    5
#define BIT_6                                    6
#define BIT_7                                    7

#define FIFO_length_by_five                      160                        //The length of 5 FIFO depth 

// Generic LSM6DS0 data structures and types

// FIFO mode
typedef enum {
    LSM6DS0_FIFO_mode_BYPASS = 0x00,                    // LSM6DS0 work in BYPASS mode
    LSM6DS0_FIFO_mode_FIFO = 0x20,                      // LSM6DS0 work in FIFO mode
    LSM6DS0_FIFO_mode_CONTINUOUS = 0xC0,                // LSM6DS0 work in CONTINUOUS mode
    LSM6DS0_FIFO_mode_CONTINUOUS_BYPASS = 0x60,         // LSM6DS0 work in CONTINUOUS-to-BYPAS mode
    LSM6DS0_FIFO_mode_BYPASS_CONTINUOUS = 0x80,         // LSM6DS0 work in BYPASS-to-CONTINUOUS mode
}LSM6DS0_FIFO_mode_t;


// Accelerometer and Gyroscope Slave Address
typedef enum {
    LSM6DS0_SAD_GND = 0x6A,                             // LSM6DS0 Slave Address when SA1 is to GND
    LSM6DS0_SAD_VCC = 0x6B                              // LSM6DS0 Slave Address when SA1 is to VCC
}LSM6DS0_SAD_t;

// Accelerometer and Gyroscope Block Data Update
typedef enum
{
    LSM6DS0_BDU_CONTINOUS = 0x00,                       // Continuos Update
    LSM6DS0_BDU_BLOCKED   = 0x40                        // Single Update: output registers not updated until MSB and LSB reading
}LSM6DS0_BDU_t;

// Accelerometer and Gyroscope Endianness
typedef enum
{
    LSM6DS0_END_LITTLE = 0x00,                          // Little Endian: data LSB @ lower address
    LSM6DS0_END_BIG    = 0x20                           // Big Endian: data MSB @ lower address
}LSM6DS0_END_t;

// Accelerometer data structures and types

// Accelerometer Decimation Mode
typedef enum {
    LSM6DS0_ACC_DEC_DISABLED = 0x00,                    // NO decimation
    LSM6DS0_ACC_DEC_X2       = 0x40,                    // Decimation update every 2 sample
    LSM6DS0_ACC_DEC_X4       = 0x80,                    // Decimation update every 4 sample
    LSM6DS0_ACC_DEC_X8       = 0xC0                     // Decimation update every 8 sample
}LSM6DS0_ACC_DEC_t;

// Accelerometer Axes Enabling
typedef enum{
    LSM6DS0_ACC_AE_DISABLED = 0x00,                     // Axes all disabled
    LSM6DS0_ACC_AE_X        = 0x08,                     // Only X-axis enabled
    LSM6DS0_ACC_AE_Y        = 0x10,                     // Only Y-axis enabled
    LSM6DS0_ACC_AE_XY       = 0x18,                     // X & Y axes enabled
    LSM6DS0_ACC_AE_Z        = 0x20,                     // Only Z-axis enabled
    LSM6DS0_ACC_AE_XZ       = 0x28,                     // X & Z axes enabled
    LSM6DS0_ACC_AE_YZ       = 0x30,                     // Y & Z axes enabled
    LSM6DS0_ACC_AE_XYZ      = 0x38                      // All axes enabled
}LSM6DS0_ACC_AE_t;

// Accelerometer Output Data Rate
typedef enum {
    LSM6DS0_ACC_ODR_PD    = 0x00,                       // Power down
    LSM6DS0_ACC_ODR_10Hz  = 0x20,                       // Output Data Rate = 10 Hz
    LSM6DS0_ACC_ODR_50Hz  = 0x40,                       // Output Data Rate = 50 Hz
    LSM6DS0_ACC_ODR_119Hz = 0x60,                       // Output Data Rate = 119 Hz
    LSM6DS0_ACC_ODR_238Hz = 0x80,                       // Output Data Rate = 238 Hz
    LSM6DS0_ACC_ODR_476Hz = 0xA0,                       // Output Data Rate = 476 Hz
    LSM6DS0_ACC_ODR_952Hz = 0xC0                        // Output Data Rate = 952 Hz
}LSM6DS0_ACC_ODR_t;

// Accelerometer Full Scale
typedef enum {
    LSM6DS0_ACC_FS_2G  = 0x00,                          // 2 g m/s^2
    LSM6DS0_ACC_FS_4G  = 0x10,                          // 4 g m/s^2
    LSM6DS0_ACC_FS_8G  = 0x18,                          // 8 g m/s^2
    LSM6DS0_ACC_FS_16G = 0x08                           // 16 g m/s^2
}LSM6DS0_ACC_FS_t;

// Accelerometer Antialiasing filter Bandwidth Selection
typedef enum {
    LSM6DS0_ACC_BW_408Hz    = 0x00,                     // AA filter bandwidth = 408 Hz
    LSM6DS0_ACC_BW_211Hz    = 0x01,                     // AA filter bandwidth = 211 Hz
    LSM6DS0_ACC_BW_105Hz    = 0x02,                     // AA filter bandwidth = 105 Hz
    LSM6DS0_ACC_BW_50Hz     = 0x03,                     // AA filter bandwidth = 50 Hz
    LSM6DS0_ACC_BW_ACCORDED = 0x04,                     // AA filter bandwidth chosen by ODR selection
}LSM6DS0_ACC_BW_t;

// Accelerometer High Resolution mode
typedef enum
{
    LSM6DS0_ACC_HR_Disabled = 0x00,                     // High resolution output mode disabled, FDS bypassed
    LSM6DS0_ACC_HR_EN_9     = 0xC4,                     // High resolution output mode enabled, LP cutoff = ODR/9, FDS enabled
    LSM6DS0_ACC_HR_EN_50    = 0x84,                     // High resolution output mode enabled, LP cutoff = ODR/50, FDS enabled
    LSM6DS0_ACC_HR_EN_100   = 0xA4,                     // High resolution output mode enabled, LP cutoff = ODR/100, FDS enabled
    LSM6DS0_ACC_HR_EN_400   = 0xE4,                     // High resolution output mode enabled, LP cutoff = ODR/400, FDS enabled
}LSM6DS0_ACC_HR_t;

// HP filter for interrupt
typedef enum
{
    LSM6DS0_ACC_HPIS1_BYPASSED = 0x00,                  // High-pass filter bypassed
    LSM6DS0_ACC_HPIS1_ENABLED  = 0x01                   // High-pass filter enabled for accelerometer interrupt function on interrupt
}LSM6DS0_ACC_HPIS1_t;

// Accelerometer configuration structure.
typedef struct {
    LSM6DS0_SAD_t       slaveaddress;       // LSM6DS0 Slave Address
    LSM6DS0_ACC_DEC_t   decimation;         // Accelerometer Decimation Mode
    LSM6DS0_ACC_ODR_t   outputdatarate;     // Accelerometer Output Data Rate
    LSM6DS0_ACC_BW_t    bandwidth;          // Accelerometer Antialiasing filter Bandwidth Selection
    LSM6DS0_ACC_FS_t    fullscale;          // Accelerometer Full Scale
    LSM6DS0_ACC_AE_t    axesenabling;       // Accelerometer Axes Enabling
    LSM6DS0_ACC_HR_t    highresmode;        // Accelerometer High Resolution mode
    LSM6DS0_ACC_HPIS1_t hpfirq;             // HP filter for interrupt
    LSM6DS0_END_t       endianess;          // LSM6DS0 Endianness
    LSM6DS0_BDU_t       blockdataupdate;    // LSM6DS0 Block Data Update
} LSM6DS0_ACC_Config;


// Gyroscope data structures and types

// Gyroscope Output Data Rate
typedef enum {
    LSM6DS0_GYRO_ODR_PD             = 0x00,             // Power down
    LSM6DS0_GYRO_ODR_14_9Hz_CO_5Hz  = 0x20,             // Output Data Rate = 14.9 Hz, CutOff = 5Hz
    LSM6DS0_GYRO_ODR_59_5Hz_CO_16Hz = 0x40,             // Output Data Rate = 59.5 Hz, CutOff = 16Hz
    LSM6DS0_GYRO_ODR_119Hz_CO_14Hz  = 0x60,             // Output Data Rate = 119 Hz, CutOff = 14Hz
    LSM6DS0_GYRO_ODR_119Hz_CO_31Hz  = 0x61,             // Output Data Rate = 119 Hz, CutOff = 31Hz
    LSM6DS0_GYRO_ODR_238Hz_CO_14Hz  = 0x80,             // Output Data Rate = 238 Hz, CutOff = 14Hz
    LSM6DS0_GYRO_ODR_238Hz_CO_29Hz  = 0x81,             // Output Data Rate = 328 Hz, CutOff = 29Hz
    LSM6DS0_GYRO_ODR_238Hz_CO_63Hz  = 0x82,             // Output Data Rate = 238 Hz, CutOff = 63Hz
    LSM6DS0_GYRO_ODR_238Hz_CO_78Hz  = 0x83,             // Output Data Rate = 476 Hz, CutOff = 78Hz
    LSM6DS0_GYRO_ODR_476Hz_CO_21Hz  = 0xA0,             // Output Data Rate = 476 Hz, CutOff = 21Hz
    LSM6DS0_GYRO_ODR_476Hz_CO_28Hz  = 0xA1,             // Output Data Rate = 238 Hz, CutOff = 28Hz
    LSM6DS0_GYRO_ODR_476Hz_CO_57Hz  = 0xA2,             // Output Data Rate = 476 Hz, CutOff = 57Hz
    LSM6DS0_GYRO_ODR_476Hz_CO_100Hz = 0xA3,             // Output Data Rate = 476 Hz, CutOff = 100Hz
    LSM6DS0_GYRO_ODR_952Hz_CO_33Hz  = 0xC0,             // Output Data Rate = 952 Hz, CutOff = 33Hz
    LSM6DS0_GYRO_ODR_952Hz_CO_40Hz  = 0xC1,             // Output Data Rate = 952 Hz, CutOff = 40Hz
    LSM6DS0_GYRO_ODR_952Hz_CO_58Hz  = 0xC2,             // Output Data Rate = 952 Hz, CutOff = 58Hz
    LSM6DS0_GYRO_ODR_952Hz_CO_100Hz = 0xC3              // Output Data Rate = 952 Hz, CutOff = 100Hz
}LSM6DS0_GYRO_ODR_t;

// Gyroscope Full Scale
typedef enum {
    LSM6DS0_GYRO_FS_245DSP  = 0x00,                     // 245 degrees per second
    LSM6DS0_GYRO_FS_500DSP  = 0x08,                     // 500 degrees per second
    LSM6DS0_GYRO_FS_2000DSP = 0x18                      // 2000 degrees per second
}LSM6DS0_GYRO_FS_t;

// Gyroscope Output Selection
typedef enum {
    LSM6DS0_GYRO_OUT_SEL_BYPASS    = 0x00,              // Output not filtered
    LSM6DS0_GYRO_OUT_SEL_FILTERED  = 0x01,              // Output filtered
}LSM6DS0_GYRO_OUT_SEL_t;

// Gyroscope Interrupt Selection
typedef enum {
    LSM6DS0_GYRO_INT_SEL_BYPASS    = 0x00,              // Interrupt generator signal not filtered
    LSM6DS0_GYRO_INT_SEL_FILTERED  = 0x08,              // Interrupt generator signal filtered
}LSM6DS0_GYRO_INT_SEL_t;

// Gyroscope Low Power Mode
typedef enum {
    LSM6DS0_GYRO_LP_MODE_HIGH_PERFORMANCE = 0x00,       // High performance
    LSM6DS0_GYRO_LP_MODE_LOW_POWER        = 0x80,       // Low power
}LSM6DS0_GYRO_LP_MODE_t;

// Gyroscope High Pass Filter Cutoff Selection
typedef enum {
    LSM6DS0_GYRO_HPCF_DISABLED = 0x00,                  // HP filter disabled
    LSM6DS0_GYRO_HPCF_0        = 0x40,                  // Config 0 refer to table 48 of DOcID025604 Rev 3
    LSM6DS0_GYRO_HPCF_1        = 0x41,                  // Config 1 refer to table 48 of DOcID025604 Rev 3
    LSM6DS0_GYRO_HPCF_2        = 0x42,                  // Config 2 refer to table 48 of DOcID025604 Rev 3
    LSM6DS0_GYRO_HPCF_3        = 0x43,                  // Config 3 refer to table 48 of DOcID025604 Rev 3
    LSM6DS0_GYRO_HPCF_4        = 0x44,                  // Config 4 refer to table 48 of DOcID025604 Rev 3
    LSM6DS0_GYRO_HPCF_5        = 0x45,                  // Config 5 refer to table 48 of DOcID025604 Rev 3
    LSM6DS0_GYRO_HPCF_6        = 0x46,                  // Config 6 refer to table 48 of DOcID025604 Rev 3
    LSM6DS0_GYRO_HPCF_7        = 0x47,                  // Config 7 refer to table 48 of DOcID025604 Rev 3
    LSM6DS0_GYRO_HPCF_8        = 0x48,                  // Config 8 refer to table 48 of DOcID025604 Rev 3
    LSM6DS0_GYRO_HPCF_9        = 0x49,                  // Config 9 refer to table 48 of DOcID025604 Rev 3
    LSM6DS0_GYRO_HPCF_10       = 0x4A                   // Config 10 refer to table 48 of DOcID025604 Rev 3
}LSM6DS0_GYRO_HPCF_t;

// Gyroscope Axes Enabling
typedef enum{
    LSM6DS0_GYRO_AE_DISABLED = 0x00,                    // Axes all disabled
    LSM6DS0_GYRO_AE_X        = 0x08,                    // Only X-axis enabled
    LSM6DS0_GYRO_AE_Y        = 0x10,                    // Only Y-axis enabled
    LSM6DS0_GYRO_AE_XY       = 0x18,                    // X & Y axes enabled
    LSM6DS0_GYRO_AE_Z        = 0x20,                    // Only Z-axis enabled
    LSM6DS0_GYRO_AE_XZ       = 0x28,                    // X & Z axes enabled
    LSM6DS0_GYRO_AE_YZ       = 0x30,                    // Y & Z axes enabled
    LSM6DS0_GYRO_AE_XYZ      = 0x38                     // All axes enabled
}LSM6DS0_GYRO_AE_t;

// Gyroscope Decimation Mode
typedef enum {
    LSM6DS0_GYRO_DEC_DISABLED = 0x00,                   // NO decimation
    LSM6DS0_GYRO_DEC_X2       = 0x40,                   // Decimation update every 2 sample
    LSM6DS0_GYRO_DEC_X4       = 0x80,                   // Decimation update every 4 sample
    LSM6DS0_GYRO_DEC_X8       = 0xC0                    // Decimation update every 8 sample
}LSM6DS0_GYRO_DEC_t;

//Gyroscope Sleep Mode
typedef enum {
    LSM6DS0_GYRO_SLP_DISABLED = 0x00,                   // Gyroscope sleep mode disabled
    LSM6DS0_GYRO_SLP_ENABLED  = 0x40                    // Gyroscope sleep mode enabled
}LSM6DS0_GYRO_SLP_t;

// Gyroscope configuration structure
typedef struct {
    LSM6DS0_SAD_t   slaveaddress;           //LSM6DS0 Slave Address
    LSM6DS0_GYRO_ODR_t   outputdatarate;    // Gyroscope Output Data Rate
    LSM6DS0_GYRO_FS_t    fullscale;         // Gyroscope Full Scale
    LSM6DS0_GYRO_OUT_SEL_t outputselect;    // Gyroscope Output Selection
    LSM6DS0_GYRO_INT_SEL_t irqselect;       // Gyroscope Interrupt Selection
    LSM6DS0_GYRO_LP_MODE_t lowpowermode;    // Gyroscope Low Power Mode
    LSM6DS0_GYRO_HPCF_t HPCfrequency;       // Gyroscope High Pass Filter Cutoff Selection
    LSM6DS0_GYRO_AE_t axesenabling;         // Gyroscope Axes Enabling
    LSM6DS0_GYRO_DEC_t decimation;          // Gyroscope Decimation Mode
    LSM6DS0_END_t   endianess;              // LSM6DS0 Endianness
    LSM6DS0_BDU_t   blockdataupdate;        // LSM6DS0 Block Data Update
} LSM6DS0_GYRO_Config;

typedef struct{
    int16_t x;
    int16_t y;
    int16_t z;
} xl_output;

class LSM6DS0{
public:

    
    /* Public Methods */
    
    void setFIFO(LSM6DS0_FIFO_mode_t work_mode);
    void getFIFO(void);
    void resetFIFO(void);
    void Init(LSM6DS0_ACC_ODR_t odr, LSM6DS0_ACC_HR_t high_res, LSM6DS0_FIFO_mode_t work_mode);
    float getSensitivity(void);
    void LSM6DS0_reset(void);
    void storeFIFO(int *offset, xl_output *data_FIFO);
    void computeValue(xl_output *data_FIFO);
    void printValue(xl_output *data_FIFO);
    void printReg(void);
    
    
};
#endif