// Paladin board control code
// control switch matrix and function generator
// by Visarute Pinrod 4/23/2015
// with AD9833 control code from Po-Cheng Chen 07-04-2014

#include "mbed.h"
#include "SPI.h"
#include "DigitalOut.h"

SPI spi1(p5, p6, p7); //mosi, miso, sclk --> don't need MISO
DigitalOut cs_AD9833(p8); //use p8 as FSync data pin (same as chip select)

Serial pc(USBTX, USBRX);    //define serial USB port
DigitalOut myled1(LED1), myled2(LED2), myled3(LED3), myled4(LED4); //define LEDs

DigitalOut set1(p15);
DigitalOut set2(p16);
DigitalOut set3(p19);
DigitalOut set4(p18);
DigitalOut set5(p23);
DigitalOut set6(p25);
DigitalOut set7(p27);
DigitalOut set8(p29);

DigitalOut rset1(p13);
DigitalOut rset2(p14);
DigitalOut rset3(p20);
DigitalOut rset4(p17);
DigitalOut rset5(p24);
DigitalOut rset6(p26);
DigitalOut rset7(p28);
DigitalOut rset8(p30);

DigitalOut enableVout(p21);
DigitalOut disableVout(p22);

// get value from command format
int getVal(char *cmd){
    int val=-1;
    char valC[4];
    valC[0]=cmd[3];
    valC[1]=cmd[4];
    valC[2]=cmd[5];
    valC[3]=cmd[6];
    sscanf(valC,"%4x",&val);
    return val;
}

// Start SPI connection
void startSPI(){
    spi1.format(8,3);
    spi1.frequency(1000000);
    cs_AD9833 = 1;
}

void writeAD9833(short dat)
{
    cs_AD9833 = 0;
    spi1.write(dat>>8);
    spi1.write(dat&0xFF);
    cs_AD9833  = 1;
}

/////////////////////////////////////////////////////////////////////////
// Function for AD9833 control code
// Create by Po-Cheng Chen 07-04-2014
// Version 0.2 (MATLAB)
/////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////
// Calculate desired frequency
/////////////////////////////////////////////////////////////////////////
long calFreq(long freq)
{
    long  freq_cal;  //define freq calculated value
    float freq_val = 0.00000000;    //define ferq calculate tempotary value
    
    //calculate the frqe reg value
    //((desired frequency)/(reference frequency)) x 0x10000000.
    freq_val = (((float)(freq))/10000000);
    freq_cal = freq_val*0x10000000;
    return freq_cal;
}
/////////////////////////////////////////////////////////////////////////
// Set frequency register
/////////////////////////////////////////////////////////////////////////
void setFreq(long frequency)
{   
    int freq_MSB;   //define freq MSB reg value
    int freq_LSB;   //define freq LSB reg value
    
    long freq_cal = calFreq(frequency); //calculate the frequency register
    
    freq_MSB = (int)((freq_cal & 0xFFFC000)>>14); // shift 14 bits
    freq_LSB = (int)(freq_cal & 0x3FFF);
    
    freq_MSB = freq_MSB | 0x4000;  //assign freq reg address
    freq_LSB = freq_LSB | 0x4000;
    ///////////////////////////////////////////////////////////////////////////////////    
    //print the data from the serial port to verifiy the function
    //printf("freq_LSB 0x%x\n\r", freq_LSB);
    //printf("freq_MSB 0x%x\n\r", freq_MSB);
    ///////////////////////////////////////////////////////////////////////////////////
    writeAD9833(0x2100);      //write control reg - apply reset
    writeAD9833(freq_LSB);    //write freq reg - LSB
    writeAD9833(freq_MSB);    //write freq reg - MSB
    writeAD9833(0xC000);      //write phase reg - (0 for now)
    writeAD9833(0x2000);      //write control reg - disable reset
}

void swInitial(){
    set1 = 0;
    set2 = 0;
    set3 = 0;
    set4 = 0;
    set5 = 0;
    set6 = 0;
    set7 = 0;
    set8 = 0;
    rset1 = 0;
    rset2 = 0;
    rset3 = 0;
    rset4 = 0;
    rset5 = 0;
    rset6 = 0;
    rset7 = 0;
    rset8 = 0;
    enableVout = 0;
    disableVout = 0;
}

void setSwitch(DigitalOut& sw){
    sw = 1;
    wait(0.005);
    sw = 0;
    wait(0.05);
}

void setRotation(){
    setSwitch(set1);
    setSwitch(rset2);
    setSwitch(rset3);
    setSwitch(set4);
    setSwitch(set5);
    setSwitch(rset6);
    setSwitch(set7);
    setSwitch(rset8);
}

void rsetRotation(){
    setSwitch(rset1);
    setSwitch(set2);
    setSwitch(set3);
    setSwitch(rset4);
    setSwitch(rset5);
    setSwitch(set6);
    setSwitch(rset7);
    setSwitch(set8);
}

void setX(){
    setSwitch(rset1);
    setSwitch(rset2);
    setSwitch(rset3);
    setSwitch(set4);
    setSwitch(set5);
    setSwitch(set6);
    setSwitch(rset7);
    setSwitch(set8);
}

void rsetX(){
    setSwitch(set1);
    setSwitch(set2);
    setSwitch(set3);
    setSwitch(rset4);
    setSwitch(rset5);
    setSwitch(rset6);
    setSwitch(set7);
    setSwitch(rset8);
}

void setY(){
    setSwitch(set1);
    setSwitch(rset2);
    setSwitch(set3);
    setSwitch(set4);
    setSwitch(rset5);
    setSwitch(set6);
    setSwitch(rset7);
    setSwitch(rset8);
}

void rsetY(){
    setSwitch(rset1);
    setSwitch(set2);
    setSwitch(rset3);
    setSwitch(rset4);
    setSwitch(set5);
    setSwitch(rset6);
    setSwitch(set7);
    setSwitch(set8);
}

void setModeR(){
    swInitial();   
    setRotation();
    rsetRotation();
    setRotation();
    setRotation();
    swInitial();   
}

void setModeX(){
    swInitial();   
    setX();
    rsetX();
    setX();
    setX();
    swInitial();   
}

void setModeY(){
    swInitial();   
    setY();
    rsetY();
    setY();
    setY();
    swInitial();   
}

void enableOutput(){
    swInitial();  
    setSwitch(enableVout);
    swInitial();  
}

void disableOutput(){
    swInitial();  
    setSwitch(disableVout);
    swInitial();  
}

// Start connection
void initialization(){
    pc.baud(921600);
    startSPI(); // Start SPI connection 
    setModeR();
    disableOutput();
    enableOutput();
    enableOutput();
}


int main(){
    // Variable for main function
    char buf[16];       // Buffer for the command from PC
    int val;          // Value from command
    
    // Start system
    initialization();
    // Infinite loop: listen to serial command
    while(1) {
        if (pc.readable()) {
            //Command format: Start Mode Submode VALUe enD   S F F FFFF D
            pc.gets(buf, 16);
            //pc.printf("Get buf: %s\n",buf);
            if ((buf[0] == 'S')&&(buf[7] == 'D')) {
                //Mode switch
                switch(buf[1]) { 
                //set axis mode
                case 'm': 
                    // Submode switch
                    switch(buf[2]) { 
                    case 'r':
                        setModeR();
                        break;
                    case 'x':
                        setModeX();
                        break;
                    case 'y':
                        setModeY();
                        break;
                    default:
                        break;
                    }
                //Output enable
                case 'o': 
                    // Submode switch
                    switch(buf[2]) { 
                    case 'T':
                        enableOutput();
                        break;
                    case 'F':
                        disableOutput();
                        break;
                    default:
                        break;
                    }
                    break;
                default:
                    break;
                }
            }
        }     
    }
    return 0;
}