 
#include "mbed.h"
#include "cc3000.h"
#include "wifi.h"
#include "UDPSocket.h"
#include "tcpTests.h"
#include "main.h"
#include "USBSerial.h"
#include "NTPClient.h"
#include <time.h>

using namespace mbed_cc3000;

/* cc3000 module declaration specific for user's board. Check also init() */
#if (MY_BOARD == WIGO)
cc3000 wifi(PTA16, PTA13, PTD0, SPI(PTD2, PTD3, PTC5), PORTA_IRQn);
#elif (MY_BOARD == WIFI_DIPCORTEX)
cc3000 wifi(p28, p27, p30, SPI(p21, p14, p37));
Serial uart(p19, p20);
USBSerial pc;               // USB CDC serial port
#else

#endif

#ifndef CC3000_UNENCRYPTED_SMART_CONFIG
  const uint8_t smartconfigkey[] = {0x73,0x6d,0x61,0x72,0x74,0x63,0x6f,0x6e,0x66,0x69,0x67,0x41,0x45,0x53,0x31,0x36};
#else
  const uint8_t smartconfigkey = 0;
#endif

const int ECHO_SERVER_PORT_UDP = 81;
uint8_t *HostToPing = (uint8_t *)"google.com";
tNetappIpconfigRetArgs ipinfo;
extern char tmpBuffer[512];

MENU_LEVEL currentMenu = MENU_TOP;

bool Connected = false;
bool UsingSmartConfig = false;
char _deviceName[] = "CC3000";

// ------------------------------------------------------------------------------------------------------------
/*!
    @brief Resolve a hostname and ping it
*/
// ------------------------------------------------------------------------------------------------------------
void PingTest ( void )
{
uint32_t ip;
int32_t resolveRetCode = 0;
    
    pc.printf("Get an IP address for %s\r\n",HostToPing);
    resolveRetCode = wifi._socket.gethostbyname(HostToPing,strlen((const char *)HostToPing), &ip);
    pc.printf("gethostbyname Returned code : %i \r\n", resolveRetCode);
    
    if (resolveRetCode > -1) 
    {
        uint8_t add0 = (ip >> 24);
        uint8_t add1 = (ip >> 16);
        uint8_t add2 = (ip >> 8);
        uint8_t add3 = (ip >> 0);
        pc.printf("IP address of %s: %d.%d.%d.%d \r\n", HostToPing, add0, add1, add2, add3);
        
        pc.printf("Sending ping\r\n");
        uint32_t reply_count = wifi.ping(ip, 5, 500, 32);
        pc.printf("Received %d replies\r\n", reply_count);
        pc.printf("Ping complete.\r\n");
    } 
    else 
    {
        pc.printf("Failed to resolve\r\n");
    }
}

// ------------------------------------------------------------------------------------------------------------
/*!
    @brief Test the NTP library
*/
// ------------------------------------------------------------------------------------------------------------
void NtpTest ( void )
{
NTPClient ntp;
struct tm *currentTime;
    if (ntp.setTime("0.pool.ntp.org",123,10000) == 0)
    {
        pc.printf("\r\nGot the time successfully\r\n");
        currentTime = localtime(&ntp.NTPLastResult);
        pc.printf("NTP Response : %s\r\n", asctime(currentTime));
    }
    else
    {
        pc.printf("NTP Update Failed\r\n");
    }
}

// ------------------------------------------------------------------------------------------------------------
/*!
    @brief Send a UDP Packet, wait for response
*/
// ------------------------------------------------------------------------------------------------------------
void UdpClientTest ( void )
{
UDPSocket socket;
int n = 0;
Endpoint outEndpoint;
Endpoint inEndpoint;

    if (0 == socket.bind(ECHO_SERVER_PORT_UDP) )
    {
        
        // 2 second timeout
        socket.set_blocking(false, 2000);
        
        pc.printf("\r\n!! Press any key to stop !!\r\n\r\n");
    
        while (1)
        {
            if( outEndpoint.set_address(ECHO_SERVER_ADDRESS, ECHO_SERVER_PORT_UDP) < 0 )
            {
                pc.printf("Failed to set endpoint address.\r\n");
                break;
            }
            else 
            {
                if ( socket.sendTo( outEndpoint, hello, sizeof(hello) ) < 0 )
                {
                    pc.printf("Failed to send the packet.\r\n");
                }
                else
                {
                    // Message sent, recv reply
                    pc.printf("UDP Socket Sent : %s \r\n", hello);   
                    n = socket.receiveFrom( inEndpoint, tmpBuffer, sizeof(tmpBuffer) );
                    if ( n < 0 )
                    {
                        pc.printf("Failed to recv the UDP packet.\r\n");
                    }
                    else
                    {
                        tmpBuffer[n] = '\0';
                        pc.printf("UDP Socket Recv'd : %s \r\n", tmpBuffer);    
                    }
                }
            }
    
            // Should we stop?
            if ( pc.readable() )
            {
                pc.getc();
                break;
            }
        }

        if ( wifi.is_connected() )
        {        
            socket.close();
        }
    }
    else
    {
        // Failed to bind to the socket
    }
}

// ------------------------------------------------------------------------------------------------------------
/*!
    @brief Listen on a UDP port for messages
*/
// ------------------------------------------------------------------------------------------------------------
void UdpServerTest ( void )
{
UDPSocket socket;
Endpoint client;
int n = 0;

    if (0 == socket.bind(ECHO_SERVER_PORT_UDP) )
    {   
        pc.printf("\r\n!! Press any key to stop listening !!\r\n\r\n");
        
        while (true) 
        {
            pc.printf("Waiting for packet...\r\n");
            n = socket.receiveFrom(client, tmpBuffer, sizeof(tmpBuffer));
        
            pc.printf("Received packet from: %s\n", client.get_address());
            socket.sendTo(client, tmpBuffer, n);
        
            // Should we stop?
            if ( pc.readable() )
            {
                pc.getc();
                break;
            }
        }
    }
}

// ------------------------------------------------------------------------------------------------------------
/*!
    @brief Print menu header
*/
// ------------------------------------------------------------------------------------------------------------
void Menu_PrintHeader ( void )
{
    if (( wifi.is_enabled() ) && ( wifi.is_dhcp_configured() ))
    {
        wifi.get_ip_config(&ipinfo);
    }
    
    pc.printf("\r\n");
    pc.printf("+-------------------------------------------+\r\n");
    pc.printf("|   WiFi DipCortex / CC3000 Kitchen Sink    |\r\n");
    pc.printf("+-------------------------------------------+\r\n");
    if (! wifi.is_enabled() )
    {
        pc.printf("|   CC3000 Disabled                         |\r\n"); 
    }
    else if ( wifi.is_dhcp_configured() ) 
    {
        pc.printf("|   SSID : %-33s|\r\n", ipinfo.uaSSID);
        pc.printf("|   IP : %-35s|\r\n", wifi.getIPAddress());   
    }
    else if ( wifi.is_connected() )
    {
        pc.printf("|   Connecting, waiting for DHCP            |\r\n"); 
    }
    else
    {
        pc.printf("|   Not Connected                           |\r\n");   
    }
    pc.printf("+-------------------------------------------+\r\n");
    pc.printf("\r\n");
}

// ------------------------------------------------------------------------------------------------------------
/*!
    @brief Control the wifi connection
*/
// ------------------------------------------------------------------------------------------------------------
char WaitForSerialCommand ( void )
{
char charIn = 0;
char prevCharIn;

    pc.printf("Enter command character : ");
  
    while (1)
    {
        prevCharIn = charIn;
        charIn = pc.getc();
        
        pc.printf("%c", charIn);
        if ((charIn == '\n') || (charIn == '\r'))
        {
            break;
        }
    }
    
    return ( prevCharIn );
}

// ------------------------------------------------------------------------------------------------------------
/*!
    @brief Configure the module for smart connect mode allow configuration over the air
*/
// ------------------------------------------------------------------------------------------------------------
void SmartConfig ( void )
{
    pc.printf("\r\nStarting Smart config, waiting for message from smartphone app ....\r\n");
    // We dont want to auto reconnect to an access point
    wifi._wlan.ioctl_set_connection_policy(0, 0, 0);
    
    // start smart config will disconnect, set the prefix
    // wait for a message via a SmartConfig app, store it to the profile list
    // finally it will reenable auto connection, triggering the module to connect to the new access point
    wifi.start_smart_config(0);
    
    UsingSmartConfig = true;
}

// ------------------------------------------------------------------------------------------------------------
/*!
    @brief Print cc3000 information
*/
// ------------------------------------------------------------------------------------------------------------
void print_cc3000_info() {
uint8_t myMAC[8];
uint8_t buffer[2];
int32_t status = 0;
tNetappIpconfigRetArgs ipinfo2;
tUserFS cc_user_info;
const char * WIFI_STATUS[] = {"Disconnected", "Scanning", "Connecting", "Connected"};

    wifi.get_user_file_info((uint8_t *)&cc_user_info, sizeof(cc_user_info));
    wifi.get_mac_address(myMAC);
    pc.printf(" MAC address : %02x:%02x:%02x:%02x:%02x:%02x\r\n", myMAC[0], myMAC[1], myMAC[2], myMAC[3], myMAC[4], myMAC[5]);
    
    if (! wifi._nvmem.read_sp_version( (unsigned char*)&buffer ) )
    {
        pc.printf(" CC3000 Firmware Version : %u.%u \r\n", buffer[0], buffer[1]);
    }
    else
    {
        pc.printf(" CC3000 Read nvmem failed!");
    }
    
    status = wifi._wlan.ioctl_statusget();
    if (( status > -1 ) && ( status < 4 ))
    {
        pc.printf(" Wifi Status    : %s\r\n", WIFI_STATUS[status]);
    }
    else
    {
        pc.printf(" Wifi Status    : %d\r\n", status);
    }
    
    if ( wifi.is_dhcp_configured() ) 
    {
        wifi.get_ip_config(&ipinfo2);
        pc.printf(" Connected to   : %s \r\n", ipinfo2.uaSSID);
        pc.printf(" IP             : %d.%d.%d.%d \r\n", ipinfo2.aucIP[3], ipinfo2.aucIP[2], ipinfo2.aucIP[1], ipinfo2.aucIP[0]);   
        pc.printf(" Gateway        : %d.%d.%d.%d \r\n", ipinfo2.aucDefaultGateway[3], ipinfo2.aucDefaultGateway[2], ipinfo2.aucDefaultGateway[1], ipinfo2.aucDefaultGateway[0]);  
        pc.printf(" Subnet         : %d.%d.%d.%d \r\n", ipinfo2.aucSubnetMask[3], ipinfo2.aucSubnetMask[2], ipinfo2.aucSubnetMask[1], ipinfo2.aucSubnetMask[0]);  
        pc.printf(" DNS            : %d.%d.%d.%d \r\n", ipinfo2.aucDNSServer[3], ipinfo2.aucDNSServer[2], ipinfo2.aucDNSServer[1], ipinfo2.aucDNSServer[0]);  
        
        pc.printf(" Cached IP      : %s \r\n", wifi.getIPAddress());   
        pc.printf(" Cached Gateway : %s \r\n", wifi.getGateway());   
        pc.printf(" Cached Subnet  : %s \r\n", wifi.getNetworkMask());   

    }
    else
    {
         pc.printf(" Not connected \r\n");
    }
}


// ------------------------------------------------------------------------------------------------------------
/*!
    @brief Control the wifi connection
*/
// ------------------------------------------------------------------------------------------------------------
void Menu_ConnectionControl ( void )
{
uint32_t ip = 0;
    Menu_PrintHeader();
    pc.printf(" 1 - Enable auto connect to any previous access point\r\n");
    pc.printf(" 2 - Disable auto connect \r\n");
    pc.printf(" 3 - Connect to %s \r\n", SSID);
    pc.printf(" 4 - Disconnect \r\n");
    pc.printf(" 5 - Start SmartConfig \r\n");  
    pc.printf(" 6 - Erase profiles \r\n");  
    pc.printf(" 7 - Get Status \r\n");
    pc.printf(" x - Top Menu \r\n");
    
    pc.printf("\r\n");
        
    switch(WaitForSerialCommand()) 
    {
        case '1':      
            UsingSmartConfig = false;
            wifi._wlan.ioctl_set_connection_policy(0, 0, 1);
        break;
        
        case '2':      
            wifi._wlan.ioctl_set_connection_policy(0, 0, 0);
        break;
        
        case '3':     
        
            UsingSmartConfig = false;
            
            // Enable DHCP
            wifi._netapp.dhcp(&ip, &ip, &ip, &ip);
            
            if ( AP_SECURITY == NONE )
            {
                wifi.connect_non_blocking((uint8_t *)SSID, 0, AP_SECURITY);
            }
            else
            {
                pc.printf("\r\n Connecting to : %s key : %s", SSID, AP_KEY );
                wifi.connect_non_blocking((uint8_t *)SSID, (uint8_t *)AP_KEY, AP_SECURITY);
                
            }      
        break;
        
        case '4' :
            // Stop the module re-connecting
            wifi._wlan.ioctl_set_connection_policy(0, 0, 0);
            // Then disconnect
            wifi.disconnect();
        break;
        
        case '5' :
            SmartConfig();
        break;
        
        case '6' :
            wifi._wlan.ioctl_del_profile(255);
        break;
        
        case '7' :
            print_cc3000_info();
        break;
        
        case '8':
            //wifi._spi.manualIrqCheck();
        break;
        case '0':  
        case 'x':      
            currentMenu = MENU_TOP;
        break;
    }
}

// ------------------------------------------------------------------------------------------------------------
/*!
    @brief 
*/
// ------------------------------------------------------------------------------------------------------------
void Menu_UdpControl ( void )
{
    Menu_PrintHeader();

    pc.printf(" 1 - UDP Client, Connect to %s:%d\r\n", ECHO_SERVER_ADDRESS, ECHO_SERVER_PORT_UDP);
    pc.printf(" 2 - UDP Server \r\n");
    pc.printf(" 3 - NTP Client \r\n");
    pc.printf(" x - Exit to top menu \r\n");
    
    pc.printf("\r\n");
    
    switch(WaitForSerialCommand())  
    {
        case '1':      
            UdpClientTest();
        break;
        case '2':      
            UdpServerTest();
        break;
        case '3':      
            NtpTest();
        break;
        case '0': 
        case 'x':      
            currentMenu = MENU_TOP;
        break;
    }
}

// ------------------------------------------------------------------------------------------------------------
/*!
    @brief 
*/
// ------------------------------------------------------------------------------------------------------------
void Menu_TcpControl ( void )
{
    Menu_PrintHeader();

    pc.printf(" 1 - TCP Client, Connect to %s:%d\r\n", ECHO_SERVER_ADDRESS, ECHO_SERVER_PORT_TCP);
    pc.printf(" 2 - TCP Server, listen on %d.%d.%d.%d:%d\r\n", ipinfo.aucIP[3], ipinfo.aucIP[2], ipinfo.aucIP[1], ipinfo.aucIP[0], ECHO_SERVER_PORT_TCP);
    pc.printf(" 3 - Web Socket Write \r\n");
    pc.printf(" 4 - Web Socket Read \r\n");
    pc.printf(" 5 - Web Socket Write ADCs \r\n");
    pc.printf(" 6 - HTTP Client \r\n");
    pc.printf(" 7 - Xively Post ADC's \r\n");
    pc.printf(" x - Exit to top menu ");
    pc.printf("\r\n");
    
    switch(WaitForSerialCommand()) 
    {
        case '1':      
            TcpClientTest();
        break;
        case '2':      
            TcpServerTest();
        break;
        case '3':      
            WebSocketTest();
        break;
        case '4':
            WebSocketReadTest();
        break;
        case '5':
            WebSocketAdcStream();
        break;
        case '6':
            HttpClientTest();
        break;
        case '7':
            XivelySimpleTest();
        break;
        case '0':   
        case 'x':      
            currentMenu = MENU_TOP;
        break;
    }
}


// ------------------------------------------------------------------------------------------------------------
/*!
    @brief Display a menu to the user
*/
// ------------------------------------------------------------------------------------------------------------
void Menu_Top ( void )
{
    Menu_PrintHeader();
    
    pc.printf(" 1 - Connection control menu \r\n" );
    pc.printf(" 2 - TCP test menu \r\n" );
    pc.printf(" 3 - UDP test menu \r\n" );
    pc.printf(" 4 - Connection status \r\n");
    pc.printf(" 5 - Ping - %s \r\n", HostToPing);
    pc.printf(" x - Top Menu \r\n");
    
    pc.printf("\r\n");
 
    switch(WaitForSerialCommand()) 
    {
        case '1':      
            currentMenu = MENU_CONNECTION;
        break;
        case '2':      
            currentMenu = MENU_TCP;
        break;
        case '3':      
            currentMenu = MENU_UDP;
        break;
        case '4':      
            print_cc3000_info();
        break;
        case '5':      
            PingTest();
        break;
        case '0':  
        case 'x':      
            currentMenu = MENU_TOP;
        break;
    }
}

// ------------------------------------------------------------------------------------------------------------
/*!
    @brief Pick which menu to display
*/
// ------------------------------------------------------------------------------------------------------------
void MenuSwitch ( void )
{
//bool connected = false;

    if ( wifi.is_dhcp_configured() )
    {
        if (!Connected)
        {
            // We have just connected
            Connected = true;
            
            if ( UsingSmartConfig )
            {
                // Start the mdns service, this tells any smart config apps listening we have succeeded
                wifi._socket.mdns_advertiser(1, (uint8_t *)_deviceName, strlen(_deviceName));
                
                UsingSmartConfig = false;
            }
        }
    }
    else
    {
        Connected = false;

    }
        
    switch ( currentMenu )
    {
        case MENU_TOP :
            Menu_Top();
        break;
        
        case MENU_CONNECTION :
            Menu_ConnectionControl();
        break;
        
        case MENU_TCP :
            if (Connected) 
            {
                Menu_TcpControl();
            }
            else
            {
                currentMenu = MENU_TOP;
            }
        break;
        
        case MENU_UDP :
            if (Connected) 
            {
                Menu_UdpControl();
            }
            else
            {
                currentMenu = MENU_TOP;
            }
        break;
        
        default :
            pc.printf("Unknown command\r\n");
        break;
    }
}

// ------------------------------------------------------------------------------------------------------------
/*!
    @brief Init
*/
// ------------------------------------------------------------------------------------------------------------
void init() 
{
    NVIC_SetPriority(SSP1_IRQn, 0x0); 
    NVIC_SetPriority(PIN_INT0_IRQn, 0x1);
    
    // SysTick set to lower priority than Wi-Fi SPI bus interrupt
    NVIC_SetPriority(SysTick_IRQn, 0x2); 
    
    
    // Enable RAM1
    LPC_SYSCON->SYSAHBCLKCTRL |= (0x1 << 26);
}

// ------------------------------------------------------------------------------------------------------------
/*!
    @brief main loop
*/
// ------------------------------------------------------------------------------------------------------------
int main( void ) 
{   
    // Initalise the WiFi Module
    init(); 
    
    uart.baud(SERIAL_BAUD_RATE);
    
    wait(1);
    wifi.start(0);
    
    wait_ms(750);

    while (1)
    {
        MenuSwitch();
    }
}
