/* Copyright (c) 2007 Axel Wachtler
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id$ */
/**
 * @file
 * @brief Interface for @ref grpTrx.
 * @ingroup grpTrx
 */
#ifndef TRANSCEIVER_H
#define TRANSCEIVER_H

/**
 *  @addtogroup grpTrx
 *  @{
 */

/* === Includes ============================================================== */
#include "board.h"

#include <stdbool.h>

/* === Externals ============================================================= */

/* === Types ================================================================= */

#if defined(DOXYGEN)

    /* this types are defined in at86rfXXX.{h,txt} in order
       to provide a radio abstraction */

    /** Data Type for Transceiver SRAM address
     */
    typedef uint8_t trx_ramaddr_t;

    /** Data Type for Transceiver register value
     */
    typedef uint8_t trx_regval_t;

    /** Data Type for Transceiver register address
     */
    typedef uint8_t trx_regaddr_t;

#endif

/** Data Type for Transceiver IRQ callback function
 */
typedef void (*trx_irq_handler_t)(uint8_t cause);



/* === Macros ================================================================ */
/* error codes */
/** trx function succeeded */
#define TRX_OK        (0)
/** trx init function failed (TRX_OFF not reached after reset) */
#define TRX_INIT_FAIL (1)
/** trx pll check function failed (PLL_LOCK coult not be observed in PLL_ON) */
#define TRX_PLL_FAIL  (2)

#define INVALID_PART_NUM (2)  /**< flag for invalid part number */
#define INVALID_REV_NUM  (1)  /**< flag for invalid revision number */

/* Data Rate macros, generated by python Tools/cmdhash.py  `cat rates.txt` */
#define BPSK20 (0x52)
#define BPSK20_STR "BPSK20"
#define BPSK40 (0x92)
#define BPSK40_STR "BPSK40"
#define OQPSK100 (0x90)
#define OQPSK100_STR "OQPSK100"
#define OQPSK200 (0x93)
#define OQPSK200_STR "OQPSK200"
#define OQPSK250 (0x33)
#define OQPSK250_STR "OQPSK250"
#define OQPSK400 (0x95)
#define OQPSK400_STR "OQPSK400"
#define OQPSK500 (0x94)
#define OQPSK500_STR "OQPSK500"
#define OQPSK1000 (0x34)
#define OQPSK1000_STR "OQPSK1000"
#define OQPSK2000 (0x54)
#define OQPSK2000_STR "OQPSK2000"

#define RATE_NONE (0xFF)

/** Maximum size in bytes of an IEEE 802.15.4 frame */
#ifndef MAX_FRAME_SIZE
# define MAX_FRAME_SIZE (127)
#endif

/* channel handling */
#define TRX_NEXT_CHANNEL(x) ((channel_t)(x+1) > TRX_MAX_CHANNEL ? TRX_MAX_CHANNEL : x+1)
#define TRX_PREV_CHANNEL(x) ((channel_t)(x-1) < TRX_MIN_CHANNEL ? TRX_MIN_CHANNEL : x-1)
#define TRX_NEXT_CHANNEL_WRAP(x) ((channel_t)(x+1) > TRX_MAX_CHANNEL ? TRX_MIN_CHANNEL : x+1 )
#define TRX_PREV_CHANNEL_WRAP(x) ((channel_t)(x-1) < TRX_MIN_CHANNEL ? TRX_MAX_CHANNEL : x-1 )


#if defined (SR_MASK_AMI) || defined(DOXYGEN)
/** @brief Enable AMI IRQ. */
# define TRX_IRQ_AMI_EI() trx_bit_write(SR_MASK_AMI, 1);
/** @brief Disable AMI IRQ. */
# define TRX_IRQ_AMI_DI() trx_bit_write(SR_MASK_AMI, 0);
#endif

#if defined (SR_MASK_BAT_LOW) || defined(DOXYGEN)
/** @brief Enable BAT_LOW IRQ. */
# define TRX_IRQ_BAT_LOW_EI() trx_bit_write(SR_MASK_BAT_LOW, 1);
/** @brief Disable BAT_LOW IRQ. */
# define TRX_IRQ_BAT_LOW_DI() trx_bit_write(SR_MASK_BAT_LOW, 0);
#endif

#if defined (SR_MASK_CCA_ED_READY) || defined(DOXYGEN)
/** @brief Enable CCA_ED_READY IRQ. */
# define TRX_IRQ_CCA_ED_READY_EI() trx_bit_write(SR_MASK_CCA_ED_READY, 1);
/** @brief Disable CCA_ED_READY IRQ. */
# define TRX_IRQ_CCA_ED_READY_DI() trx_bit_write(SR_MASK_CCA_ED_READY, 0);
#endif

#if defined (SR_MASK_PLL_UNLOCK) || defined(DOXYGEN)
/** @brief Enable PLL_UNLOCK IRQ. */
# define TRX_IRQ_PLL_UNLOCK_EI() trx_bit_write(SR_MASK_PLL_UNLOCK, 1);
/** @brief Disable PLL_UNLOCK IRQ. */
# define TRX_IRQ_PLL_UNLOCK_DI() trx_bit_write(SR_MASK_PLL_UNLOCK, 0);
#endif

#if defined (SR_MASK_RX_START) || defined(DOXYGEN)
/** @brief Enable RX_START IRQ. */
# define TRX_IRQ_RX_START_EI() trx_bit_write(SR_MASK_RX_START, 1);
/** @brief Disable RX_START IRQ. */
# define TRX_IRQ_RX_START_DI() trx_bit_write(SR_MASK_RX_START, 0);
#endif

#if defined (SR_MASK_TRX_IRQ_END) || defined(DOXYGEN)
/** @brief Enable TRX_IRQ_END IRQ. */
# define TRX_IRQ_TRX_IRQ_END_EI() trx_bit_write(SR_MASK_TRX_IRQ_END, 1);
/** @brief Disable TRX_IRQ_END IRQ. */
# define TRX_IRQ_TRX_IRQ_END_DI() trx_bit_write(SR_MASK_TRX_IRQ_END, 0);
#endif

#if defined (SR_MASK_TRX_IRQ_START) || defined(DOXYGEN)
/** @brief Enable TRX_IRQ_START IRQ. */
# define TRX_IRQ_TRX_IRQ_START_EI() trx_bit_write(SR_MASK_TRX_IRQ_START, 1);
/** @brief Disable TRX_IRQ_START IRQ. */
# define TRX_IRQ_TRX_IRQ_START_DI() trx_bit_write(SR_MASK_TRX_IRQ_START, 0);
#endif

#if defined (SR_MASK_UR) || defined(DOXYGEN)
/** @brief Enable TX/RX underun IRQ. */
# define TRX_IRQ_UR_EI() trx_bit_write(SR_MASK_UR, 1);
/** @brief Disable TX/RX underun IRQ. */
# define TRX_IRQ_UR_DI() trx_bit_write(SR_MASK_UR, 0);
#endif

/**
 *  @}
 */
#endif /* TRANSCEIVER_H */
