#include "mbed.h"
#include "FSR.h"

#include <ctype.h>
#include <string>

#include "SDFileSystem.h"
#include "Adafruit_FONA.h"

#define FONA_RST p12
#define FONA_TX p13
#define FONA_RX p14
#define FONA_RI p11

Serial pcSerial(USBTX, USBRX);

DigitalOut led1(LED1);      //Fona status - on(connecting), off(disconnecting)
DigitalOut led2(LED2);
DigitalOut led3(LED3);      //Pressure status - on(child on seat), off(off seat)
DigitalOut led4(LED4);      //Temperature status - on(Too hot), off(not hot)

Adafruit_FONA fona(FONA_TX, FONA_RX, FONA_RST, FONA_RI);    //GSM

SDFileSystem sd(p5, p6, p7, p8, "sd");      //SD card reader

DigitalOut CtrlRelay(p16);      //Relay Controller

FSR bottomFSR1(p20, 10); // Pin 17-20 is used as the AnalogIn pin and a 10k resistor is used as a voltage divider for the FSR
FSR topFSR1(p19, 10);
FSR topFSR2(p18, 10);
FSR bottomFSR2(p17, 10);

I2C i2c(p28, p27);      //I2C commands

Serial bt(p9, p10);     //Bluetooth module

float temp, avgTemp; //temperature in degrees C
const int addr1 = 0xB4;     //Default MLX90614 address - 0x5A
const int addr2 = 0x08;     //Changed another MLX90614 address to 0x04
char data[3];
bool danger;
int onForce;

float temp1,temp2,temp3,temp4,avgF;

char pnum[22];
char message[141]= "Emergency! Your child is still in the Smart Car Seat!";
//Message to send to phone and contacts

class FonaEventListener : public Adafruit_FONA::EventListener {
    virtual void onRing() {
        led1 = 1;
    }
    
    virtual void onNoCarrier() {
        led1 = 0; 
    }
};
FonaEventListener fonaEventListener;
//GSM Module initalization

bool onSeat(float fLevel);      //Checks if there is a child on the seat
float readTemp(int addr, char* cmd);    //Reads the temperature through the I2/C
void callback(void);        //Changes the stored number on the SD card

int main() {
    pcSerial.baud(9600);        //Set the serial baudrate to 9600
    wait(2);                    //Wait 2 secs to initialize and settle
    i2c.frequency(50000);       //Set the I2C frequency to 50kHz for the MLX90614
    wait(2);                    //Wait another 2 secs for initalization
    
    
    char cmd1[1];
    char cmd2[1];
    cmd1[0] = 0x07;     //Command to read object temperature
    cmd2[0] = 0x06;     //Command to read ambient temperature
    
    
    pcSerial.printf("FONA basic test\r\n");
    pcSerial.printf("Initializing....(May take 3 seconds)\r\n");
    // See if the FONA is responding
    if (! fona.begin(9600)) {
        pcSerial.printf("Couldn't find FONA\r\n");
        while (1);
    }
    fona.setEventListener(&fonaEventListener);
    pcSerial.printf("FONA is OK\r\n");
    
    
    bt.attach(&callback);       //Allows the app to change the phone number
    
    
    //Reads the stored phone number
    FILE *fpr = fopen("/sd/mydir/myphone.txt", "r");
    if(fpr == NULL) {
        error("Could not open file for read\n");
    }
    fscanf(fpr,"%s", pnum); 
    pcSerial.printf("Phone Number: %s\n", pnum);
    fclose(fpr);
    
    
    danger = false;
    
    while (1) {
        if (onSeat(0.2))        //Checks if child is on the seat
                                // 0.2 is very low, 0.5 for moderate touch, 0.8 for heavy pressing
        {   
            led3 = 1;
            
            temp1 = readTemp(addr1, cmd1);      //Read object temperature (C)
            temp2 = readTemp(addr2, cmd1);
            //temp3 = readTemp(addr1, cmd2);    //Read ambient temperature (C)
            //temp4 = readTemp(addr2, cmd2);
            
            avgTemp = (temp1+temp2)/2;          //Average value
            //avgTemp = (temp3+temp4)/2;
            printf("%5.2f\r\n",avgTemp);
            
            if ( avgTemp > 30)      //If value exceed a "bad" temperature
            {
                led4 = 1;
                CtrlRelay = 1;      //Turn on cooling system
                danger = true;
            } else if ( avgTemp < 25)   //When it falls back down
            {
                led4 = 0;
                CtrlRelay = 0;      //Turn off cooling system
            }
            if (danger) {       //When child is in "danger"
                fona.sendSMS(pnum, message);    //FONA will send message to the contact
                wait(1);        //Adjust wait time after sending a message
            }
        } else {
            led3 = 0;
            led4 = 0;
            CtrlRelay = 0;      //When child is no longer on the seat
            danger = false;     //The system resets
        }
        wait(1);        //Adjust wait time after reading
    }
}

float readTemp(int addr, char* cmd) {   //read the value from the MLX90614
    i2c.write(addr, cmd, 1, true);
    i2c.read(addr, data, 3);        //Two bytes of data + 1 word of codes
    
    temp = (((float)((data[1]<<8)|data[0]))*0.02) - 273.15;
    return temp;
}

bool onSeat(float fLevel) {     //Detects child by activating enough pressure sensors
    onForce = 0;
    if  (bottomFSR1.readRaw() > fLevel) {
        onForce += 1;
    }
    if  (bottomFSR2.readRaw() > fLevel) {
        onForce += 1;
    }
    if  (topFSR1.readRaw() > fLevel) {
        onForce += 1;
    }
    if  (topFSR2.readRaw() > fLevel) {
        onForce += 1;
    }
    return onForce >= 2;
}

void callback(void) {       //Rewrites the phone number saved in the SD card
    bt.gets(pnum,11);
    pcSerial.printf("New number: %s\n", pnum);
            
    FILE *fp = fopen("/sd/mydir/myphone.txt", "w");
    if(fp == NULL) {
        error("Could not open file for write\n");
    }
    fprintf(fp, "%s\n", pnum);
    fclose(fp);
}