#include "SPS.h"

//! SPS, class default constructor
/*! 
    Initializing different class member values, creating a header string.       
*/
SPS::SPS(){

    fileNameString = "";
    currentString = "";
    currentErrString = "";
    lastString = "";
    headerString = "";
    crc_tab16_init = 0;
    errStatus = true;
           
};


//! UpdateCurrentString, sps string creation method, taking data and info as parameters.
/*! 

    Generates an sps string containing everything needed in a BMAG sps string including crc, brackets etc.

    \param tag: string containing the tag for the sps logging unit eg. "BMAG"
    \param identifier_id: string containing the identifier id of the unit.
    \param group_id: string containing the group id of the unit.
    \param date: string containing the date in the format "YYYY/MM/DD"  
    \param time: string containing the current time in format "HH:MM:SS.FFF"
    \param ZZZ: string containing "ZZZ" :)
    \param encoding: string containing encoding as described in the *.sps documentation
    \param source_id: string containing the source id of the unit eg. "0000"
    \param interpreter_id: string containing the interpreter id of the unit eg. "00"
    \param data_line_version: string containing data line version as described in the *.sps documentation, eg. "01" 
    \param source_firmware_version: string containing the firmware version of the unit eg. "1.2"
    \param barcode: string containing the barcode of the unit, eg. "12345" 
    \param latitude: string containing the latitude of the unit in decimal degrees. For further info read the *.sps documentation.
    \param longitude: string containing the longitude of the unit in decimal degrees. For further info read the *.sps documentation.
    \param gpsFixFlag: bool containing the gps fix status, 1 if gps fix is present, else 0.
    \param batteryvoltage: string containing the battery voltage of the units battery pack, eg. "14.2"
    \param preassure: string containing preassure measurement in hPa
    \param temperature: string containing temperature measured in deg C
    \param humidity: string containing air humidity percentage
    \param mag_time: string containing the last timestamp received from GSM-19 device eg. "000048.0"
    \param mag_nt: string containing the last mag measurement in nT eg. "036418.77"
    \param mag_sq: string containing info about the measurement quality of the last mag measurement eg. "99"
              
*/
void SPS::UpdateCurrentString(string tag, string identifier_id, string group_id, string date, string time, string ZZZ,string encoding, string source_id, string interpreter_id, string data_line_version, string source_firmware_version, string interpreter_firmware_version, string barcode, string latitude, string longitude, bool gpsFixFlag, string batteryvoltage, string preassure, string temperature, string humidity, string altitude,string mag_time, string mag_nt, string mag_sq, Serial * dbg){

    this->currentString = "";
    this->currentString.resize(256);
    
    int strLength = 0;
    char checkSum[5];
    memset(checkSum,'\0',5);
    unsigned short chkSum = 0;
    
    addToCurrentString(LINESTART);
    addToCurrentString(tag);
    addToCurrentString(SPACE);
    addToCurrentString(identifier_id);
    addToCurrentString(SPACE);
    addToCurrentString(group_id);
    addToCurrentString(SPACE);
    addToCurrentString(date);
    addToCurrentString(SPACE);
    addToCurrentString(time);
    addToCurrentString(SPACE);
    if(!gpsFixFlag){
        addToCurrentString("NOC");    
    }
    if(gpsFixFlag){
        addToCurrentString("ZZZ");    
    }  
    addToCurrentString(SPACE);
    addToCurrentString(HEADEREND);
    addToCurrentString(SPACE);
    addToCurrentString(source_id);
    addToCurrentString(SPACE);
    addToCurrentString(interpreter_id);
    addToCurrentString(SPACE);
    addToCurrentString(data_line_version);
    addToCurrentString(SPACE);
    addToCurrentString(encoding);
    addToCurrentString(SPACE);
    addToCurrentString(source_firmware_version);
    addToCurrentString(SPACE);
    addToCurrentString(interpreter_firmware_version);
    addToCurrentString(SPACE);
    addToCurrentString(HEADEREND);
    addToCurrentString(SPACE);
    addToCurrentString(barcode);
    addToCurrentString(SPACE);
    if(gpsFixFlag){
        addToCurrentString(latitude);    
    }
    if(!gpsFixFlag || (strlen(latitude.c_str()) < 5)){
        addToCurrentString("NaN");    
    }  
    addToCurrentString(SPACE);
    if(gpsFixFlag && (strlen(longitude.c_str()) > 5)){
        addToCurrentString(longitude);    
    }
    if(!gpsFixFlag || (strlen(longitude.c_str()) < 5)){
        addToCurrentString("NaN");    
    } 
    addToCurrentString(SPACE); 
    
    if(gpsFixFlag){
        addToCurrentString('1');    
    }
    if(!gpsFixFlag){
        addToCurrentString('0');
    }
    addToCurrentString(SPACE);
    addToCurrentString(batteryvoltage);
    addToCurrentString(SPACE);
    addToCurrentString(preassure);
    addToCurrentString(SPACE);
    addToCurrentString(temperature);
    addToCurrentString(SPACE);
    addToCurrentString(humidity);
    addToCurrentString(SPACE);
    addToCurrentString(altitude);
    addToCurrentString(SPACE);
    addToCurrentString(mag_time);
    addToCurrentString(SPACE);
    addToCurrentString(mag_nt);
    addToCurrentString(SPACE);
    addToCurrentString(mag_sq);
    
    
    strLength = strlen(this->currentString.c_str());
    
    //calculate checksum
    for(int i = 0; i < strLength; i++){
        if(this->currentString[i] != '>'){
            
            chkSum = update_crc_16(chkSum, this->currentString[i]);
            
        }         
    }
    
    sprintf(checkSum, "%04X", chkSum);
      
    //append rest of string
    addToCurrentString(SPACE);
    addToCurrentString(checkSum);
    addToCurrentString(LINESTOP); 
};

//! UpdateHeaderString, header / RMRK update method
/*! 

    Updates header / RMRK string

    \param identifier_id: string containing the identifier id of the unit.
    \param group_id: string containing the group id of the unit. 
    \param ZZZ: string containing "ZZZ" :)
    \param encoding: string containing encoding as described in the *.sps documentation
    \param source_id: string containing the source id of the unit eg. "0000"
    \param interpreter_id: string containing the interpreter id of the unit eg. "00"
    \param data_line_version: string containing data line version as described in the *.sps documentation, eg. "01" 
    \param source_firmware_version: string containing the firmware version of the unit eg. "1.2"
    \param barcode: string containing the barcode of the unit, eg. "12345" 
    \param interpreter_firmware_version: string containing interpreter firmware version
              
*/
void SPS::UpdateHeaderString(string barcode, string identifier_id, string group_id, string ZZZ, string encoding,string source_id, string interpreter_id, string data_line_version, string source_firmware_version, string interpreter_firmware_version){

    this->headerString = "";
    this->headerString.resize(256);

    int strLength = 0;
    char checkSum[5];
    memset(checkSum,'\0',5);
    unsigned short chkSum = 0;
    
    addToCurrentHeaderString(LINESTART);
    addToCurrentHeaderString("RMRK");
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString(identifier_id);
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString(group_id);
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString("0000/00/00");
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString("00:00:00.000");
    addToCurrentHeaderString(SPACE); 
    addToCurrentHeaderString("ZZZ");
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString(HEADEREND);
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString(source_id);
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString(interpreter_id);
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString("01");
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString(encoding);
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString(source_firmware_version);
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString(interpreter_firmware_version);
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString(HEADEREND);
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString("Barcode");
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString("Latitude");
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString("Longitude");
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString("gpsFixFlag");   
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString("batteryVoltage");
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString("preassure");
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString("temperature");
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString("humidity");
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString("altitude");
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString("mag_time");
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString("mag_value");
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString("mag_quality");
    
    
    strLength = strlen(this->headerString.c_str());
    
    //calculate checksum
    for(int i = 0; i < strLength; i++){
        if(this->headerString[i] != '>'){
            
            chkSum = update_crc_16(chkSum, this->headerString[i]);
            
        }         
    }
    
    sprintf(checkSum, "%04X", chkSum);
      
    //append rest of string
    addToCurrentHeaderString(SPACE);
    addToCurrentHeaderString(checkSum);
    addToCurrentHeaderString(LINESTOP);             
}

//! UpdateCurrentErrString, sps ERR string creation method, taking data and info as parameters.
/*! 

    Generates the current sps ERR string containing everything needed in a BMAG sps string including crc, brackets etc.

    \param tag: string containing the tag for the sps logging unit eg. "BMAG"
    \param identifier_id: string containing the identifier id of the unit.
    \param group_id: string containing the group id of the unit.
    \param date: string containing the date in the format "YYYY/MM/DD"  
    \param time: string containing the current time in format "HH:MM:SS.FFF"
    \param ZZZ: string containing "ZZZ" :)
    \param encoding: string containing encoding as described in the *.sps documentation
    \param source_id: string containing the source id of the unit eg. "0000"
    \param interpreter_id: string containing the interpreter id of the unit eg. "00"
    \param data_line_version: string containing data line version as described in the *.sps documentation, eg. "01" 
    \param source_firmware_version: string containing the firmware version of the unit eg. "1.2"
    \param barcode: string containing the barcode of the unit, eg. "12345" 
    \param latitude: string containing the latitude of the unit in decimal degrees. For further info read the *.sps documentation.
    \param longitude: string containing the longitude of the unit in decimal degrees. For further info read the *.sps documentation.
    \param gpsFixFlag: bool containing the gps fix status, 1 if gps fix is present, else 0.
    \param mag_time: string containing the last timestamp received from GSM-19 device eg. "000048.0"
    \param mag_nt: string containing the last mag measurement in nT eg. "036418.77"
    \param mag_sq: string containing info about the measurement quality of the last mag measurement eg. "99"
              
*/ 
void SPS::UpdateCurrentErrString(string tag, string identifier_id, string group_id, string date, string time, string ZZZ,string encoding, string source_id, string interpreter_id, string data_line_version, string source_firmware_version, string interpreter_firmware_version, string latitude, string longitude, bool gpsFixFlag, string mag_time, string mag_nt, string mag_sq, Serial * dbg){

    this->currentErrString = "";
    this->currentErrString.resize(256);
    char checkSum[5];
    memset(checkSum,'\0',5);
    unsigned short chkSum = 0;
    int strLength = 0;
    
    addToCurrentErrString(LINESTART);
    addToCurrentErrString(tag);
    addToCurrentErrString(SPACE);
    addToCurrentErrString(identifier_id);
    addToCurrentErrString(SPACE);
    addToCurrentErrString(group_id);
    addToCurrentErrString(SPACE);
    addToCurrentErrString(date);
    addToCurrentErrString(SPACE);
    addToCurrentErrString(time);
    addToCurrentErrString(SPACE);
    addToCurrentErrString("ZZZ");
    addToCurrentErrString(SPACE);
    addToCurrentErrString(HEADEREND);
    addToCurrentErrString(SPACE);
    addToCurrentErrString(source_id);
    addToCurrentErrString(SPACE);
    addToCurrentErrString(interpreter_id);
    addToCurrentErrString(SPACE);
    addToCurrentErrString("02");
    addToCurrentErrString(SPACE);
    addToCurrentErrString(encoding);
    addToCurrentErrString(SPACE);
    addToCurrentErrString(source_firmware_version);
    addToCurrentErrString(SPACE);
    addToCurrentErrString(interpreter_firmware_version);
    addToCurrentErrString(SPACE);
    addToCurrentErrString(HEADEREND);
    addToCurrentErrString(SPACE);
    addToCurrentErrString("BMAG");
    if(!gpsFixFlag){
        addToCurrentErrString(SPACE);
        addToCurrentErrString("RTC");
    }
    addToCurrentErrString(SPACE);
    addToCurrentErrString("C02");
    addToCurrentErrString(SPACE);
    addToCurrentErrString("C03");
    
    strLength = strlen(this->currentErrString.c_str());
    
    //calculate checksum
    for(int i = 0; i < strLength; i++){
        if(this->currentErrString[i] != '>'){
            
            chkSum = update_crc_16(chkSum, this->currentErrString[i]);
            
        }         
    }
        
    sprintf(checkSum, "%04X", chkSum);
    
    //append rest of string
    addToCurrentErrString(SPACE);
    addToCurrentErrString(checkSum);
    addToCurrentErrString(LINESTOP);   
};

//! init_crc16_tab, creates a crc tab.
/*! 
    Populates a crc array for later use           
*/ 
void SPS::init_crc16_tab()
{
    int i, j;
    unsigned short crc, c;

    for (i = 0; i < 256; i++)
    {
        crc = 0;
        c = (unsigned short) i;

        for (j = 0; j < 8; j++)
        {
            if ((crc ^ c) & 0x0001) 
                crc = (crc >> 1) ^ P_16;
            else
                crc = crc >> 1;

            c = c >> 1;
        }
        crc_tab16[i] = crc;
    }
    crc_tab16_init = 1;
};



//! update_crc_16, updates crc tab.
/*! 
    \param crc: the crc currently getting calculated  
    \param c: next character to be part of crc
    \return returns crc value           
*/ 
unsigned short SPS::update_crc_16(unsigned short crc, char c)
{
    unsigned short tmp, short_c;

    short_c = 0x00ff & (unsigned short) c;

    if (!crc_tab16_init)
        init_crc16_tab();
    

    tmp = crc ^ short_c;
    crc = (crc >> 8) ^ crc_tab16[tmp & 0xff];

    return crc;
};



//! getCurrentString, returns the current sps data string
/*! 
    \return returns current sps data string          
*/ 
string SPS::getCurrentString(void){
    return this->currentString; 
};


//! getErrStatus, returns the current error status flag
/*! 
    \return returns the current error status flag (bool)         
*/ 
bool SPS::getErrStatus(void){
    return this->errStatus;    
}; 
  
//! setErrStatus, sets error status flag
/*! 
    \param status: boolean value assigned to errorStatus        
*/ 
void SPS::setErrStatus(bool status){
    this->errStatus = status;
};

//! getHeaderString, getter method, returning header string for *.sps file
/*! 
    \return returns the header string for the *.sps file       
*/ 
string SPS::getHeaderString(void){
    return this->headerString;    
};

//add data to current string
//! addToCurrentString, appends string of data to the sps string currently being manipulated.
/*! 
    \param data: string containing data   
*/ 
void SPS::addToCurrentString(string data){
    int currentStartIndex = strlen(this->currentString.c_str());
    char dataStrLen = 0;
    int index = 0;
    
    dataStrLen = strlen(data.c_str());
    
    for(int i = 0; i < dataStrLen; i++){
        
        if((this->currentString[currentStartIndex+i] != '\r') || (this->currentString[currentStartIndex+i] != '\n')){
            
            this->currentString[currentStartIndex+i] = data[index];
            index += 1;
                  
        }
                     
    }                
}

//! addToCurrentString, appends char to the sps string currently being manipulated.
/*! 
    \param data: data character 
*/ 
void SPS::addToCurrentString(char data){
    int currentStartIndex = strlen(this->currentString.c_str());
    this->currentString[currentStartIndex] = data;          
};



//! addToCurrentHeaderString, appends string to the current header string
/*! 
    \param data: data string
*/ 
void SPS::addToCurrentHeaderString(string data){
    int currentStartIndex = strlen(this->headerString.c_str());
    char dataStrLen = 0;
    int index = 0;
    
    dataStrLen = strlen(data.c_str());
    
    for(int i = 0; i < dataStrLen; i++){
        
        if((this->headerString[currentStartIndex+i] != '\r') || (this->headerString[currentStartIndex+i] != '\n')){
            
            this->headerString[currentStartIndex+i] = data[index];
            index += 1;
                  
        }
                     
    }                
};


//! addToCurrentHeaderString, appends char to the current header string
/*! 
    \param data: data char
*/ 
void SPS::addToCurrentHeaderString(char data){
    int currentStartIndex = strlen(this->headerString.c_str());
    this->headerString[currentStartIndex] = data;          
};

//! addToCurrentErrString, appends string to the sps err_string currently being manipulated.
/*! 
    \param data: data string 
*/ 
void SPS::addToCurrentErrString(string data){
    int currentStartIndex = strlen(this->currentErrString.c_str());
    char dataStrLen = 0;
    dataStrLen = strlen(data.c_str());
    
    for(int i = 0; i < dataStrLen; i++){
        this->currentErrString[currentStartIndex+i] = data[i];               
    }      
    
};

//! addToCurrentErrString, appends a character to the sps err_string currently being manipulated.
/*! 
    \param data: data char 
*/ 
void SPS::addToCurrentErrString(char data){
    int currentStartIndex = strlen(currentErrString.c_str());
    currentErrString[currentStartIndex] = data;      
        
};

//! generateSpsFilename, generates a sps file name, including file postfix, using the current date as input.
/*! 
    \param formattedDate: Date string using a formatted date with format: "YYYY/MM/DD"
    \param barcode: Char array with barcode serial number in ASCII
    \param time: String containing a timestamp formatted "HH:MM:SS.FFF" 
*/ 
void SPS::generateSpsFilename(string formattedDate, string barcode, string time){
    
    fileNameString.resize(40);
    
    fileNameString.replace(0, 5, "/usb/");   

    fileNameString[5] = formattedDate[0];
    fileNameString[6] = formattedDate[1];
    fileNameString[7] = formattedDate[2];
    fileNameString[8] = formattedDate[3];
    fileNameString[9] = formattedDate[5];
    fileNameString[10] = formattedDate[6];
    fileNameString[11] = formattedDate[8];
    fileNameString[12] = formattedDate[9];
    fileNameString[13] = '_';
    fileNameString[14] = time[0];
    fileNameString[15] = time[1];
    fileNameString[16] = time[3];
    fileNameString[17] = time[4];
    fileNameString[18] = time[6];
    fileNameString[19] = time[7]; 
    fileNameString[20] = '_';
    fileNameString[21] = 'B';
    fileNameString[22] = 'M';
    fileNameString[23] = 'A';
    fileNameString[24] = 'G';
    fileNameString[25] = '_';
    fileNameString[26] = barcode[0];
    fileNameString[27] = barcode[1];
    fileNameString[28] = barcode[2];
    fileNameString[29] = barcode[3];
    fileNameString[30] = barcode[4];        
    fileNameString[31] = '.';
    fileNameString[32] = 's';
    fileNameString[33] = 'p';
    fileNameString[34] = 's';   

};

//! getSpsFileName, getter method, returning a *.sps filename based on the formatted date generated using generateSpsFilename(string formattedDate) method.
/*! 
    \return fileNameString returns file name including postfix. 
*/ 
string SPS::getSpsFileName(void){
    
    return this->fileNameString;   
    
};

//! getCurrentErrString, getter method returning current sps error string
/*! 
    \return currentErrString: returns the current sps error string.
*/ 
string SPS::getCurrentErrString(void){
    return this->currentErrString;    
};