#include "rtc.h"
#include "menu.h"
#include "serial_lcd.h"
#include "keypad.h"

HAL_StatusTypeDef HAL_status;
HAL_LockTypeDef HAL_lock;
RTC_HandleTypeDef rtch;
HAL_RTCStateTypeDef rtc_state;
RTC_InitTypeDef rtc_init;
RTC_TimeTypeDef rtc_time;
RTC_DateTypeDef rtc_date;
RTC_AlarmTypeDef rtc_alarm;
volatile int tmp_date, tmp_sec;
volatile int AlarmA_Enabled = 0;
volatile int AlarmA_triggered = 0;
extern FSM_State Main_FSM_States;

const char BCD2HEX[16] = {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};
const char DOW[8][3] = { {' ',' ',' '}, {'H',' ',' '}, {'K',' ',' '}, {'S','z','e'}, {'C','s',' '}, {'P',' ',' '}, {'S','z','o'}, {'V',' ',' '} };
const char Yes_No[16] = {'*',':','O','K',' ',' ',' ',' ',' ','#',':','M',0x82,'g','s','e'};//"*:OK     #:Mégse"};

void RTC_AlarmAEventCallback(void) {
    AlarmA_triggered = 1;
}

void RTC_AlarmIRQHandler(void) {
    if(RTC->ISR & RTC_ISR_ALRAF) {
        /* Get the status of the Interrupt */
        if ((RTC->CR & RTC_IT_ALRA) != 0) {
            /* AlarmA callback */ 
            RTC_AlarmAEventCallback();

            /* Clear the Alarm interrupt pending bit */
            RTC->ISR &= ~(RTC_FLAG_ALRAF);
        }
    }

    /* Clear the EXTI's line Flag for RTC Alarm */
    EXTI->PR = RTC_EXTI_LINE_ALARM_EVENT;
    __HAL_RTC_CLEAR_FLAG(RTC_EXTI_LINE_ALARM_EVENT);

    /* Change RTC state */
    rtch.State = HAL_RTC_STATE_READY; 
}

void ShowAlarm(void) {
    char tt[16];

    write_ser_lcd(0x01, false);     // Clear display
    wait_us(1100);
    write_ser_lcd(0x06, false);     // Entry mode set
    wait_us(30);
    write_ser_lcd(0x80, false);     // set DDRAM addr to 0x00, beginning of 1st line
    tt[0] = BCD2HEX[((rtc_alarm.AlarmTime.Hours & 0xF0) >> 4)];
    tt[1] = BCD2HEX[(rtc_alarm.AlarmTime.Hours & 0x0F)];
    tt[2] = ':';
    tt[3] = BCD2HEX[((rtc_alarm.AlarmTime.Minutes & 0xF0) >> 4)];
    tt[4] = BCD2HEX[(rtc_alarm.AlarmTime.Minutes & 0x0F)];
    tt[5] = ':';
    tt[6] = BCD2HEX[((rtc_alarm.AlarmTime.Seconds & 0xF0) >> 4)];
    tt[7] = BCD2HEX[(rtc_alarm.AlarmTime.Seconds & 0x0F)];
    tt[8] = ' ';
    tt[9] = DOW[rtc_alarm.AlarmDateWeekDay][0];
    tt[10] = DOW[rtc_alarm.AlarmDateWeekDay][1];
    tt[11] = DOW[rtc_alarm.AlarmDateWeekDay][2];
    write_ser_text(tt, 12);
    write_ser_lcd(0xC0, false);
    wait_us(30);
    write_ser_text(Yes_No, 16);
}

void ShowTime(void) {
    char tt[16];

    write_ser_lcd(0x80, false);     // set DDRAM addr to 0x00, beginning of 1st line
    tt[0] = BCD2HEX[((rtc_time.Hours & 0xF0) >> 4)];
    tt[1] = BCD2HEX[(rtc_time.Hours & 0x0F)];
    tt[2] = ':';
    tt[3] = BCD2HEX[((rtc_time.Minutes & 0xF0) >> 4)];
    tt[4] = BCD2HEX[(rtc_time.Minutes & 0x0F)];
    tt[5] = ':';
    tt[6] = BCD2HEX[((rtc_time.Seconds & 0xF0) >> 4)];
    tt[7] = BCD2HEX[(rtc_time.Seconds & 0x0F)];
    tmp_sec = rtc_time.Seconds;
    tt[8] = ' ';
    if (AlarmA_Enabled) {
        tt[9] = 0x01;
    } else {
        tt[9] = ' ';
    }
    write_ser_text(tt, 10);
}

void ShowDate(void) {
    char tt[16];

    write_ser_lcd(0xC0, false);     // set DDRAM addr to 0x00, beginning of 1st line
    tt[0] = BCD2HEX[2];
    tt[1] = BCD2HEX[0];
    tt[2] = BCD2HEX[((rtc_date.Year & 0xF0) >> 4)];
    tt[3] = BCD2HEX[(rtc_date.Year & 0x0F)];
    tt[4] = '.';
    tt[5] = BCD2HEX[((rtc_date.Month & 0xF0) >> 4)];
    tt[6] = BCD2HEX[(rtc_date.Month & 0x0F)];
    tt[7] = '.';
    tt[8] = BCD2HEX[((rtc_date.Date & 0xF0) >> 4)];
    tt[9] = BCD2HEX[(rtc_date.Date & 0x0F)];
    tt[10] = '.';
    tmp_date = rtc_date.Date;
    tt[11] = ' ';
    tt[12] = DOW[rtc_date.WeekDay][0];
    tt[13] = DOW[rtc_date.WeekDay][1];
    tt[14] = DOW[rtc_date.WeekDay][2];
    write_ser_text(tt,15);
}

void ShowTimeforSet(char * tt) {
    write_ser_lcd(0x01, false);     // Clear display
    wait_us(1100);
    write_ser_lcd(0x06, false);     // Entry mode set
    wait_us(30);
    write_ser_lcd(0x0E, false);     // Display ON, Cursor ON, Blink OFF
    wait_us(30);
    write_ser_lcd(0x80, false);     // set DDRAM addr to 0x00, beginning of 1st line
    tt[0] = BCD2HEX[((rtc_time.Hours & 0xF0) >> 4)];
    tt[1] = BCD2HEX[(rtc_time.Hours & 0x0F)];
    tt[2] = ':';
    tt[3] = BCD2HEX[((rtc_time.Minutes & 0xF0) >> 4)];
    tt[4] = BCD2HEX[(rtc_time.Minutes & 0x0F)];
    tt[5] = ':';
    tt[6] = BCD2HEX[((rtc_time.Seconds & 0xF0) >> 4)];
    tt[7] = BCD2HEX[(rtc_time.Seconds & 0x0F)];
    write_ser_text(tt, 8);
    write_ser_lcd(0x80, false);     // set DDRAM addr to 0x00, beginning of 1st line
}

void ShowAlarmforSet(char * tt) {
    write_ser_lcd(0x01, false);     // Clear display
    wait_us(1100);
    write_ser_lcd(0x06, false);     // Entry mode set
    wait_us(30);
    write_ser_lcd(0x0E, false);     // Display ON, Cursor ON, Blink OFF
    wait_us(30);
    write_ser_lcd(0x80, false);     // set DDRAM addr to 0x00, beginning of 1st line
    tt[0] = BCD2HEX[((rtc_alarm.AlarmTime.Hours & 0xF0) >> 4)];
    tt[1] = BCD2HEX[(rtc_alarm.AlarmTime.Hours & 0x0F)];
    tt[2] = ':';
    tt[3] = BCD2HEX[((rtc_alarm.AlarmTime.Minutes & 0xF0) >> 4)];
    tt[4] = BCD2HEX[(rtc_alarm.AlarmTime.Minutes & 0x0F)];
    tt[5] = ':';
    tt[6] = BCD2HEX[((rtc_alarm.AlarmTime.Seconds & 0xF0) >> 4)];
    tt[7] = BCD2HEX[(rtc_alarm.AlarmTime.Seconds & 0x0F)];
    tt[8] = ' ';
    tt[9] = DOW[rtc_date.WeekDay][0];
    tt[10] = DOW[rtc_date.WeekDay][1];
    tt[11] = DOW[rtc_date.WeekDay][2];
    write_ser_text(tt, 8);
    write_ser_lcd(0x80, false);     // set DDRAM addr to 0x00, beginning of 1st line
}

void ShowDateforSet(char * tt) {
    write_ser_lcd(0x01, false);     // Clear display
    wait_us(1100);
    write_ser_lcd(0x06, false);     // Entry mode set
    wait_us(30);
    write_ser_lcd(0x0E, false);     // Display ON, Cursor ON, Blink OFF
    wait_us(30);
    write_ser_lcd(0x80, false);     // set DDRAM addr to 0x00, beginning of 1st line
    tt[0] = BCD2HEX[2];
    tt[1] = BCD2HEX[0];
    tt[2] = BCD2HEX[((rtc_date.Year & 0xF0) >> 4)];
    tt[3] = BCD2HEX[(rtc_date.Year & 0x0F)];
    tt[4] = '.';
    tt[5] = BCD2HEX[((rtc_date.Month & 0xF0) >> 4)];
    tt[6] = BCD2HEX[(rtc_date.Month & 0x0F)];
    tt[7] = '.';
    tt[8] = BCD2HEX[((rtc_date.Date & 0xF0) >> 4)];
    tt[9] = BCD2HEX[(rtc_date.Date & 0x0F)];
    tt[10] = '.';
    tt[11] = ' ';
    tt[12] = DOW[rtc_date.WeekDay][0];
    tt[13] = DOW[rtc_date.WeekDay][1];
    tt[14] = DOW[rtc_date.WeekDay][2];
    write_ser_text(tt,15);
    write_ser_lcd(0x82, false);     // set DDRAM addr to 0x00, beginning of 1st line
}

void InitRTC(void) {
    uint32_t dummy;

    rtc_init.AsynchPrediv = 0x7F;
    rtc_init.SynchPrediv = 0xFF;
    rtc_init.HourFormat = RTC_HOURFORMAT_24;
    rtc_init.OutPutType = RTC_OUTPUT_TYPE_OPENDRAIN;
    rtc_init.OutPutPolarity = RTC_OUTPUT_POLARITY_HIGH;
    rtc_init.OutPut = RTC_OUTPUT_DISABLE;

    rtc_date.Date = 27;
    rtc_date.Month = 05;
    rtc_date.WeekDay = 2;
    rtc_date.Year = 14;
    tmp_date = 27;

    rtc_time.DayLightSaving = RTC_DAYLIGHTSAVING_NONE;
    rtc_time.Hours = 00;
    rtc_time.Minutes = 25;
    rtc_time.Seconds = 00;
    rtc_time.StoreOperation =RTC_STOREOPERATION_SET;
    tmp_sec = 0;

    rtch.Instance = RTC;
    rtch.Init = rtc_init;
    rtch.Lock = HAL_lock;
    rtch.State = rtc_state;

    __PWR_CLK_ENABLE();
    HAL_PWR_EnableBkUpAccess();
    __HAL_RCC_LSE_CONFIG(RCC_LSE_ON);
    __HAL_RCC_RTC_CONFIG(RCC_RTCCLKSOURCE_LSE);
    dummy = RCC->BDCR;
    dummy |= RCC_BDCR_RTCEN;
    RCC->BDCR = dummy;
//    __HAL_RCC_RTC_ENABLE();
    EXTI->PR = RTC_EXTI_LINE_ALARM_EVENT;
    RTC->ISR &= ~(RTC_FLAG_ALRAF);
    HAL_RTC_Init(&rtch);
    dummy = (RTC->BKP15R & 0x00000007);
    AlarmA_Enabled = ((dummy & 0x00000004) == 1);
    if (dummy == 0x0) {
        HAL_RTC_SetDate(&rtch, &rtc_date, FORMAT_BIN);
        HAL_RTC_SetTime(&rtch, &rtc_time, FORMAT_BIN);
    }
}

void AlarmA_Enable(void) {
    int keypress;
    
    HAL_RTC_GetAlarm(&rtch, &rtc_alarm, RTC_ALARM_A, FORMAT_BCD);
    //if (rtc_alarm.AlarmDateWeekDay == 0) { rtc_alarm.AlarmDateWeekDay = 1;}     // There was no alarm set yet
    ShowAlarm();
    keypress = Poll_keypad_yesno();
    if (keypress == 0x0C) {
        HAL_RTC_DeactivateAlarm(&rtch, RTC_ALARM_A);
        HAL_RTC_SetAlarm_IT(&rtch, &rtc_alarm, FORMAT_BCD);
        NVIC_SetVector(RTC_Alarm_IRQn, (uint32_t)RTC_AlarmIRQHandler);
        HAL_NVIC_EnableIRQ(RTC_Alarm_IRQn);
        AlarmA_Enabled = 1;
        RTC->BKP15R |= 0x00000004;
    }
    Main_FSM_States = FINISH;
}

void AlarmA_Disable(void) {
    HAL_RTC_DeactivateAlarm(&rtch, RTC_ALARM_A);
    AlarmA_Enabled = 0;
    RTC->BKP15R &= ~(0x00000004);
    Main_FSM_States = FINISH;
}

void SetRTCTime(void) {
    int keypress;
    char TT[8];
    int index = 0;
    int exit = 0;

    HAL_RTC_GetTime(&rtch, &rtc_time, FORMAT_BCD);
    ShowTimeforSet(TT);

    while (exit == 0) {
        keypress = Poll_keypad_full();
        switch (keypress) {
            case '*':
                if (index > 0) {index--;}
                if ((index == 2) | (index == 5)) {index--;}
                break;
            case '#':
                if (index < 8) {index++;}
                if ((index == 2) | (index == 5)) {index++;}
                if (index == 8) {exit = 1;}
                break;
            default:
                TT[index] = keypress;
                if (index < 8) {index++;}
                if ((index == 2) | (index == 5)) {index++;}
                if (index == 8) {exit = 1;}
                break;
        }
        write_ser_lcd(0x80, false);
        wait_us(30);
        write_ser_text(TT,8);
        write_ser_lcd(0x80 + index, false);
    }
    rtc_time.Hours = ((TT[0] - 0x30) << 4) + (TT[1] - 0x30);
    rtc_time.Minutes = ((TT[3] - 0x30) << 4) + (TT[4] - 0x30);
    rtc_time.Seconds = ((TT[6] - 0x30) << 4) + (TT[7] - 0x30);
    HAL_RTC_SetTime(&rtch, &rtc_time, FORMAT_BCD);
    RTC->BKP15R |= 0x00000001;
    Main_FSM_States = FINISH;
}

void SetRTCAlarm(void) {
    int keypress;
    char TT[12];
    int index = 0;
    int exit = 0;
    int DoW;

    HAL_RTC_GetAlarm(&rtch, &rtc_alarm, RTC_ALARM_A, FORMAT_BCD);
    DoW = rtc_alarm.AlarmDateWeekDay;
    rtc_alarm.AlarmDateWeekDaySel = RTC_ALARMDATEWEEKDAYSEL_WEEKDAY;    // We only used weekday alarm
    ShowAlarmforSet(TT);

    while (exit == 0) {
        keypress = Poll_keypad_full();
        switch (keypress) {
            case '*':
                if (index > 0) {index--;}
                if ((index == 2) | (index == 5)) {index--;}
                if (index == 8) {index--;}
                break;
            case '#':
                if (index < 10) {index++;}
                if ((index == 2) | (index == 5)) {index++;}
                if (index == 8) {index++;}
                if (index == 10) {exit = 1;}
                break;
            default:
                if (index < 8) {TT[index] = keypress;} else {
                    DoW = (keypress - 0x30);
                    TT[9] = DOW[DoW][0];
                    TT[10] = DOW[DoW][1];
                    TT[11] = DOW[DoW][2];
                }
                if (index < 10) {index++;}
                if ((index == 2) | (index == 5)) {index++;}
                if (index == 8) {index++;}
                if (index == 10) {exit = 1;}
                break;
        }
        write_ser_lcd(0x80, false);
        wait_us(30);
        write_ser_text(TT,12);
        write_ser_lcd(0x80 + index, false);
    }
    rtc_alarm.AlarmTime.Hours = ((TT[0] - 0x30) << 4) + (TT[1] - 0x30);
    rtc_alarm.AlarmTime.Minutes = ((TT[3] - 0x30) << 4) + (TT[4] - 0x30);
    rtc_alarm.AlarmTime.Seconds = ((TT[6] - 0x30) << 4) + (TT[7] - 0x30);
    rtc_alarm.AlarmDateWeekDay = DoW;
    if (DoW == 0x00) {
        rtc_alarm.AlarmMask = RTC_ALARMMASK_DATEWEEKDAY;        // Alarm is triggered every day, no date or weekday match
    } else {
        rtc_alarm.AlarmMask = RTC_ALARMMASK_NONE;               // Alarm is triggered when everything match
    }
    HAL_RTC_DeactivateAlarm(&rtch, RTC_ALARM_A);
    HAL_RTC_SetAlarm_IT(&rtch, &rtc_alarm, FORMAT_BCD);
    NVIC_SetVector(RTC_Alarm_IRQn, (uint32_t)RTC_AlarmIRQHandler);
    HAL_NVIC_EnableIRQ(RTC_Alarm_IRQn);
    RTC->BKP15R |= 0x00000006;
    AlarmA_Enabled = 1;
    Main_FSM_States = FINISH;
}

void SetRTCDate(void) {
    int keypress;
    char TT[15];
    int index = 2;
    int exit = 0;
    int DoW;

    HAL_RTC_GetDate(&rtch, &rtc_date, FORMAT_BCD);
    DoW = rtc_date.WeekDay;
    ShowDateforSet(TT);

    while (exit == 0) {
        keypress = Poll_keypad_full();
        switch (keypress) {
            case '*':
                if (index > 2) {index--;}
                if ((index == 4) | (index == 7)) {index--;}
                if (index == 11) {index -= 2;}
                break;
            case '#':
                if (index < 13) {index++;}
                if ((index == 4) | (index == 7)) {index++;}
                if (index == 10) {index += 2;}
                if (index == 13) {exit = 1;}
                break;
            default:
                if (index < 10) {TT[index] = keypress;} else {
                    DoW = (keypress - 0x30);
                    TT[12] = DOW[DoW][0];
                    TT[13] = DOW[DoW][1];
                    TT[14] = DOW[DoW][2];
                }
                if (index < 13) {index++;}
                if ((index == 4) | (index == 7)) {index++;}
                if (index == 10) {index += 2;}
                if (index == 13) {exit = 1;}
                break;
        }
        write_ser_lcd(0x80, false);
        wait_us(30);
        write_ser_text(TT,15);
        write_ser_lcd(0x80 + index, false);
    }
    rtc_date.Year = ((TT[2] - 0x30) << 4) + (TT[3] - 0x30);
    rtc_date.Month = ((TT[5] - 0x30) << 4) + (TT[6] - 0x30);
    rtc_date.Date = ((TT[8] - 0x30) << 4) + (TT[9] - 0x30);
    rtc_date.WeekDay = DoW;
    HAL_RTC_SetDate(&rtch, &rtc_date, FORMAT_BCD);
    RTC->BKP15R |= 0x00000001;
    Main_FSM_States = FINISH;
}
