// Includes and declarations for Pelion DM Client to work
#include "simple-mbed-cloud-client.h"
#include "BlockDevice.h"
#include "LittleFileSystem.h"
#include "NetworkInterface.h"
#include "Nanostack.h"
#include "ns_file_system.h"

// Thunderboard Sense 2 connects over 802.15.4 by default. Since the mesh stack is a bit iffy.
// We'll register a 'network down' handler to act as a kind of watchdog for the Pelion DM Client.
NetworkInterface *net = NetworkInterface::get_default_instance();

// Thunderboard Sense 2 has a 1M external flash, which is being shared between upgrade storage and LittleFS.
// LittleFS is instantiated at the start of storage, until the start address for upgrade storage.
// Currently, this split is at 256/768 for FS/upgrade.
BlockDevice* bd = BlockDevice::get_default_instance();
SlicingBlockDevice sd(bd, 0, MBED_CONF_UPDATE_CLIENT_STORAGE_ADDRESS);

#if COMPONENT_SD || COMPONENT_NUSD
// Use FATFileSystem for SD card type blockdevices
FATFileSystem fs("fs");
#else
// Use LittleFileSystem for non-SD block devices to enable wear leveling and other functions
LittleFileSystem fs("fs");
#endif

// Default User button for GET example and for resetting the storage
InterruptIn button(BTN0);

// How often to fetch sensor data (in seconds)
#define SENSORS_POLL_INTERVAL 3.0

// Send all sensor data or just limited (useful for when running out of memory)
#define SEND_ALL_SENSORS

// Sensors related includes and initialization
#include "BMP280.h"
#include "Si1133.h"
#include "SI7021.h"
#include "Si7210.h"
#include "AMS_CCS811.h"
#include "ICM20648.h"

/* Turn on power supply to ENV sensor suite */
DigitalOut env_en(PF9, 1);
/* Turn on power to CCS811 sensor */
DigitalOut ccs_en(PF14, 1);
/* Turn on power to hall effect sensor */
DigitalOut hall_en(PB10, 1);
/* Turn on power to IMU */
DigitalOut imu_en(PF8, 1);
 
I2C env_i2c(PC4, PC5);
BMP280 sens_press_temp(env_i2c);
Si1133 sens_light(PC4, PC5);
SI7021 sens_hum_temp(PC4, PC5, SI7021::SI7021_ADDRESS, 400000);
I2C hall_i2c(PB8, PB9);
silabs::Si7210 sens_hall(&hall_i2c);
InterruptIn ccs_int(PF13);
I2C ccs_i2c(PB6, PB7);
AMS_CCS811 sens_aqs(&ccs_i2c, PF15);
ICM20648 sens_imu(PC0, PC1, PC2, PC3, PF12);

// Declaring pointers for access to Pelion Client resources outside of main()
MbedCloudClientResource *res_button;
MbedCloudClientResource *res_led;

// Additional resources for sensor readings
#ifdef SEND_ALL_SENSORS
MbedCloudClientResource *res_light;
MbedCloudClientResource *res_pressure;
MbedCloudClientResource *res_temperature1;
MbedCloudClientResource *res_humidity;
MbedCloudClientResource *res_temperature2;
MbedCloudClientResource *res_co2;
MbedCloudClientResource *res_tvoc;
MbedCloudClientResource *res_field;
MbedCloudClientResource *res_temperature3;
MbedCloudClientResource *res_accelerometer_x;
MbedCloudClientResource *res_accelerometer_y;
MbedCloudClientResource *res_accelerometer_z;
MbedCloudClientResource *res_gyroscope_x;
MbedCloudClientResource *res_gyroscope_y;
MbedCloudClientResource *res_gyroscope_z;
MbedCloudClientResource *res_temperature4;
#endif /* SEND_ALL_SENSORS */

// An event queue is a very useful structure to debounce information between contexts (e.g. ISR and normal threads)
// This is great because things such as network operations are illegal in ISR, so updating a resource in a button's fall() function is not allowed.
EventQueue eventQueue;

// When the device is registered, this variable will be used to access various useful information, like device ID etc.
static const ConnectorClientEndpointInfo* endpointInfo;

/**
 * POST handler - prints the content of the payload
 * @param resource The resource that triggered the callback
 * @param buffer If a body was passed to the POST function, this contains the data.
 *               Note that the buffer is deallocated after leaving this function, so copy it if you need it longer.
 * @param size Size of the body
 */
void blink_callback(MbedCloudClientResource *resource, const uint8_t *buffer, uint16_t size) {
    static int num_count = 0;
    static int event = 0;

    static DigitalOut led_ch_red(PD11, 1);
    static DigitalOut led_ch_green(PD12, 1);
    static DigitalOut led_ch_blue(PD13, 1);

    static DigitalOut led_com_0(PI0, 0);
    static DigitalOut led_com_1(PI1, 0);
    static DigitalOut led_com_2(PI2, 0);
    static DigitalOut led_com_3(PI3, 0);

    static DigitalOut led_rgb_en(PJ14, 0);

    if (buffer != NULL) {
        printf("POST received. POST data: %s\n", buffer);

        if (event > 0) {
            printf("Not blinking since previous blink still in progress\n");
            return;
        }
        num_count = 0;
        for (size_t i = 0, num_arg = 0; i < 20 || buffer[i] == 0; i++) {
            if (buffer[i] == ':') {
                num_arg++;
                continue;
            }

            if (buffer[i] >= '0' && buffer[i] <= '9') {
                switch (num_arg) {
                    case 0:
                        if (buffer[i] == '1') {
                            led_ch_red = 1;
                        } else {
                            led_ch_red = 0;
                        }
                        break;
                    case 1:
                        if (buffer[i] == '1') {
                            led_ch_green = 1;
                        } else {
                            led_ch_green = 0;
                        }
                        break;
                    case 2:
                        if (buffer[i] == '1') {
                            led_ch_blue = 1;
                        } else {
                            led_ch_blue = 0;
                        }
                        break;
                    case 3:
                        num_count = ((buffer[i] - 0x30) * 2) - 1;
                        printf("blinking %d\n", num_count);
                        break;
                    default:
                        break;
                }
            } else {
                //garbage...
                continue;
            }
            if (num_count > 0) {
                break;
            }
        }

        if (num_count > 0) {
            led_rgb_en = 1;
            led_com_0 = 1;
            led_com_1 = 1;
            event = eventQueue.call_in(1000, blink_callback, resource, (const uint8_t *)NULL, 0);
            if (event == 0) {
                led_rgb_en = 0;
                num_count = 0;
            }
        }
    } else {
        num_count--;
        led_com_0 = (num_count & 1);
        led_com_1 = (num_count & 1);

        if (num_count == 0) {
            led_rgb_en = 0;
            event = 0;
        } else {
            event = eventQueue.call_in(1000, blink_callback, resource, (const uint8_t *)NULL, 0);
            if (event == 0) {
                led_rgb_en = 0;
                num_count = 0;
            }
        }
    }
}

/**
 * Button function triggered by the physical button press.
 */
void button_press() {
    int v = res_button->get_value_int() + 1;
    res_button->set_value(v);
    printf("*** Button clicked %d times                                 \n", v);
}

/**
 * Notification callback handler
 * @param resource The resource that triggered the callback
 * @param status The delivery status of the notification
 */
void button_callback(MbedCloudClientResource *resource, const NoticationDeliveryStatus status) {
    printf("*** Button notification, status %s (%d)                     \n", MbedCloudClientResource::delivery_status_to_string(status), status);
}

/**
 * Registration callback handler
 * @param endpoint Information about the registered endpoint such as the name (so you can find it back in portal)
 */
void registered(const ConnectorClientEndpointInfo *endpoint) {
    printf("Registered to Pelion Device Management. Endpoint Name: %s\n", endpoint->internal_endpoint_name.c_str());
    endpointInfo = endpoint;
}

/**
 * Initialize sensors
 */
void sensors_init() {
    sens_press_temp.initialize();

    SI7021::SI7021_status_t result = sens_hum_temp.SI7021_SoftReset();
    if (result == SI7021::SI7021_SUCCESS) {
        wait_ms(15);
        SI7021::SI7021_vector_data_t result_data;
        result = sens_hum_temp.SI7021_Conf(SI7021::SI7021_RESOLUTION_RH_11_TEMP_11,
                                        SI7021::SI7021_HTRE_DISABLED);
        result = sens_hum_temp.SI7021_GetElectronicSerialNumber(&result_data);
        result = sens_hum_temp.SI7021_GetFirmwareRevision(&result_data);
        printf("Si7021 Electronic Serial Number: %16x %16x, firmware rev %02x\n",
                result_data.ElectronicSerialNumber_MSB,
                result_data.ElectronicSerialNumber_LSB,
                result_data.FirmwareRevision);
    }

    if (!sens_light.open()) {
        printf("ERROR: Failed to initialize sensor Si1133\n");
    }

    if (!sens_aqs.init()) {
        printf("ERROR: Failed to initialize sensor CCS811\n");
    } else {
        if (!sens_aqs.mode(AMS_CCS811::SIXTY_SECOND)) {
            printf("ERROR: Failed to set mode for sensor CCS811\n");
        }
//        sens_aqs.enable_interupt(true);
    }
 
    if (!sens_imu.open()) {
        printf("ERROR: Failed to initialize sensor ICM20648\n");
    }
}

/**
 * Update sensors and report their values.
 * This function is called periodically.
 */
void sensors_update() {
    SI7021::SI7021_status_t sens_hum_temp_reading, humidity_reading, temp2_reading;
    SI7021::SI7021_vector_data_t humidity_data, temp2_data;

    // BMP280 pressure and temperature (1)
    float pressure_value = sens_press_temp.getPressure();
    float temp1_value = sens_press_temp.getTemperature();

    // Si7021 humidity and temperature (2)
    sens_hum_temp_reading = sens_hum_temp.SI7021_TriggerHumidity(SI7021::SI7021_NO_HOLD_MASTER_MODE);
    if (sens_hum_temp_reading == SI7021::SI7021_SUCCESS) {
        wait_ms(30);
        humidity_reading = sens_hum_temp.SI7021_ReadHumidity(&humidity_data);
        temp2_reading = sens_hum_temp.SI7021_ReadTemperatureFromRH(&temp2_data);
        sens_aqs.env_data(humidity_data.RelativeHumidity, temp2_data.Temperature);
    }

    // Si1133 light and UV index
    float light_value, uv_index_value;
    bool light_reading = sens_light.get_light_and_uv(&light_value, &uv_index_value);

    // CCS811 air quality CO2 and TVoC
    sens_aqs.has_new_data();
    int co2_value = sens_aqs.co2_read();
    int tvoc_value = sens_aqs.tvoc_read();

    // Si7210 field and temperature (3)
    sens_hall.measureOnce();
    float field_value = (float)sens_hall.getFieldStrength() / 1000.0, temp3_value = (float)sens_hall.getTemperature() / 1000.0;
 
    // ICM20648 accelerometer and gyroscope
    float acc_x, acc_y, acc_z, gyr_x, gyr_y, gyr_z, temp4_value;
    sens_imu.get_accelerometer(&acc_x, &acc_y, &acc_z);
    sens_imu.get_gyroscope(&gyr_x, &gyr_y, &gyr_z);
    sens_imu.get_temperature(&temp4_value);

    printf("                                                                 \n");
    printf("BMP280 temp:   %8.3f C,   pressure: %8.3f [mbar]            \n", temp1_value, pressure_value);
    printf("Si7021 temp:   %8.3f C,   humidity: %8.3f %%                \n", temp2_data.Temperature, humidity_data.RelativeHumidity);
    printf("Si7210 temp:   %8.3f C,   field:    %8.3f [mT]              \n", temp3_value, field_value);
    printf("Si1133 light:  %8.3f lux, UV level: %8.3f                   \n", light_value, uv_index_value);
    printf("CCS811 CO2:    %8d ppm, VoC:      %8d ppb                   \n", co2_value, tvoc_value);
    printf("ICM20648 acc:  %8.3f x, %7.3f y,   %7.3f z [mg]            \n", acc_x, acc_y, acc_z);
    printf("ICM20648 gyro: %8.3f x, %7.3f y,   %7.3f z [mdps]          \n", gyr_x, gyr_y, gyr_z);

    printf("\r\033[8A");

    if (endpointInfo) {
#ifdef SEND_ALL_SENSORS
        res_pressure->set_value(pressure_value);
        res_temperature1->set_value(temp1_value);
 
        if (humidity_reading == SI7021::SI7021_SUCCESS) {
            res_humidity->set_value(humidity_data.RelativeHumidity);
        }
        if (temp2_reading == SI7021::SI7021_SUCCESS) {
            res_temperature2->set_value(temp2_data.Temperature);
        }

        res_field->set_value(field_value);
        res_temperature3->set_value(temp3_value);

        if (light_reading) {
            res_light->set_value(light_value);
        }

        res_co2->set_value(co2_value);
        res_tvoc->set_value(tvoc_value);

        res_accelerometer_x->set_value(acc_x);
        res_accelerometer_y->set_value(acc_y);
        res_accelerometer_z->set_value(acc_z);
        res_gyroscope_x->set_value(gyr_x);
        res_gyroscope_y->set_value(gyr_y);
        res_gyroscope_z->set_value(gyr_z);
#endif /* SEND_ALL_SENSORS */
    }
}

int main() {
    printf("\nStarting Simple Pelion Device Management Client example\n");

    int storage_status = fs.mount(&sd);
    if (storage_status != 0) {
        printf("Storage mounting failed.\n");
    }
    // If the User button is pressed ons start, then format storage.
    bool btn_pressed = (button.read() == MBED_CONF_APP_BUTTON_PRESSED_STATE);
    if (btn_pressed) {
        printf("User button is pushed on start...\n");
    }

    if (storage_status || btn_pressed) {
        printf("Formatting the storage...\n");
        int storage_status = StorageHelper::format(&fs, &sd);
        if (storage_status != 0) {
            printf("ERROR: Failed to reformat the storage (%d).\n", storage_status);
        }
    } else {
        printf("You can hold the user button during boot to format the storage and change the device identity.\n");
    }

    sensors_init();

    Nanostack::get_instance(); // ensure Nanostack is initialised
    ns_file_system_set_root_path("/fs/");

    // Connect to the internet (DHCP is expected to be on)
    printf("Connecting to the network using 802.15.4...\n");

    nsapi_error_t net_status = -1;
    for (int tries = 0; tries < 3; tries++) {
        net_status = net->connect();
        if (net_status == NSAPI_ERROR_OK) {
            break;
        } else {
            printf("Unable to connect to network. Retrying...\n");
        }
    }

    if (net_status != NSAPI_ERROR_OK) {
        printf("ERROR: Connecting to the network failed (%d)!\n", net_status);
        return -1;
    }

    printf("Connected to the network successfully. IP address: %s\n", net->get_ip_address());

    printf("Initializing Pelion Device Management Client...\n");

    /* Initialize Simple Pelion DM Client */
    SimpleMbedCloudClient client(net, &sd, &fs);
    int client_status = client.init();
    if (client_status != 0) {
        printf("ERROR: Pelion Client initialization failed (%d)\n", client_status);
        return -1;
    }

    /* Create resources */
    res_led = client.create_resource("3201/0/5853", "LED blinking (R:G:B:cnt)");
    res_led->observable(false);
    res_led->set_value("0:0:0:0");
    res_led->attach_post_callback(blink_callback);
    res_led->methods(M2MMethod::POST);

    res_button = client.create_resource("3200/0/5501", "button_count");
    res_button->set_value(0);
    res_button->methods(M2MMethod::GET);
    res_button->observable(true);
    res_button->attach_notification_callback(button_callback);

#ifdef SEND_ALL_SENSORS 
    // Sensor BMP280
    res_pressure = client.create_resource("3323/0/5853", "Barometric pressure (hPa)");
    res_pressure->set_value(0);
    res_pressure->observable(true);
    res_pressure->methods(M2MMethod::GET);
 
    res_temperature1 = client.create_resource("3303/0/5853", "Temperature BMP280 (C)");
    res_temperature1->set_value(0);
    res_temperature1->observable(true);
    res_temperature1->methods(M2MMethod::GET);
 
    // Sensor Si7021
    res_humidity = client.create_resource("3304/0/5853", "Humidity (%)");
    res_humidity->set_value(0);
    res_humidity->observable(true);
    res_humidity->methods(M2MMethod::GET);
 
    res_temperature2 = client.create_resource("3303/1/5853", "Temperature Si7021 (C)");
    res_temperature2->set_value(0);
    res_temperature2->observable(true);
    res_temperature2->methods(M2MMethod::GET);
 
     // Sensor Si7210
    res_field = client.create_resource("33257/0/5853", "Magnetic Field (mT)");
    res_field->set_value(0);
    res_field->observable(true);
    res_field->methods(M2MMethod::GET);
 
    res_temperature3 = client.create_resource("3303/2/5853", "Temperature Si7210 (C)");
    res_temperature3->set_value(0);
    res_temperature3->observable(true);
    res_temperature3->methods(M2MMethod::GET);

    // Sensor Si1133
    res_light = client.create_resource("3301/0/5853", "LightIntensity (LUX)");
    res_light->set_value(0);
    res_light->observable(true);
    res_light->methods(M2MMethod::GET);

    // Sensor CCS811
    res_co2 = client.create_resource("33255/0/5853", "CO2 (ppm)");
    res_co2->set_value(0);
    res_co2->observable(true);
    res_co2->methods(M2MMethod::GET);

    res_tvoc = client.create_resource("33256/0/5853", "VOC (ppm)");
    res_tvoc->set_value(0);
    res_tvoc->observable(true);
    res_tvoc->methods(M2MMethod::GET);
 
    // Sensor ICM20648
    res_accelerometer_x = client.create_resource("3313/0/5702", "Accelerometer X");
    res_accelerometer_x->set_value(0);
    res_accelerometer_x->methods(M2MMethod::GET);
    res_accelerometer_x->observable(true);
 
    res_accelerometer_y = client.create_resource("3313/0/5703", "Accelerometer Y");
    res_accelerometer_y->set_value(0);
    res_accelerometer_y->methods(M2MMethod::GET);
    res_accelerometer_y->observable(true);
 
    res_accelerometer_z = client.create_resource("3313/0/5704", "Accelerometer Z");
    res_accelerometer_z->set_value(0);
    res_accelerometer_z->methods(M2MMethod::GET);
    res_accelerometer_z->observable(true);
 
    res_gyroscope_x = client.create_resource("3334/0/5702", "Gyroscope X");
    res_gyroscope_x->set_value(0);
    res_gyroscope_x->methods(M2MMethod::GET);
    res_gyroscope_x->observable(true);
 
    res_gyroscope_y = client.create_resource("3334/0/5703", "Gyroscope Y");
    res_gyroscope_y->set_value(0);
    res_gyroscope_y->methods(M2MMethod::GET);
    res_gyroscope_y->observable(true);
 
    res_gyroscope_z = client.create_resource("3334/0/5704", "Gyroscope Z");
    res_gyroscope_z->set_value(0);
    res_gyroscope_z->methods(M2MMethod::GET);
    res_gyroscope_z->observable(true);
 
    res_temperature4 = client.create_resource("3303/3/5853", "Temperature ICM20648 (C)");
    res_temperature4->set_value(0);
    res_temperature4->observable(true);
    res_temperature4->methods(M2MMethod::GET);
#endif /* SEND_ALL_SENSORS */

    // Callback that fires when registering is complete
    client.on_registered(&registered);

    /* Register the device */
    client.register_and_connect();

    button.fall(eventQueue.event(&button_press));

    // The timer fires on an interrupt context, but debounces it to the eventqueue, so it's safe to do network operations
    Ticker timer;
    timer.attach(eventQueue.event(&sensors_update), SENSORS_POLL_INTERVAL);

    eventQueue.dispatch_forever();
}
