/***************************************************************************//**
 * @file asymmetricPart.h
 * @brief class for an asymmetric head object for gecko.h
 *******************************************************************************
 * @section License
 * <b>(C) Copyright 2015 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

#include "settings.h"
#include "LS013B7DH03.h"

/* Bitfield containing the position of the top left corner of the part and the orientation
 * The minimum number of bits to store x, y and orientation is 10 bits (there are 255 tiles).
 * Since this will be expanded to 16 bits the x and y values are stored in 7 bits each.
 * */

struct positionAndDirection{
	uint8_t x:7;
	uint8_t y:7;
	uint8_t direction:2;
};

class asymmetricPart{
private:
	positionAndDirection _posAndDir;
	const uint8_t *_px_map;
	uint8_t _nPix;

	/* Private member functions */
	void draw(silabs::LS013B7DH03 &display, uint8_t color) const;
public:
	asymmetricPart();
	asymmetricPart(uint8_t x, uint8_t y, Direction dir,  const uint8_t *px_map, uint8_t nPix);

	/* Set all member variables */
	void init(uint8_t x, uint8_t y, Direction dir,  const uint8_t *px_map, uint8_t nPix);

	/* Draw the part on the screen */
	void draw(silabs::LS013B7DH03 &display) const {draw(display, Black);};

	/* Erase the part from the screen */
	void remove(silabs::LS013B7DH03 &display) const {draw(display, White);};

	/* Get member variables */
	uint8_t getX() const {return _posAndDir.x;};
	uint8_t getY() const {return _posAndDir.y;};
	uint8_t getDir() const {return _posAndDir.direction;};

	/* Set member variables */
	void setX(uint8_t x) {_posAndDir.x = x&0x7F;};
	void setY(uint8_t y) {_posAndDir.y = y&0x7F;};
	void setDir(Direction dir) {_posAndDir.direction = static_cast<int>(dir)&0x0003;};
};

