/***************************************************************************//**
 * @file main.cpp
 * @brief Demo program for game Hungry Gecko
 *******************************************************************************
 * @section License
 * <b>(C) Copyright 2015 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

#include "LS013B7DH03.h"
#include "gecko.h"
#include "food.h"
#include "settings.h"

/**************************** Define I/O **************************************/

InterruptIn in(SW1);
InterruptIn inB1(SW0);
#define SCK     PE12
#define MOSI 	PE10

DigitalOut CS(PA10);
DigitalOut EXTCOM(PF3);
DigitalOut DISP(PA8);

SPI displaySPI(MOSI, NC, SCK);
silabs::LS013B7DH03 display(&displaySPI, &CS, &EXTCOM);

/**************************** Define Timers ***********************************/

LowPowerTicker ticker;

/**************************** Global variables ********************************/

/* Flag that is set to true when the display is refreshed */
volatile bool refreshed = false;

/* Flag that is set to true by the ticker. Makes the gecko move at regular time intervals */
volatile bool updateDisplay = true;

/* A flag that ensures the controller to only read one click per frame */
volatile bool PBenabled = true;

/* Direction in which the gecko moves */
Direction dir = UP;

uint8_t score = 0;

/**************************** Define callback handlers ************************/
void tickerCallback(void);

/* Push button handlers */
void in_handler_B0();
void in_handler_B1();

/* Define game modes */
typedef enum {
	PLAY, STOP
} Modes;

/* Set the game mode */
Modes mode = PLAY;

void in_handler_B0() {
	/* Only change the direction if push button is enabled */
	if (PBenabled)
	{
		switch (dir) {
		case (UP):
				dir = LEFT;
			break;
		case (DOWN):
				dir = RIGHT;
			break;
		case (RIGHT):
				dir = UP;
			break;
		case (LEFT):
				dir = DOWN;
			break;
		}
		PBenabled = false;
	}
}

void in_handler_B1() {
	/* Only change the direction if push button is enabled */
	if (PBenabled)
	{
		switch (dir) {
		case UP:
			dir = RIGHT;
			break;
		case DOWN:
			dir = LEFT;
			break;
		case RIGHT:
			dir = DOWN;
			break;
		case LEFT:
			dir = UP;
			break;
		}
		PBenabled = false;
	}
}


/* Callback functions */
void tickerCallback(void) {
	updateDisplay = true;

	/* Enable push buttons if the display is refreshed */
	PBenabled = refreshed;
}


void refreshCallback(void) {
	refreshed = true;
}

/**************************** Fill the boarder ********************************/

void fillBoarder(silabs::LS013B7DH03 &display){
	display.fill(0, 0, DISPLAY_WIDTH, TOPEDGE*STEPSIZE, Black);

	/* Fill right edge */
	display.fill(BOARD_WIDTH*STEPSIZE + BOARDERWIDTH/2, TOPEDGE*STEPSIZE + BOARDERWIDTH/2, 1, BOARD_HEIGHT*STEPSIZE, Black);
	for (uint8_t i=0;i<BOARD_HEIGHT;i++){
		for (uint8_t j=0;j<(DISPLAY_WIDTH-BOARD_WIDTH*STEPSIZE - BOARDERWIDTH/2);j++){
			display.pixel(BOARD_WIDTH*STEPSIZE + BOARDERWIDTH/2 +j, (i+TOPEDGE)*STEPSIZE + BOARDERWIDTH/2+j, Black);
		}
	}

	/* Fill bottom edge */
	display.fill(BOARDERWIDTH/2, (BOARD_HEIGHT+TOPEDGE)*STEPSIZE + BOARDERWIDTH/2, BOARD_WIDTH*STEPSIZE, 1, Black);

	for (uint8_t i=0;i<=BOARD_WIDTH;i++){
		for (uint8_t j=0;j<(DISPLAY_WIDTH-BOARD_WIDTH*STEPSIZE - BOARDERWIDTH/2);j++){
			display.pixel(i*STEPSIZE + BOARDERWIDTH/2 +j, (BOARD_HEIGHT+TOPEDGE)*STEPSIZE + BOARDERWIDTH/2+j, Black);
		}
	}

	/* Fill left edge */
	display.fill(BOARDERWIDTH/2-1, TOPEDGE*STEPSIZE + BOARDERWIDTH/2, 1, BOARD_HEIGHT*STEPSIZE, Black);
	for (uint8_t i=0;i<BOARD_HEIGHT;i++){
		for (uint8_t j=0;j<(DISPLAY_WIDTH-BOARD_WIDTH*STEPSIZE - BOARDERWIDTH/2 - 1);j++){
			display.pixel(j, (i+TOPEDGE)*STEPSIZE + BOARDERWIDTH/2+j, Black);
		}
	}

	/* Fill top edge */
	display.fill(BOARDERWIDTH/2, TOPEDGE*STEPSIZE + BOARDERWIDTH/2 - 1, BOARD_WIDTH*STEPSIZE, 1, Black);

	for (uint8_t i=0;i<=BOARD_WIDTH;i++){
		for (uint8_t j=0;j<(DISPLAY_WIDTH-BOARD_WIDTH*STEPSIZE - BOARDERWIDTH/2 - 1);j++){
			display.pixel(i*STEPSIZE + BOARDERWIDTH/2 +j, TOPEDGE*STEPSIZE + j, Black);
		}
	}

}

/**************************** MAIN ********************************************/
int main() {

	/* Initialize pushbutton handlers */
	in.fall(in_handler_B0);
	inB1.fall(in_handler_B1);

	/* Enable the LCD */
	DISP = 1;

	/* Start generating the 3Hz call */
	ticker.attach(&tickerCallback, 0.3333f);

	/* Reset the LCD to a blank state. (All white) */
	refreshed = false;
	if (display.clearImmediate(refreshCallback) == LS013B7DH03_OK){
		while (refreshed == false) sleep();
	}

	fillBoarder(display);
	refreshed = false;
	if (display.update(refreshCallback) == LS013B7DH03_OK)
	{
		while (refreshed == false) sleep();
	}
	Gecko gck;
	Food fd;
	gck.draw(display);
	fd.draw(display);

	/* Push update to the display */
	refreshed = false;
	if (display.update(refreshCallback) == LS013B7DH03_OK)
	{
		while (refreshed == false) sleep();
	}
	display.foreground(White);
	display.background(Black);
	display.locate(4,0);
	display.printf("Score: ");

	display.locate(11,0);
	display.printf("%d", score);

	/* Main loop */
	while (1) {
		sleep();
		if (updateDisplay && refreshed && (mode==PLAY)) {
			updateDisplay = false;

			gck.move(display, dir);

			if (fd.isEaten(gck))
			{
				fd.reset(display, gck);
				gck.increaseLength(display, dir);

				/* Redraw gecko */
				gck.draw(display);
				/* Update the score */
				score++;
				display.locate(11,0);
				display.printf("%d", score);
			}

			/* Update display */
			refreshed = false;
			display.update(refreshCallback);


			if (gck.selfCollision()) {
				mode = STOP;
				gck.move(display, dir);
				display.locate(3, 6);
				display.printf("GAME OVER!");
				refreshed = false;
				display.update(refreshCallback);
			}
		}
	}
}


