#include "dot_util.h"
#include "RadioEvent.h"
#include "InterruptIn.h"
#include "Callback.h"
#include "MTSText.h"

#define CHANNEL_PLAN CP_AS923       // Uncomment for different Freq. plans.
// #define CHANNEL_PLAN CP_AU915

////// Change below //////
static uint16_t locationID = 001;       // Change for different locations. (1 --> 65535)
float uplinkInterval = 1;         // Value of MINUTES between transmissions
                                            // *This is the default interval
uint16_t rain_total = 0;
static float batMax = 4.18;             // Change for different battery set up.
uint8_t downlink_msg = 0x00;

//// Update if different TTN/TTI application is used. ////
    // TTN Application
//static uint8_t network_id[] = { 0x70, 0xB3, 0xD5, 0x7E, 0xD0, 0x03, 0x3B, 0xB9 };
//static uint8_t network_key[] = { 0x12, 0x1A, 0x81, 0x8F, 0x10, 0x6B, 0x18, 0x67, 0x54, 0xE6, 0x9E, 0x70, 0x53, 0x7C, 0xAD, 0xCF };
    // TTI Application
static uint8_t network_id[] = { 0x70, 0xB3, 0xD5, 0x7E, 0xD0, 0x03, 0x3B, 0xB9 };
static uint8_t network_key[] = { 0xBE, 0x20, 0x8D, 0x61, 0x38, 0xAE, 0xE0, 0x76, 0x47, 0xFC, 0x23, 0x9D, 0xE1, 0xD7, 0x8D, 0x40 };

static std::string network_name = "MultiTech";
static std::string network_passphrase = "MultiTech";
static uint8_t frequency_sub_band = 0;
static lora::NetworkType network_type = lora::PUBLIC_LORAWAN;
static uint8_t join_delay = 5;
static uint8_t ack = 0;
static bool adr = true;

static bool deep_sleep = false;
I2C i2c(PC_9, PA_8 ); 

mDot* dot = NULL;
lora::ChannelPlan* plan = NULL;

Serial pc(USBTX, USBRX);    

// Initialize I2C
const int addr7bit = 0x40;      // 7 bit I2C address
const int addr8bit = addr7bit << 1; // 8bit I2C address, 0x80

volatile int _count = 0;

void incrementCounter(){
    _count++;
    wait_us(200000);    // delay for debouncing
}
void voidCounter(){
    wait_us(200000);    // delay for debouncing
}
int readCounter(){
    return _count;
}
void clearCounter(){
    logInfo("Setting counter to 0");
    _count = 0;
}

int main(){
    AnalogIn bat(A0);                   // Battery % analog input pin
    DigitalIn tamp(PA_3);
    
    InterruptIn in(PB_0);           // interrupt in pin on A1, analog due to 
                                    // nature of Rain Gauge pulse signal.
    in.fall(&incrementCounter);     // rise OR fall
    in.rise(&voidCounter);
    in.mode(PullUp);
    __enable_irq();
    
    RadioEvent events;
    
    pc.baud(115200);
    i2c.frequency(400000);
    
    mts::MTSLog::setLogLevel(mts::MTSLog::TRACE_LEVEL);
    
#if CHANNEL_PLAN == CP_AU915
    plan = new lora::ChannelPlan_AU915();
#elif CHANNEL_PLAN == CP_AS923
    plan = new lora::ChannelPlan_AS923();
#endif
    assert(plan);

    dot = mDot::getInstance(plan);
    assert(dot);

    // attach the custom events handler
    dot->setEvents(&events);

    if (!dot->getStandbyFlag() && !dot->getPreserveSession()) {
        logInfo("mbed-os library version: %d.%d.%d", MBED_MAJOR_VERSION, MBED_MINOR_VERSION, MBED_PATCH_VERSION);

        // start from a well-known state
        logInfo("defaulting Dot configuration");
        dot->resetConfig();
        dot->resetNetworkSession();

        // make sure library logging is turned on
        dot->setLogLevel(mts::MTSLog::INFO_LEVEL);

        // update configuration if necessary
        if (dot->getJoinMode() != mDot::OTA) {
            logInfo("changing network join mode to OTA");
            if (dot->setJoinMode(mDot::OTA) != mDot::MDOT_OK) {
                logError("failed to set network join mode to OTA");
            }
        }

        // To preserve session over power-off or reset enable this flag
        // dot->setPreserveSession(true);

        update_ota_config_id_key(network_id, network_key, frequency_sub_band, network_type, ack);

        // configure network link checks
        // network link checks are a good alternative to requiring the gateway to ACK every packet and should allow a single gateway to handle more Dots
        // check the link every count packets
        // declare the Dot disconnected after threshold failed link checks
        // for count = 3 and threshold = 5, the Dot will ask for a link check response every 5 packets and will consider the connection lost if it fails to receive 3 responses in a row
        update_network_link_check_config(3, 5);
//        update_network_link_check_config(2, 3);
        
        // enable or disable Adaptive Data Rate
        dot->setAdr(adr);
        
        // Configure the join delay
        dot->setJoinDelay(join_delay);

        // save changes to configuration
        logInfo("saving configuration");
        if (!dot->saveConfig()) {
            logError("failed to save configuration");
        }

        // display configuration
        display_config();
    } else {
        // restore the saved session if the dot woke from deepsleep mode
        // useful to use with deepsleep because session info is otherwise lost when the dot enters deepsleep
        logInfo("restoring network session from NVM");
        dot->restoreNetworkSession();
    }

    while(true){
        logInfo("Entering main loop!");
        uint16_t rain;
        uint16_t temp;
        uint16_t humid;
        uint32_t lat;
        uint32_t lon;
        uint8_t tamper;
        
        char cmd[2];        // I2C command address byte 8-bit
        char my_data[2];    // I2C return address bytes 16-bit
        
        std::vector<uint8_t> tx_data;
        
        //  join network if not joined
        if (!dot->getNetworkJoinStatus()) {
            join_network();
        }
        
        // Wake sensor /////////////////////////////////////////////////////////
        cmd[0] = 0x01;
        cmd[1] = 0x00;
        i2c.write(addr8bit, cmd, 2);
        
        // Read TEMERATURE
        cmd[0] = 0xE3;     // For 
        i2c.write(addr8bit, cmd, 1);
        i2c.read(addr8bit, my_data, 2);
        wait_us(20000);       
        
        uint16_t amb = (my_data[0] << 8) | my_data[1];
        float temperature = amb;
        temperature *= 175.72;
        temperature /= 65536;
        temperature -= 46.85;
        logInfo("Temp = %4.2f*C", temperature);
        
        wait_us(6000);
        
        // Read HUMIDITY
        cmd[0] = 0xE5;     // For
        i2c.write(addr8bit, cmd, 1);
        i2c.read( addr8bit, my_data, 2);
        wait_us(20000);       

        uint16_t hum = (my_data[0] << 8) | my_data[1];
        float humidity = hum;
        humidity *= 125;
        humidity /= 65536;
        humidity -= 6;
        if(humidity > 100.00){
            humidity = 100.00;
        }
        logInfo("Humidity = %4.2f%", humidity);
        
        // Shutdown temp sensor
        cmd[0] = 0x00;
        cmd[1] = 0x00;
        i2c.write(addr8bit, cmd, 2);
        ////////////////////////////////////////////////////////////////////////
        
        float battery = bat.read();
        battery = ((3*battery)*2) / batMax * 100;   // Turn battery V to a %
        logInfo("Battery = %f", battery);
        if(battery > 100){
            battery = 100;
        }
                    
        // Send it to the Cloudtracker
        rain = readCounter();   // Read the pulse count
        rain_total += rain;
        tx_data.push_back((rain >> 8) & 0xFF);
        tx_data.push_back(rain & 0xFF);
        logInfo("count = %d", readCounter());
        logInfo("count_total = %d", rain_total);
        clearCounter();         // Set pulse count back to 0
        
        uint16_t battery_send = battery * 10;
        tx_data.push_back((battery_send >> 8) & 0xFF);
        tx_data.push_back(battery_send & 0xFF);
        
        humid = humidity * 10;  
        tx_data.push_back((humid >> 8) & 0xFF);
        tx_data.push_back(humid & 0xFF);
        
        temp = temperature * 10;
        tx_data.push_back((temp >> 8) & 0xFF);
        tx_data.push_back(temp & 0xFF);
        
        tamper = 0;
        if(tamp == 0){
            tamper = 1;
        }
        tx_data.push_back(tamper & 0xFF);
        
        tx_data.push_back((locationID >> 8) & 0xFF);
        tx_data.push_back(locationID & 0xFF);
        
        tx_data.push_back((rain_total >> 8) & 0xFF);
        tx_data.push_back(rain_total & 0xFF);
        
        send_data(tx_data);
        
        // Send to sleep for desired time. 
        ThisThread::sleep_for((uplinkInterval * 60000) - 1850);  // (-1.85 sec for time to think)
    }
    
    return 0;
}
