/**
    Snake Game Timer
    gameTimer.cpp
    Board ID    3
    Purpose:    This function runs on a FRDM KL25Z. It uses the TextLCD library to write to a 2x16
                LCD. It uses two interrupt buttons to start and stop the timer. The timer and game 
                are started by a push button on the board and ends when the timer receives 
                GameRunning = 0 from the server.
    @author Daniel Lock
    @version
*/

#include "mbed.h"
#include "TextLCD.h"

Serial pc(USBTX, USBRX); // tx, rx  used for printing to the PC screen during debugging

int GameRunning;                    //True if game is running  <----- could be a bool?
int c;                              //count variable
Timer timer;                        //Use mbed's built in timer function
float StartTime, NowTime;           //Floats for the timer reading at gamestart and currently

InterruptIn btnStart(PTA4);         //Temporarily use switch to start - will be WiFi command
InterruptIn btnStop(PTA5);          //Temporarily use switch to stop - will be WiFi command

TextLCD lcd(PTE30, PTE29, PTE23, PTE22, PTE21, PTE20); // rs, e, d4-d7

RawSerial dev(PTE0,PTE1);  // for KL25Z... asuming one can't use the PTA1 version which is the stdio
DigitalOut rst(PTD1); // single digital pin to drive the esp8266 reset line


// subroutine to run anytime a serial interrupt arrives from the device
// this basically passes everything thatthe device produces on to the pc terminal screen
void dev_recv()
{
    //led1 = !led1;
    while(dev.readable()) {
        pc.putc(dev.getc());
        //wait_us(1);
    }
}
// subroutine to service the serial interrupt on the pc connection
// this is a bit more complex - it takes what the use sends on the pc and copies it on to the device 
// the esp should echo these straight back to the the pc if all is well
// this also detects the end of command character which is ascii 13 (0x0d) adn adds a linefeed after it =asscii 10 (0x0a)
void pc_recv()
{
    char c;
    //led4 = !led4;
    while(pc.readable()) {
        c=pc.getc();
        dev.putc(c);
        pc.putc(c); // echo back
        if(c==13) {dev.putc(10); // send the linefeed to complement the carriage return generated by return key on the pc
        pc.putc(10);
            }
    }
}


//Takes an argument and sends it to the server (this could be dodgy if length would vary but only using 1 1bit number
void sendToServer(char message)
{
    dev.printf("AT+CIPSEND=3,7\r\n");
    wait(0.1);
            
    dev.printf("3gar%d\r\n",message);              //Identifier,GameRunning Tag,GameRunning Value
    pc.printf("send %d\n\r",message);
}

//function to start the timer
void timerStart()  
{
    timer.start(); 
    StartTime = timer.read_us();    //reads time in micro seconds
    GameRunning = 1;
    pc.printf("%d\n\r", GameRunning);
    sendToServer(GameRunning);      //Tell the server to start the game
}

//function to stop the timer
void timerStop() 
{
    timer.stop();
    NowTime = timer.read_us();      //reads time in micro seconds
    GameRunning = 0;
    pc.printf("%d\n\r", GameRunning);
    sendToServer(GameRunning);      
}

void setupWiFi()
{
    dev.printf("AT+RST\r\n");                       //Reset WiFi
    pc.printf("RESET\r\n");
    
    wait(2);
    dev.printf("AT+CWMODE=1\r\n");                  //Set mode to client
    
    wait(2);
    dev.printf("AT+CWJAP=\"CWMWIFI\",\"CWM2016TT\"\r\n");       //Login to the WiFi
    
    wait(7);
    dev.printf("AT+CIFSR\r\n");                     //Find IP and MAC address - not necessary?
    
    wait(5);
    dev.printf("AT+CIPMUX=1\r\n");                  //Allow multiple connections
    
    wait(2);
    dev.printf("AT+CIPSTART=3,\"TCP\",\"192.168.1.6\",5050\r\n");       //Open connection with the server
    
    wait(2);
}

int main()
{   
    rst=0;
    wait(1);
    rst=1; // send the esp8266 reset
    wait(1);
    pc.printf("ok off we go....\n\r");
    
    pc.baud(115200); // NB maybe this should go before this
    dev.baud(115200);

    pc.attach(&pc_recv, Serial::RxIrq); // attach the two interrupt services
    dev.attach(&dev_recv, Serial::RxIrq);
    
    setupWiFi();

    btnStart.rise(&timerStart);     //on rising edge start timer
    btnStop.rise(&timerStop);       //on rising edge stop timer
    
    while(1) 
    {

        NowTime = timer.read_us();                                          //Read the current time in micro seconds
        lcd.printf("Time: %4.3f \n\n", (NowTime-StartTime)/1000000);        //Print the difference between current time and start time to the LCD
        wait(0.05);
        
       /* if(GameRunning == 0)
        {
            timerStop();            //When GameRunning is received as zero stop the timer
        }*/
    }
}
