#include "MAX31820.h"
#include "DS1Wire.h"
#include "mbed.h"
#include <stdint.h>

DigitalOut conversionInProgress(LED2);  		// conversion in progress
DigitalOut resetFailure(LED3);          		// for error reporting
extern DigitalInOut sensor;     				// sensor connected to pin 

static void inError()
{
    while (1) {
        resetFailure = !resetFailure;
        wait(0.2);
    }
}

ROM_Code_t ReadROM()
{
    ROM_Code_t ROM_Code;
    if (Reset(sensor) != 0) {
        inError();
    } else {
        WriteByte(sensor, READ_ROM);    		// Read ROM
        for (uint32_t i = 0; i < 8; ++i) {
            ROM_Code.rom[i] = ReadByte(sensor);
        }
    }
    return ROM_Code;
}

void setResolution(uint8_t resln)
{
	uint32_t config = 0;
    config |= (resln << 5);
    WriteByte(sensor, WRITE_SCRATCHPAD);
    WriteByte(sensor, TH_ALARM);
    WriteByte(sensor, TL_ALARM);
    WriteByte(sensor, config);
    Reset(sensor);
    WriteByte(sensor, SKIP_ROM);				// Skip ROM
    WriteByte(sensor, COPY_SCRATCHPAD);
    wait_ms(15);
}

void DoConversion()
{
    if (Reset(sensor) != 0) {
        inError();
    } else {
        conversionInProgress = 1;       		// led on
        WriteByte(sensor, SKIP_ROM);            // Skip ROM
        WriteByte(sensor, CONVERT);				// Convert
        while (ReadBit(sensor) == 0) {
            // wait for conversion to complete
        }
        wait(1);
        conversionInProgress = 0;       		// led off
    }
}

float calculateTemperature(ScratchPad_t *lecture)
{
    uint32_t read_temp = ((lecture->MSB << 8) | lecture->LSB);;
    bool signBit = false;
    if (lecture->MSB & 0x80)
        signBit = true;
    if (signBit) {
        read_temp = (read_temp ^ 0xFFFF) + 1;    // two's complement
        read_temp *= -1;
    }
    uint32_t resolution = (lecture->config & 0x60) >> 5; // mask off bits 6,5 and move to 1,0
    switch (resolution) {
        case NINE_BITS:    // 0.5 deg C increments
            read_temp &= 0xFFF8;                	// bits 2,1,0 are undefined
            break;
        case TEN_BITS:     // 0.25 deg C increments
            read_temp &= 0xFFFC;                	// bits 1,0 are undefined
            break;
        case ELEVEN_BITS:  // 0.125 deg C increments
            read_temp &= 0xFFFE;                	// bit 0 is undefined
            break;
        case TWELVE_BITS:  // 0.0625 deg C increments
            break;
    }
    float realTemp = (float)read_temp/16 ;
    return realTemp;
}

float GetTemperature()
{
    float temperature = 0;
    if (Reset(sensor) != 0) {
        inError();
    } else {
        ScratchPad_t dataPad;
        WriteByte(sensor, SKIP_ROM);    		// Skip ROM
        WriteByte(sensor, READ_SCRATCHPAD);		// Read Scrachpad
        dataPad.LSB = ReadByte(sensor);
        dataPad.MSB = ReadByte(sensor);
		Reset(sensor);
		
        temperature = calculateTemperature(&dataPad);
    }
    return temperature;
}

void displayTemperature(Serial& screen)
{
    DoConversion();
    float temp = GetTemperature();
    screen.printf("\n\rTemp is %2.1f\n\r", temp);    // display in 2.1 format
}

