/* Project: Data Fitting Code
Description: Base functionality for lighting controller.
Works with IR sensor. Fixed negative problem.

Author: Alex Mueller and Marissa Gore
Date: 10/28/2020

*/

//Includes
#include "mbed.h"               //Includes mbed header file
#include "LCD_DISCO_L476VG.h"   //Includes LCD libraries
#include <stdio.h>              //Includes studio header file
#include <fstream>              //Include file handing library


DigitalOut Ctrl(PA_2);              //Controls relay
AnalogIn ain(PA_1);                 //Sensor

LCD_DISCO_L476VG lcd;

//Global variables
float sampledData[10000];
int numDataSamples = 0;

//Initalize functions
void SensorValues(float value);
int PersonId();
int PeakLocation();
int MultiplePeaks(int lowestValue); 
void Counter(int count);
int PeakLowValue();


int main()
{
    float value;
    bool incident = false;           //Setting incident intially to false. Incident = when someone has waked through doorway.
    float baseline = 65;             //65cm
    int count = 0;                  
    int person;
    uint8_t DisplayedString[7] = {0};
    Ctrl = 1;                        //Start relay control off so doesn't flicker light.

   while(1)
   {
       value = 30 * pow(ain.read(), -1.173);    
       printf("%f \n", value);                  
       if(value < baseline)                     //If a sensor value drops below the baseline, incidient is set to 
       {                                        // true(person walked through doorway and sensor values of incident are read into an array.
           incident = true;                     
           SensorValues(value);                 
        }
        else if(incident == true && value >= baseline)      //If incident was previously set to true and the sensor valuesgo back up to the baseline,
        {                                                   // send to person Id funciton to determine if a person entered or exited. 
            person = PersonId();                            //Control relay and incriment/decriment count accordingly.
            if(person == 1)                         
            {
                Ctrl = 0;                 //Set low to turn on relay.
                count++;                 
                Counter(count);        
            }
            else if(person == 2)                  
            {
                count--;                 
                Counter(count);         
                if(count <= 0)          
                {
                    Ctrl = 1;          //Set high to turn off relay.
                    count = 0;        //If count is negative set it to zero.
                    Counter(count);
                }
            }
            incident = false;            
            numDataSamples = 0;         
        }
        ThisThread::sleep_for(10);      
    }  
}      

void SensorValues(float value)              //Function to read sensor values into array, numDataSamples are increased for number of data samples in incident.
{
   sampledData[numDataSamples] = value;     
   numDataSamples++;                        
}   

int PeakLocation()                          //Function to find position of min value in incident array
{
    int min = sampledData[0];               
    int saveLow = 0;        
    for (int i = 1; i < numDataSamples; ++i)       
    {
        if(sampledData[i] < min)               
        {   
            min = sampledData[i];              
            saveLow = i;                      
        }   
    }
    return saveLow;                         
}

int PeakLowValue()                      //Function to find minimum value in incident array        
{
    int min = sampledData[0];               
    int saveLow = 0;        
    for (int i = 1; i < numDataSamples; ++i)       
    {
        if(sampledData[i] < min)               
        {   
            min = sampledData[i];                                   
        }   
    }
    return min;                         
}

int MultiplePeaks(int lowestValue)              //Function to determine if there are multiple peaks. (multiple peaks = false entry/exit)
{
    bool mul = false;
    int lowFirst = sampledData[0];
    int lowSecond = sampledData[0];
    int firstPos = 0;
    int secondPos = 0;
    for(int i = 1; i < numDataSamples/2; i++)
    {
         if(sampledData[i] < lowFirst)               
        {   
            lowFirst = sampledData[i];  
            firstPos = i;                                  
        } 
    }
    for(int i = numDataSamples/2; i < numDataSamples; i++)
    {
         if(sampledData[i] < lowSecond)               
        {   
            lowSecond = sampledData[i];    
            secondPos = i;                             
        } 
    }
    if(firstPos <= (secondPos*0.85) || firstPos >=  (secondPos*0.85))
    {
        mul = false;
    }
    else
    {
        if(lowFirst < (lowestValue*0.9) && lowSecond < (lowestValue*0.9))
        {
            mul = true;
        }
        else
        {
            mul = false;
        }
    }  
    return mul;
}
    
int PersonId()                              //Function to determine if person entered or exited room. 
{
    int lowest = PeakLocation();   
    int lowestValue = PeakLowValue();
    bool mul = MultiplePeaks(lowestValue);    
    
    if(mul == true)
    {
        return 0;
    }
    else
    {     
        if(lowest < (numDataSamples/3))        // If Lowest(min value position) < 1/3 of data, person exited
        {
            return 2;                         
        }
        else if(lowest > (2*numDataSamples/3))  // If Lowest(min value position) > 2/3 of data, person entered
        {
            return 1;                       
        }
        else
        {
            return 0;                       
        }
    } 
}

void Counter(int count)                 //Function to print count to LCD.
{
    lcd.Clear();
    uint8_t DisplayedString[7] = {0};
    sprintf((char *)DisplayedString, "%d\n", count);
    lcd.DisplayString(DisplayedString);
}